/*
	PREP - This program will prepare a winchester disk for use
    by various operating systems. The program uses drive characteristics
    available from the controller.
	The disk is formatted and NUM_PASS media test passes are
    performed. Any bad sectors are entered into a table for OS's to use.
    The disk is formatted again and a final pattern test is made. Finally
    the boot code and partition table, and bad sector tables with checksums
    are written to the disk.
    
    
    Author: RJM
    
    Date: 10/26/83

    Version 1.76 - Modify for 8 winchester drives
    Version 1.77 - Modify bad sector table to hold interleave factor.
		   Use old bad sector info. if interleave is the same.
    Version 1.90 - Update utilities to all V1.90
    Version 1.91 - 1/5/84 Change interleave to 3 for WD controller
    Version 2.00 - Updated to V2.0
    Version 2.01 - Fix allocation of default partition to allocate full disk
    Version 2.02 - Put interleave in bst sector
    Version 2.03 - Change so PREP does no retries on media passes
    Version 2.04 - 4/16/84
		   1) PREP now formats till it receives an error
		      then fills in sectors with data and continues
		      the format
		   2) Allow the user to clear the bad sector table
		      by specifying /I with no parameters
		   3) Interleave defaults to 5 (was 3) for XEBEC boards
		   4) Controller retries are disabled in media test passes
    Version 2.06 - 7/14/84 Modified PREP to allow booting from cartridge 
		   winchesters that were PREP'ed in another unit.
    Version 2.07 - PREP gets Sectors per track info. from the controller.
		   rather than assuming 17 SPT. Fixed "bug" retry not turned
		   off on last partition table.
    Version 2.08 - PREP now does a reset and a recalibrate for error
		   recovery

*/

/*
		RESTRICTED RIGHTS LEGEND
		------------------------
	
	    "Use, duplication, or disclosure by the
	Government is subject to restrictions as set forth
	in paragraph (b) (3) (B) of the Rights in Technical
	Data and Computer Software clause in DAR
	7-104.9(a).  Contractor/manufacturer is Zenith
	Data Systems Corporation of Hilltop Road, St.
	Joseph, Michigan 49085.
*/

#include "stdio.h"
#include "conio.h"
#include "ctype.h"
#include "z150rom.h"
#include "fixed.h"
#include "bst.h"

#define	TRUE	1
#define	FALSE	0


/* Flags for signon messages */
#define	ZENITH	TRUE
#define	NBI	FALSE


/* Define the version and release numbers for PREP */
#define VERSION 2
#define RELEASE 8

#define DISK_STATUS 0x42	/* Offset into ROM_DATA of status bytes */

/* Define a bad sector table */
struct badsec_table bst;

/*    Define the number of media test passes (Byte value in test_pat[i]) */
#define	NUM_PASS 6
char test_pat[NUM_PASS] = {0x0, 0xff, 0xaa, 0x55, 0x4e, 0xa1};

/* Define the format test pattern */
#define FORMAT_PATTERN 0xe5

#define CARRY	0x0001		/* Carry flag position in flag word */
#define MAN_UTIL FALSE

#define DOSF_INSTR 0xa		/* DOS input string function */

unsigned user_cyl, bs_cyl;
char max_hds;
unsigned max_spt;		/* Maximum number of sectors per track */

/*      Variable for interleave factor      */
char interleave = INTERLEAVE;

unsigned head, cyl, sector;

/* General purpose buffer, and drive number variable */
char disk_buf[SECTOR_SIZE], drive;

/* Variable for the number of bad sectors in table */
int bad_count;

/* Buffer for ensuring no DMA errors */
char buf1[2*SECTOR_SIZE];
char *buf;			/* Pointer to correct part of buf1 */

char switch_char = '/';		/* Default switch character */

char iswitch = FALSE;		/* Flag for I switch specified */

unsigned dpb_segment, dpb_offset;

struct
{
    struct dpb dpbt[4];
} dpb_table;

/* Define the externals used in the z150int() module */
extern char ral, rah, rbl, rbh, rcl, rch, rdl, rdh;
extern unsigned rax, rbx, rcx, rdx, rsi, rdi, rsp, rbp;
extern unsigned rcs, rds, res, rss, flags;
    
/*
    _main() - This routine will get the command line, skip over the
	first character (a 'c') and scans past white space then passes
	the command line to PREP
*/
_main(line)
char *line;
{
    do ++line;
    while (isspace(*line) && *line != '\0');
    main(line);
    return;
}

    
main(line)
char *line;
{
    init();
    signon();			/* Greet the user */
    scan(line);			/* Scan the input string for switches */
    if (!setup()) exit(0);	/* Make sure the controller is there */
    bad_count = init_bst();	/* Get count of secs. in the bad sec. table */
    if (!format()) exit(0);	/* Format the drive */
    bad_sector();		/* Do bad sector detection */
    put_default();		/* Write the default tables to the disk */
    exit(0);
}

init()
{
    res = mydsreg();		/* Set up ES register to this data seg. */
    get_buf();
    return;
}

/*
    init_bst() - Initialize the bad sector table to either the old
	one or to no bad sectors.
*/
init_bst()
{
    int i;
    long bst_sec;
    unsigned check_sum(), check;

    if (!iswitch)    
    {
	/* Attempt to get the old bad sector table */
	bst_sec = (long)bs_cyl * max_hds * max_spt;
	for (i = 0; i < 2; ++i)
	{
	    set_sector(bst_sec);
	    /* If the sector can be read, verify the check sum */
	    rbx = &bst;
	    if (get_sector())
	    {
		check = bst.check_sum;
		bst.check_sum = 0;
	
		/* Verify check sum */
		if (check == check_sum(mydsreg(), &bst, SECTOR_SIZE))
		{
		    /* Check interleave, same as last time else info. bad */
		    if (bst.interleave == interleave) return(cbs());
		    else break;
		}
	    }
	    /* Try next bad sector table */
	    bst_sec += max_spt;
	}
    }

    /* No bad sector table exists, make a new one */
    for (i = 0; i < MAX_BAD; ++i) 
    {
	bst.bad_secs[i].head = 0;
	bst.bad_secs[i].cyl = 0;
	bst.bad_secs[i].sec = 0;
    }
    bst.check_sum = 0;
    bst.interleave = interleave;	/* Set interleave */
    bst.res[0] = 0;			/* Set reserved to 0 */
    bst.res[1] = 0;
    return(0);				/* No bad sectors */
}

/*
    cbs() - This routine will count the number of bad sectors
	in a bad sector table.
*/
cbs()
{
    int count;

    count = 0;
    while (bst.bad_secs[count].head != 0 ||
	    bst.bad_secs[count].cyl != 0 ||
	    bst.bad_secs[count].sec != 0)   ++count;
    return(count);
}

/*
    signon() - Greet the user and tell him about PREP.
*/
signon()
{
    float a;
    a = VERSION + (float)RELEASE / 100;

#if ZENITH
    printf("\r\n                PREP version %3.2f\r\n", a);
    puts("Copyright(C) 1984, Zenith Data Systems Corporation\r\n");
#endif

#if NBI
    printf("\r\n     PREP version %3.2f\r\n", a);
    puts("(C)Copyright NBI, Inc. 1984\r\n");
#endif

    puts("\r\nThe PREP utility helps you to:\r\n\n");
    puts("    * Initialize surface of Winchester disk\r\n");
    puts("    * Test data retention capabilities of Winchester disk media\r\n");
    puts("    * Isolate questionable disk sectors\r\n");

    puts("\r\n\nPREP will prompt you for the winchester drive unit number.\r\n");
    puts("Then PREP displays messages as it operates on the disk.\r\n\n");
    puts("Caution: Using PREP will destroy all files on your Winchester disk.\r\n");
    puts("Winchester disks supplied by Zenith or Heath are normally prepared\r\n");
    puts("by PREP before they are shipped. Users of these disks will need to use\r\n");
    puts("PREP only if they are consistently encountering an unreasonable number\r\n");
    puts("of disk access errors. Do not use PREP until you have transferred\r\n");
    puts("your Winchester disk files to floppy disks.\r\n");

    return;
}


/*
    scan() - This routine will scan the input line for switches
	using external list of switch characters at sw_chars.
*/
scan(line)
char *line;
{
    char *intlv();
    while(TRUE)
    {
	while(*line != switch_char && *line != '\0') ++line;
	if (*line == '\0') return;
	++line;
	if (islower(*line)) *line = toupper(*line);
	switch (*line)
	{
	    case 'I':
		line = intlv(++line);
		break;
		    
	    default:
		break;
	}
    }
}

/*
    intlv()  - This routine will process the interleave switch
*/
char *intlv(line)
char *line;
{
    int num;
    
    iswitch = TRUE;
    num = 0;
    while(isdigit(*line))
    {
	num = num * 10 + *line++ - '0';
	if (num >= max_spt) return(line);
    }
    if (*line != '\0' && *line != '/' && !isspace(*line)) return(line);
    if (num < 1) return(line);
    interleave = num;
    return(line);
}


/*
    get_buf() - This routine will prepare the "buf1" buffer and
	the "buf" pointer so it doesn't cross a 64K memory boundry.
*/
get_buf()
{
    unsigned address;
    
    buf = &buf1[0];
    address = (((unsigned)mydsreg()) << 4) + (unsigned)&buf1[0];	/* Get physical address of buffer */
    address += SECTOR_SIZE;		/* Bump address by SECTOR_SIZE */

    /* Modify buffer address if the current buffer crosses boundry */
    if (address < (unsigned) buf) buf += SECTOR_SIZE;
    return;
}


/*
    setup() - This routine will ensure that the user wants to PREP the disk
	and that the controller actually exists.
*/
setup()
{
    char c, flag;
    /* Ask the user if he is sure */
    puts("\r\nDo you wish to proceed with PREP (Y/N)?");
    if (!yes_no()) return(FALSE);
    
    /* Ask the dummy again */
    puts("\r\nType P to proceed\r\n");
    c = getchar();
    if (c != 'P' && c != 'p') return(FALSE);
    
    /* Get the drive number */
    printf("\r\nWinchester drive unit number (0-%c): ", NUM_FIXED-1+'0');
    while(TRUE)
    {
	drive = getchar() - '0';
	if (drive >= NUM_FIXED) beep();
	else break;
    }
    printf("%c\r\n", drive+'0');

    /* Verify controller exists (get parameters for drive 0) */
    flag = controller(drive);
    if (!flag || drive >= rdl)
    {
	puts("\r\nCan not communicate with Winchester controller\r\n");
	exit(0);
    }
    
    /* Get the maximum number of heads */
    max_hds = rdh + 1;

    /* Get the maximum number of sectors per track */
    max_spt = rcl & 0x3f;
    
    /* Get the users number of cylinders */
    user_cyl = rch + ((rcl & 0xc0) << 2);

    /* The bad sector cylinder is 1 greater than the user cyl. */
    bs_cyl = user_cyl + 1;	  /* reserved cyl is one after last user */

    /* Ensure drive is a valid size */
    if ((long)(bs_cyl+1) * max_hds * max_spt >= MAX_PART)
    {
	beep();
	puts("Error - Drive capacity > 32 megabytes!");
	exit(0);
    }

    /* Reset the disk system */
    disk_reset();

    return(TRUE);		/* All OK */
}

/*
    format() - This routine will format the Winchester drive. It formats until
	it receives an error, marks that track bad and then continues from
	that point until done. If the format is sucessful TRUE is returned,
	else FALSE is returned.
*/
format()
{
    int flag;

    if (interleave == INTERLEAVE) puts("\r\n\nInitializing the disk...");
    else printf("\r\n\nInitializing the disk at interleave %d...", interleave);
    set_sector(0L);    		/* Start formatting at track 0, head 0 */
    flag = format_disk();
    if (flag) puts("Completed\r\n");
    else puts("\r\n\nError during formatting of the drive.\r\n");
    return(flag);
}

/*
    put_default() - This routine will put the default partitioning
	information on the disk. It first writes out the boot code
	then does a check sum on the bad sector table and then writes
	the bad sector table to the disk.
*/
put_default()
{
    unsigned check_sum(), sbcseg();
    int i;
    struct p_ent *ptr;

    /* Patch the boot code for a DOS partition using the entire disk */
    movbyte(sbcseg(), 0, mydsreg(), &disk_buf[0], SECTOR_SIZE);
    ptr = (struct p_ent *) &disk_buf[SECTOR_SIZE - sizeof(struct part_tbl)];
    ptr->boot_flag = 0x80+drive;
    ptr->os_id = DOS_ID;

    /* Set the start sector */
    
    ptr->begin_hcs.head = 0;
    ptr->begin_hcs.cyl = 0;
    ptr->begin_hcs.sec = 2;	/* Leave room for the boot code */

    /* Set the end sector */
    ptr->end_hcs.head = max_hds-1;
    ptr->end_hcs.cyl = user_cyl & 0xff;
    ptr->end_hcs.sec = ((user_cyl >> 2) & 0xc0) + max_spt;

    /* Set the relocation sector */
    ptr->rel_sec = 1L;

    /* Set the partition size */
    ptr->part_size = ((long)(user_cyl+1)) * max_hds * max_spt - 1L;
 
    /* Write the boot code to the disk */
    rbx = &disk_buf[0];
    set_sector(0L);
    if (!put_sector())
    {
	puts("Boot sector is bad\r\n");
	exit(0);
    }
    res = mydsreg();

    /* Check sum the bad sector table */
    bst.check_sum = check_sum(mydsreg(), &bst, SECTOR_SIZE);
    
    /* Write bad sector table A */
    rbx = &bst;
    set_sector((long)bs_cyl*max_hds*max_spt);
    put_sector();
    
    /* Write bad sector table B */
    rbx = &bst;
    set_sector((long)bs_cyl*max_hds*max_spt + max_spt);
    put_sector();
    
    return;
}

/*
    beep() - This routine will sound a beep to the user to indicate an error
*/
beep()
{
    putchar(7);
    return;
}

/*
    yes_no() - This routine will retreive a response from the user
	to a yes no question. If 'Y' or 'y' is pressed, TRUE is returned.
	If a 'N' or a 'n' is pressed then FALSE is returned.
*/
yes_no()
{
    char c1;
    
    while(TRUE)
    {
	c1 = getchar();
	if (islower(c1)) c1 = toupper(c1);
	if (c1 != 'N' && c1 != 'Y') beep();
	else
	{
	    putchar(c1);
	    puts("\r\n");
	    return(c1 == 'Y');
	}
    }
}

/*
    bad_sector() - This routine will do NUM_PASS media test passes
	to attempt to find bad sectors
*/
bad_sector()
{
    char pass = 1;

    puts("\r\n");
    retry_off();
    for (pass = 0; pass < NUM_PASS; ++pass)
    {
	printf("\rMedia test in progress, pass %d", pass+1);
	set_pattern(test_pat[pass]);	/* Set the test pattern */
	put_pattern(test_pat[pass]);	/* Write the test pattern to the disk */
	get_pattern(test_pat[pass]);	/* Get the test pattern from the disk */
    }
    retry_on();
    format();				/* Format the disk again */

    return;
}


/*
    retry_on() - This routine will turn on controller retries of
	disk I/O
*/
retry_on()
{
    putword(0, (DPB_POINTER << 2) + 2, dpb_segment);
    putword(0, DPB_POINTER << 2, dpb_offset);
    rdl = 0x80 + drive;
    rah = DIO_SETPARMS;
    z150int(DISK_IO_INTR);
    return;
}

    
/*
    retry_off() - This routine will turn off controller retries of
	disk I/O
*/
retry_off()
{
    int i;

    dpb_segment = getword(0, (DPB_POINTER << 2) + 2);
    dpb_offset = getword(0, DPB_POINTER << 2);
    movbyte(dpb_segment, dpb_offset, mydsreg(), &dpb_table, sizeof(dpb_table));
    for (i = 0; i < 4; ++i) dpb_table.dpbt[i].control |= 0xc0;
    putword(0, (DPB_POINTER << 2) + 2, mydsreg());
    putword(0, DPB_POINTER << 2, &dpb_table);
    rdl = 0x80 + drive;
    rah = DIO_SETPARMS;
    z150int(DISK_IO_INTR);
    return;
}

    
/*
    put_pattern() - This routine will put the specified pattern on every sector
	of the disk. If a sector can not be written it is flagged as bad.
*/
put_pattern(pat)
char pat;
{
    unsigned sec;

    set_sector(0L);
    do
    {
	rbx = &disk_buf[0];
	if (!put_sector())
	{
	    if (!flag_sector()) bs_full();
	}
    }
    while(next_sector());
    return;
}
    
/*
    get_pattern() - This routine will read every sector on the disk. If the
	sector can not be read or if the pattern does not match it
	is flagged as a bad sector.
*/
get_pattern(pat)
char pat;
{
    int i;

    set_sector(0L);			/* Start at the beginning of the disk */
    do
    {
	rbx = &disk_buf[0];
	if (!get_sector())
	{
	    if (!flag_sector()) bs_full();	/* If bad sector, flag it */
	}
	else
	{
	    if (!charcomp(pat, &disk_buf, SECTOR_SIZE))
	    {
		if (!flag_sector()) bs_full();	/* If pattern does not match, flag it */
	    }
	}
    }
    while (next_sector());

    return;
}

/*
    set_pattern() - This routine will set a pattern into the disk buffer
	for writing test patterns.
*/
set_pattern(pat)
char pat;
{
    int i;

    for (i = 0; i < SECTOR_SIZE; ++i) disk_buf[i] = pat;
    return;
}

/*
    next_sector() - This routine will set the variables head, track and sector
	to the next sector value. If the sector exists then TRUE is
	returned, if the end of the disk is hit then FALSE is returned.
*/
next_sector()
{
    if (++sector > max_spt)
    {
	sector = 1;
	if (++head == max_hds)
	{
	    head = 0;
	    if (++cyl == bs_cyl+1) return(FALSE);
	}
    }
    return(TRUE);
}

/*
    set_sector() - This routine will set a logical sector into the actual
	sector, track and head.
*/
set_sector(sec)
long sec;
{
    sector = sec % max_spt + 1;		/* Get sector (based at 1) */
    sec  = sec / max_spt;		/* Update sec, get number of tracks */
    head = sec % max_hds;		/* Get head */
    cyl  = sec / max_hds;	/* Get cylinder */
    return;
}

/*
    flag_sector() - Flag a sector as bad in the bad sector table. If more than MAX_BAD
	sectors are found or if a bad sector exists before the first user
	sector then PREP prints an error message and aborts.
*/
flag_sector()
{
    int i,j;
    long logical_sector();
    struct hcs temp_hcs;
 
    encode();
    temp_hcs.head = rdh;
    temp_hcs.cyl = rch;
    temp_hcs.sec = rcl;
    
    
    /* See if the sector is already present */
    for (i = 0; i < bad_count; ++i)
    {
	if (logical_sector(&temp_hcs) == logical_sector(&bst.bad_secs[i])) return(TRUE);
	if (logical_sector(&temp_hcs) < logical_sector(&bst.bad_secs[i])) break;
    }
    
    /* Have found bad sector, see if over the limit */
    if (++bad_count == MAX_BAD) return(FALSE);

    /* Shift the other bad sectors down to enter this one in order */
    for (j = bad_count; j > i; --j)
    {
	bst.bad_secs[j].head = bst.bad_secs[j-1].head;
	bst.bad_secs[j].cyl = bst.bad_secs[j-1].cyl;
	bst.bad_secs[j].sec = bst.bad_secs[j-1].sec;
    }
    
    /* Enter the new bad sector */
    bst.bad_secs[i].head = temp_hcs.head;
    bst.bad_secs[i].cyl = temp_hcs.cyl;
    bst.bad_secs[i].sec = temp_hcs.sec;
    
    return(TRUE);
}


/*
    logical_sector() - this routine will take the head cylinder and
	sector passed to it and return the logical sector it corresponds
	to.
*/
long logical_sector(p)
struct hcs *p;
{
    long tmp;
    
    tmp = p->head + (p->cyl + (((unsigned)p->sec & 0xc0) << 2)) * max_hds;
    return((long) (tmp * max_spt + (p->sec & 0x3f) - 1L));
}

/*
    check_sum() - This routine will return a check sum of a block of memory.
*/
unsigned check_sum(seg, off, len)
unsigned seg, off, len;
{
    unsigned sum;
    
    sum = 0;
    do
    {
	sum += getbyte(seg, off++);
    }
    while(--len > 0);
    return(~sum);
}

/*
    controller() - This routine will return TRUE if a winchester controller
	is present in the system.
*/
controller()
{
    rah = DIO_GETPARMS;
    rdl = drive + 0x80;
    z150int(DISK_IO_INTR);
    return(!(flags & CARRY));
}

/*
    format_disk() - This routine will format a winchester disk.
*/
format_disk()
{    
    int i;

    /* Set the format pattern */
    set_pattern(FORMAT_PATTERN);
 
    /* Write the format pattern */   
    rah = DIO_WRTBUF;
    ral = 1;
    rbx = &disk_buf[0];
    res = mydsreg();
    set_sector(0L);
    rdl = 0x80 + drive;
    z150int(DISK_IO_INTR);

    while(TRUE)
    {
	/* Do the actual format */
	rah = DIO_FMTDISK;
	ral = interleave;		/* Interleave factor */
	rdl = 0x80+drive;
	rbx = buf;			/* ensure no DMA problems */
	encode();
	z150int(DISK_IO_INTR);
	if (!(flags & CARRY)) break;
	
	fix_track();
    }
    return(TRUE);
}


/*
    fix_track() - This routine will fill in a track that a format
	error has occurred on with the format pattern.
	No bad sector entry takes place at this point but is left for
	the media test passes.
*/
fix_track()
{
    set_pattern(FORMAT_PATTERN);
    rbx = &disk_buf[0];
    find_loc();		/* Find the location of the error */
    for (sector = 1; sector <= max_spt; ++sector) put_sector();
    sector = 1;
    if (++head >= max_hds)
    {
	head = 0;
	++cyl;
    }
    return;
}


/*
    encode() - This routine will encode the head, cylinder and sector number
	into the values needed by the ROM.
*/
encode()
{
    rcl = ((cyl & 0x0300) >> 2) | sector;
    rch = cyl & 0x00ff; 
    rdh = head;
    return;
}


/*
    find_loc() - This routine will return the location of a disk error
	from the ROM data segment into the variables head, cyl and sector
*/
find_loc()
{
    char temp;
    
    head = getbyte(ROM_DATA, DISK_STATUS+1) & 0x1f;
    temp = getbyte(ROM_DATA, DISK_STATUS+2);
    cyl = getbyte(ROM_DATA, DISK_STATUS+3) + ((temp & 0xc0) << 2);
    sector = temp & 0x3f;
    return;
}


/*
    disk_reset() - This routine will reset the disk system.
*/
disk_reset()
{
    char temp;

    /* Reset the drives */    
    rah = DIO_RESET;
    temp = rdl;
    rdl = 0x80;
    z150int(DISK_IO_INTR);
    rdl = temp;
    if (flags & CARRY) return(FALSE);

    /*	Recalibrate the drive */
    rah = DIO_HOME;
    rdl = drive + 0x80;
    z150int(DISK_IO_INTR);

    return(!(flags & CARRY));
}

/*
    get_sector() - This routine will read a sector from the winchester
	disk. It assumes all other parameters except rax have been set
	previously. (i.e. the externals head, sector, cyl and transfer
	address (res:rbx) are set). If the operation is sucessful TRUE
	is returned else FALSE is returned.
*/
get_sector()
{
    unsigned temp;
    int i;
    char flg;
    
    for (i = 0; i < NUM_RETRY; ++i)
    {
	encode();			/* Set the head, sector and cylinder */
	temp = rbx;
	rbx = buf;
	rdl = 0x80+drive;
	rah = DIO_READ;			/* Read function (restore in case of reset)*/
	ral = 1;			/* do 1 sector */
	z150int(DISK_IO_INTR);
	flg = !(flags & CARRY);
	if (flg) break;
	disk_reset();
    }
    if (flg)
    {
	movbyte(mydsreg(), buf, mydsreg(), temp, SECTOR_SIZE);
    }
    return(flg);	/* Return status */
}

/*
    put_sector() - This routine will put a sector to the disk
*/
put_sector()
{
    int i;
    char flg;

    movbyte(mydsreg(), rbx, mydsreg(), buf, SECTOR_SIZE);
    
    for (i = 0; i < NUM_RETRY; ++i)
    {
	encode();			/* Prepare for ROM call */
	rbx = buf;
	rdl = 0x80 + drive;
	rah = DIO_WRITE;	/* Restore these in case of reset */
	ral = 1;
	z150int(DISK_IO_INTR);
	flg = !(flags & CARRY);
	if (flg) break;
	disk_reset();
    }
    return(flg);
}

/*
    bs_full() - This routine will reset the pointer to the original
	fixed disk tables and tell the user the table is full. Then
	it will exit the PREP program.
*/
bs_full()
{
    retry_on();
    puts("\r\n\nBad sector count exceeded for this drive.\r\n");
    exit(0);
}
