	PAGE	,132
	TITLE	Z-150 System Functions

;**********************************************************************
;
;                       ------------------------
;----------------------- Z-150 System Functions -----------------------
;                       ------------------------
;
;	      Copyright (C) 1983, by Zenith Data Systems
;
;**********************************************************************

ROM_DATA SEGMENT AT (40H)
	EXTRN	DRIVE_MOTOR_COUNT:BYTE, DRIVE_STATUS:BYTE
ROM_DATA ENDS

MONITOR_SEGMENT SEGMENT WORD PUBLIC

	ASSUME CS:MONITOR_SEGMENT, DS:ROM_DATA, SS:NOTHING, ES:NOTHING

	INCLUDE	../ROM/ROM.LIT
	INCLUDE	../ROM/IO.LIT
	INCLUDE	../ROM/INTR.LIT
	INCLUDE ../DISK/DISK.LIT
	INCLUDE	../DISK/DISK.EXT
	INCLUDE ../VIDEO/VIDEO.LIT
	INCLUDE ../VIDEO/VIDEO.EXT
	INCLUDE	SYS.LIT
	INCLUDE	SYS.EXT

	EXTRN	DELAY:NEAR, DATA_SEGMENT:WORD, INIT_KEY_BUFF:NEAR

	PAGE

;**********************************************************************
; TIME_OF_DAY_INTERRUPT: (FUNCTION, PARAMETERS)	     Interrupt 1AH (26)
;
;	Time_of_Day_Interrupt is called to set or reset the current
; time-of-day.  It uses special units of time which are a multiple of
; the system timer, which interrupts the CPU at an 18.206482 Hz rate.
; This count is kept as a word pair (long unsigned integer), which
; contains the number of time periods since midnight.  An hour is
; represented as 0:65543, while a day is 24:176.
;	To execute a time-of-day function, load register AH with the
; function code which follows (as well as any parameters required by
; the particular operation desired).  Next, perform an 'INT 1AH' - 
; the function will be performed, and any results will be returned
; in the registers specified.
;
; Function Code 0 - Read the Current Time of Day
;   Output:
;	CX: Most significant word of timer count
;	DX: Least significant word of timer count
;	AL: Number of days which have passed since last time read
;
; Function Code 1 - Set the Time of Day
;    Input:
;	CX: Most significant count word
;	DX: Least significant count word
;**********************************************************************
TIME_OF_DAY_INTERRUPT PROC FAR
	PUBLIC	TIME_OF_DAY_INTERRUPT
	PUSHREG	<BX,DS>
	MOV	DS,DATA_SEGMENT		;Point to the current data segment
	TEST	AH,AH			;Are we to read the time of day?
	JNZ	TODI1			;No, set the time of day
	MOV	AL,TOD.DAYS		;Yes - read the day counter
	MOV	TOD.DAYS,0		;Clear day count for next read
	MOV	CX,TOD.MSCOUNT		;Get the most-significant timer word
	MOV	DX,TOD.LSCOUNT		;...and the least-significant word
	JMP	SHORT TODI2		;Return
TODI1:	MOV	TOD.DAYS,0		;Force day count to be 0
	MOV	TOD.MSCOUNT,CX		;Write the most-significant time word
	MOV	TOD.LSCOUNT,DX		;...and the least-significant word
TODI2:	POPREG	<DS,BX>
	IRET
TIME_OF_DAY_INTERRUPT ENDP



;**********************************************************************
; TIMER_INTERRUPT: 				     Interrupt 08H (08)
;
;	Timer_Interrupt is the system real-time clock entry.  Roughly
; 18 times a second it updates a set of real-time variables which can
; be set or read by the user 'TIME_OF_DAY_INTERRUPT' function call.
; This routine is responsible for turning the floppy disk drive motors
; off after they have not been used for a while, and for flashing the
; 'software' cursor in high-res graphics mode (if it is enabled).
;**********************************************************************
TIMER_INTERRUPT PROC FAR
	PUBLIC	TIMER_INTERRUPT
	PUSHREG	<DS,AX,DX>
	MOV	DS,DATA_SEGMENT		;Point to the current data segment
;
; Update Time-of-Day
;
	INC	TOD.LSCOUNT		;Increment the L.S. timer word
	JNZ	TI1			;No overflow - time O.K.
	INC	TOD.MSCOUNT		;Increment the M.S. timer word
TI1:	CMP	TOD.MSCOUNT,DAY_COUNTM	;M.S. count reached a day?
	JNZ	TI2			;No, time all set
	CMP	TOD.LSCOUNT,DAY_COUNTL	;L.S. count reached a day?
	JNZ	TI2			;No, finished with time update
	INC	TOD.DAYS		;Yes - increment day count
	MOV	TOD.MSCOUNT,0		;Set the most-significant word to 0
	MOV	TOD.LSCOUNT,0		;Set the least-significant word to 0
;
; Turn off floppy drive motors if unused
;
TI2:	MOV	AH,0			;Set compatible data value
	DEC	DRIVE_MOTOR_COUNT	;Has the floppy been used for awhile?
	JNZ	TI3			;Yes - let it keep spinning
	MOV	AH,0CH			;Set compatible data for drive timeout
	AND	DRIVE_STATUS,NOT DISK_MOTOR_MASKS ;No, flag all motors as off
	MOV	DX,DISK_CONTROL_PORT	;Point to the disk hardware port
	MOV	AL,DISK_IO_ENABLE	;Get the command to shut down
	OUT	DX,AL			;Turn the motors off
;
; Turn software cursor (for graphics) on/off if enabled
;
TI3:	TEST	CURSOR_FLAG,CURSOR_ENABLED  ;Is the cursor enabled?
	JZ	TI6			;No, ignore it
	PUSH	BX			;Yes - save the user's BX register
	MOV	BX,SCREEN_SEGMENT	;Get the screen segment in DS
	MOV	DS,BX			;Set/Reset the physical cursor
	MOV	BX,CURSOR_PTR		;Point to the cursor location
	INC	CURSOR_TIMER		;Increment the cursor time counter
	CMP	CURSOR_TIMER,CURSOR_ON_TIME ;Time to turn the cursor on?
	JNE	TI4			;No, continue
	CMP	CURSOR_ACCESS,FALSE	;Is it OK to update physical cursor?
	JNE	TI5			;No, skip memory write this pass
	TEST	CURSOR_FLAG,CURSOR_ON	;Is the cursor already on?
	JNZ	TI5			;Yes - don't turn it in again!
	OR	CURSOR_FLAG,CURSOR_ON 	;Turn the cursor on
	MOV	AL,[BX+SCAN_LINE_INCREMENT*3+SCREEN_PARTITION_OFS]
	MOV	CURSOR_SAVE,AL		;Save screen data at cursor position
	MOV	BYTE PTR [BX+SCAN_LINE_INCREMENT*3+SCREEN_PARTITION_OFS],CURSOR
	JMP	SHORT TI5		;Finished w/cursor update
TI4:	CMP	CURSOR_TIMER,CURSOR_OFF_TIME;Time to turn the cursor off?
	JNE	TI5			;No, execute user's intr routine
	MOV	CURSOR_TIMER,0		;Clear the cursor time counter
	CMP	CURSOR_ACCESS,FALSE	;Is it OK to update physical cursor?
	JNE	TI5			;No, skip memory write this pass
	TEST	CURSOR_FLAG,CURSOR_ON	;Is the cursor already off?
	JZ	TI5			;Yes - don't remove it again!
	AND	CURSOR_FLAG,NOT CURSOR_ON   ;Turn the cursor off
	MOV	AL,CURSOR_SAVE		;Restore old data at cursor position
	MOV	[BX+SCAN_LINE_INCREMENT*3+SCREEN_PARTITION_OFS],AL
TI5:	POP	BX			;Restore register BX
;
; Execute user-supplied interrupt service routine
;
TI6:	MOV	AL,AH			;Get compatible disk timeout value
	MOV	AH,0			;Set compatible value in AH
	INT	USER_CLOCK_INTR		;Execute user clock routine
	MOV	AL,END_OF_INTR		;Get the end-of-interrupt command
	OUT	INTERRUPT_CTRL,AL	;Let hardware know we are done
	POPREG	<DX,AX,DS>		;Restore registers
	IRET
TIMER_INTERRUPT ENDP



;**********************************************************************
; SET_MEMORY_SIZE:
;
;	Set_Memory_Size determines how much memory is in the system.
; This is done by reading the DIP switches in the Z-150 which contain
; the actual amount of memory placed in the machine.  The amount of 
; memory thus found is placed in the global variable 'MEMORY_SIZE'
; (as the number of K bytes).
;**********************************************************************
SET_MEMORY_SIZE PROC NEAR
	PUBLIC	SET_MEMORY_SIZE
	PUSHREG	<AX,BX,CX,DX>
	IFE	CHEAP			;If normal Z-150, read DIP switches to get mem
	CALL	READ_DIP_SWITCHES	;Read the system DIP switches in AX
	MOV	DX,AX			;Save copy of switch data in DX
	AND	AX,BUS_MEMORY_MASK	;Only interested in memory size
	MOV	CL,5			;Shift to get bus mem. size in K
	SHL	AX,CL
	MOV	BUS_MEMORY_SIZE,AX	;Record BUS memory size in global var
	MOV	BX,AX			;Save off-board memory size
	MOV	AX,DX			;Get a copy of the switch data again
	AND	AX,SYS_MEMORY_MASK SHL 8;Determine the amount of on-board RAM
	MOV	CL,6			;Make size into K byte count
	SHR	AX,CL
	ADD	AX,16			;Add in size of base 16K of RAM
	ADD	AX,BX			;Now have total RAM size in K
	MOV	MEMORY_SIZE,AX		;Save for later mem. config requests
	ELSE				;If cost-reduced machine...
	CALL	POLL_MEMORY_SIZE	;Poll to get system memory size
	MOV	MEMORY_SIZE,AX		;Record it in memory
	SUB	AX,64			;Subtract size of system memory
	MOV	BUS_MEMORY_SIZE,AX	;Update bus memory size variable
	ENDIF
	POPREG	<DX,CX,BX,AX>
	RET
SET_MEMORY_SIZE ENDP



;**********************************************************************
; GET_MEMORY_SIZE_INTERRUPT:
;
;	Get_Memory_Size_Interrupt returns the amount of memory in the 
; system.  The actual amount of memory is determined at power up - 
; this is set by dip switches.
;
; Output:
;	AX: Size of memory in K bytes. (ie, 64 = 40H = 64 Kbytes)
;**********************************************************************
GET_MEMORY_SIZE_INTERRUPT PROC FAR
	PUBLIC	GET_MEMORY_SIZE_INTERRUPT
	PUSHREG	<DS>
	MOV	DS,DATA_SEGMENT		;Point to the current data segment
	MOV	AX,MEMORY_SIZE		;Retrieve the size of memory
	POPREG	<DS>
	IRET
GET_MEMORY_SIZE_INTERRUPT ENDP



	IF	CHEAP			;Only for cost-reduced machine!

;***********************************************************************
; POLL_MEMORY_SIZE:
;
;	Poll_Memory_Size is called in the cost reduced machine
; configuration to determine the amount of available memory by
; polling all system memory in 64K chunks.
;
; Output:
;	AX: Memory size in K bytes
;***********************************************************************
POLL_MEMORY_SIZE PROC NEAR
	PUBLIC	POLL_MEMORY_SIZE
	PUSHREG	<BX,CX,ES>
	PUSHF				;Save the current interrupt enable status
	CLI				;Keep interrupts out here!
	MOV	BX,0			;Start with the base 64K
	MOV	CX,BX			;Set starting count to 0
PMS1:	MOV	ES,BX			;Point ES to this segment
	MOV	AL,BYTE PTR ES:[0]	;Get a byte from this segment
	MOV	AH,AL			;Save it in AH
	NOT	AL			;Complement the byte
	MOV	BYTE PTR ES:[0],AL	;Write the inverted byte to memory
	XCHG	AH,BYTE PTR ES:[0]	;Write original byte, get status of write
	XOR	AL,AH			;Did the first write operation take?
	JNZ	PMS2			;No, check for memory errors
	MOV	AL,AH			;Get the inverted byte again
	NOT	AL			;Convert it to the original byte
	XOR	AL,BYTE PTR ES:[0]	;Did the second write work?
	JZ	PMS4			;Yes - continue scanning
PMS2:	PUSH	CX			;No, save the current memory size
	MOV	CL,8			;Set repeat count to 8 bits/byte
	MOV	AH,0			;Set bad-bit count to 0
PMS3:	SHL	AL,1			;Get a bit into the carry flag
	ADC	AH,0			;If set, increment bad bit count
	LOOP	PMS3			;Check all 8 bits
	CMP	AH,1			;Did more than 1 bit fail?
	POP	CX			;[Restore the memory size]
	JA	PMS5			;Yes - done scanning memory
PMS4:	ADD	BX,1000H		;Point to the next memory segment
	ADD	CX,64			;Include this bank in memory size
	JMP	PMS1			;Continue till at end of memory
PMS5:	MOV	AX,CX			;Get the global memory size in AX
	POPF				;Restore old interrupt-enable status
	CMP	AX,128 * K		;Have we found at least 128K?
	JAE	PMS6			;Yes - go with this configuration
	MOV	AX,128 * K		;No, there's at least 128K here!
PMS6:	POPREG	<ES,CX,BX>
	RET
POLL_MEMORY_SIZE ENDP

	ENDIF



;**********************************************************************
; SET_IO_CONFIGURATION:
;
;	Set_IO_Configuration determines the number of peripherals
; attached to the system, and records the information in the global
; variable 'IO_CONFIG'.  See the Get_IO_Configuration routine for
; more details regarding the values recorded in this variable.
;**********************************************************************
SET_IO_CONFIGURATION PROC NEAR
	PUBLIC	SET_IO_CONFIGURATION
	PUSHREG	<AX,BX,CX,DX,DI>	;Save registers
	CALL	INIT_KEY_BUFF		;Initialize the keyboard buffer first
	MOV	AH,0			;Clear number of printers to 0
	MOV	BX,0			;Point to the base printer addresses
	MOV	DI,BX			;Point to the output addresses array
	MOV	CX,PRINTER_PORTS	;Number of printers to be checked
SIOC1:	MOV	DX,CS:PRINTER_BASE[BX]	;Get the base addr of a printer port
	MOV	AL,0AAH			;Write an alternating bit pattern...
	OUT	DX,AL			;...to the printer port as an ID
	PUSH	BX			;Make sure bus doesn't retain data
	IN	AL,DX			;Re-read printer port data
	POP	BX			;Clean up stack
	CMP	AL,0AAH			;Is there a valid port here?
	JNZ	SIOC2			;No, try the next port
	MOV	PRINTER_TABLE[DI],DX	;Yes - place the port addr in table
	ADD	DI,TYPE PRINTER_TABLE	;Point to the next output spot
	INC	AH			;Increment printer count
SIOC2:	ADD	BX,TYPE PRINTER_BASE	;Point to the next printer port
	LOOP	SIOC1			;Loop until all printers checked
	MOV	AL,0			;Extend printer count to a word
	MOV	CL,6			;Get shift count to put in right spot
	SHL	AX,CL			;Adjust the printer count
	PUSH	AX			;Save the current I/O configuration
	MOV	BX,0			;Point to bases of serial ports
	MOV	DI,BX			;Point to output array
	MOV	CX,SERIAL_PORTS		;Number of serial ports to check
	MOV	AH,0			;Clear number of serial ports to 0
SIOC3:	MOV	DX,CS:SERIAL_BASE[BX]	;Get the base addr of an RS-232 port
	ADD	DX,2			;Point to the interrupt ID register
	IN	AL,DX			;Read the interrupt ID
	AND	AL,SERIAL_INTR_MASK	;Is there a serial card here?
	JNZ	SIOC4			;No, try the next port
	SUB	DX,2			;Yes - point back to the base port
	MOV	SERIAL_TABLE[DI],DX	;Place port address in array
	ADD	DI,TYPE SERIAL_TABLE	;Point to the next output location
	INC	AH			;Increment serial port count
SIOC4:	ADD	BX,TYPE SERIAL_BASE	;Point to the next serial port
	LOOP	SIOC3			;Loop until all ports checked out
	MOV	AL,0			;Extend serial port count to a word
	SHL	AX,1			;Place port count in right bits
	POP	CX			;Get old I/O config in CX
	OR	CX,AX			;OR in printer and serial counts
	MOV	DX,GAME_BASE		;Point to the base port of game card
	IN	AL,DX			;Read the card's port
	AND	AL,GAME_CARD_MASK	;Is a valid port here (no ADC)?
	JNZ	SIOC5			;No, continue
	OR	CX,GAME_FLAG		;Yes - flag game card in IO_CONFIG
SIOC5:	MOV	AX,IO_CONFIG		;Get the old I/O configuration
	AND	AX,GAME_FLAG		;Isolate the game card bit
	OR	CX,AX			;If game card was there, it still is!
	CALL	READ_DIP_SWITCHES	;Read the system config switches
	OR	CL,AH			;Place DIP switch #1 data in config
	MOV	IO_CONFIG,CX		;Record global variable
	POPREG	<DI,DX,CX,BX,AX>	;Restore registers
	RET
SET_IO_CONFIGURATION ENDP



;**********************************************************************
; GET_IO_CONFIG_INTERRUPT:
;
;	Get_IO_Config_Interrupt returns the current status of peripherals
; attached to the system at power-up (the configuration is actually
; determined then).
;
; Output:
;	AX: Current I/O device configuration, as follows:
;		Bits 14-15:	Number of printers attached to system
;		Bit 13:		Currently unused
;		Bit 12:		Game I/O board in system
;		Bits 9-11:	Number of RS-232 ports in system
;		Bit 8:		Currently unused
;		Bit 6-7:	Number of 5-1/4" disks.  If these
;				bits are 0, and bit 0 = 1, then 4
;				drives are currently connected.
;		Bit 4-5:	Initial video mode, as follows:
;				00:	Fast 80x25 Graphics mode
;				01:	40x25 Color card mode
;				02:	80x25 Color card mode
;				03:	80x25 Monochrome card mode
;		Bit 2-3:	Amount of RAM up to 64K:
;				00:	16K RAM
;				01:	32K RAM
;				02:	48K RAM
;				03:	64K RAM
;		Bit 1:		8087 attached to the system
;		Bit 0:		Floppy drives connected
;**********************************************************************
GET_IO_CONFIG_INTERRUPT PROC FAR
	PUBLIC	GET_IO_CONFIG_INTERRUPT
	PUSHREG	<DS>
	MOV	DS,DATA_SEGMENT		;Point to the current data segment
	MOV	AX,IO_CONFIG		;Get the I/O configuration word
	POPREG	<DS>
	IRET
GET_IO_CONFIG_INTERRUPT ENDP



	IF	CHEAP

;***********************************************************************
; SET_DIP_SWITCHES:
;
;	Set_DIP_Switches is called in the cost-reduced machine to
; synthesize the values used by the IBM-compatible DIP switches
; by using the 4 available jumpers and by polling the hardware to
; determine device availability.
;***********************************************************************
SET_DIP_SWITCHES PROC NEAR
	PUBLIC	SET_DIP_SWITCHES
	PUSHREG	<AX,CX,DX>
	MOV	DX,0			;Set virtual DIP switches to 0
	CALL	READ_DIP_SWITCHES	;Read the physical DIP switches
	MOV	AL,AH			;Get the DIP values in AL
	AND	AL,CHEAP_REFRESH_RATE	;Get video refresh rate
	OR	DL,AL			;Place boot & refresh in switch 2 value
	MOV	AL,BOOT_FLOPPY		;Assume that the floppy will be booted
	TEST	AH,CHEAP_BOOT_MASK	;Is the winchester to be booted?
	JZ	SDP1			;No, have setting all set
	MOV	AL,BOOT_WINCHESTER	;Yes - set boot-from-winchester bit
SDP1:	OR	DL,AL			;Set in boot drive selection
	MOV	AL,AH			;Get DIP switch values again
	AND	AL,CHEAP_VIDEO		;Mask out video select
	OR	DH,AL			;Place it in switch 1 value
	OR	DH,VIDEO_MODE_80	;Always have 80 character display!
    	CALL	POLL_8087		;See if the 8087 is installed
    	AND	AL,FPP_INSTALLED	;Set right bit if so...
    	OR	DH,AL			;OR in 8087-installed if it is!
	TEST	AH,CHEAP_FLOPPY		;Are floppy disks attached?
	JZ	SDP2			;No, skip drive stuff
	OR	DH,FLOPPY_ATTACHED	;Yes - show that we have drives
	MOV	AL,DRIVE_COUNT		;Get the drive count in AL
	DEC	AL			;Decrement it for switch value
	MOV	CL,6			;Get shift for right position
	SHL	AL,CL			;Place drive count in right bits
	OR	DH,AL			;Copy number of drives into switch 1
SDP2:	OR	DH,SYS_MEMORY_MASK	;Always have at least 64K of memory!
    	MOV	AX,BUS_MEMORY_SIZE	;Get the expansion memory size
    	MOV	CL,5			;Get shift to divide into 32K banks
    	SHR	AX,CL			;Get number of 32K banks
    	OR	DL,AL			;Include memory size in switch 2
    	MOV	AX,DX			;Get DIP switches in AX
    	XCHG	AL,AH			;Get switch 1 value in AL
    	MOV	DX,SWITCH_1_INIT	;Point to DIP switch 1 port
    	OUT	DX,AL			;Set DIP switch number one
    	MOV	AL,AH			;Get switch 2's value
    	MOV	DX,SWITCH_2_INIT	;Point to switch 2 port
    	OUT	DX,AL			;Set value for DIP switch 2
	POPREG	<DX,CX,AX>
	RET
SET_DIP_SWITCHES ENDP

	ENDIF



;**********************************************************************
; READ_DIP_SWITCHES:
;
;	Read_DIP_Switches is used to read the values of the system
; configuration switches 1 and 2.
;
; Output:
;	AH: Value for DIP switch 1, as follows:
;		Bit 6-7:	Number of 5-1/4" disks.  If these
;				bits are 0, and bit 0 = 1, then 4
;				drives are currently connected.
;		Bit 4-5:	Initial video mode, as follows:
;				00:	80x25 Graphics mode
;				01:	40x25 Text mode
;				02:	80x25 Text mode
;				03:	80x25 Monochrome mode
;		Bit 2-3:	Amount of RAM up to 64K:
;				00:	16K RAM
;				01:	32K RAM
;				02:	48K RAM
;				03:	64K RAM
;		Bit 1:		8087 numeric coprocessor attached
;		Bit 0:		Floppy drives connected
;	AL: Value for DIP switch 2, as follows:
;		Bit 7:		Video refresh rate select:
;				0:	60 Hertz display
;				1:	50 Hertz display
;		Bit 5-6:	Boot drive select:
;				00:	No auto-boot
;				01:	5-1/4" Drive 0 auto-boot
;				10:	Winchester Drive 0 auto-boot
;				11:	Reserved
;		Bit 0-4:	System memory size (in 32K byte increments).
;				This value does NOT include the first
;				64K of RAM memory (ie, a value of 8 equals
;				8*32=256K of ADDITIONAL RAM - indicating a
;				system total of 256K + 64K = 320K).
;**********************************************************************
READ_DIP_SWITCHES PROC NEAR
	PUBLIC	READ_DIP_SWITCHES
	PUSHREG	<BX,CX>
	IN	AL,SYS_CTRL_PORT	;Read the hardware control port
	MOV	BX,AX			;Save the status of the port
	MOV	AL,READ_SWITCH_A	;Get the I/O command to read DIP sws
	OUT	SYS_CTRL_PORT,AL	;Enable DIP switches
	IN	AL,SYS_SWITCH_1		;Read the value for Switch #1
	MOV	AH,AL			;Place it in AH
	IN	AL,SYS_SWITCH_2		;Place value for Switch #2A in AL
	AND	AL,SWITCH_2_MASK	;Mask out all but switch data
	MOV	BH,AL			;Save data in BH
	MOV	AL,READ_SWITCH_B	;Get code to read other half of SW #2
	OUT	SYS_CTRL_PORT,AL	;Turn on Switch #2, part B
	IN	AL,SYS_SWITCH_2		;Read Switch #2, bits 4-7
	MOV	CL,4			;Get shift to put in high nibble
	SHL	AL,CL
	OR	AL,BH			;OR in to get all of Switch #2 in AL
	XCHG	BX,AX			;Put switch data in BX, ctrl in AX
	OUT	SYS_CTRL_PORT,AL	;Restore the control port status
	MOV	AX,BX			;Get the dip switch data again
	POPREG	<CX,BX>
	RET
READ_DIP_SWITCHES ENDP



	IF	CHEAP

;***********************************************************************
; POLL_8087:
;
;	Poll_8087 is called in the cost-reduced machine to determine
; if an 8087 floating-point coprocessor is installed in the machine.
; Note that the mnemonics are coded as DBs, to prevent the FWAITs
; automatically inserted by the Microsoft assembler from hanging up
; a system without an 8087 chip.
;
; Output:
;	AL: True = 8087 installed, False = 8087 not present
;***********************************************************************
POLL_8087 PROC NEAR
	PUBLIC	POLL_8087
	PUSHREG	<CX>
	MOV	WORD PTR BUFF_8087,1234H;Place a known value in 8087 buffer
	DB	0DBH, 0E3H		;FINIT	;Init Chip
	MOV	CX,5			;Delay for chip to catch up
	CALL	DELAY
	DB	0D9H, 0E8H		;FLD1	;Load 1, and store at BUFF_8087
	MOV	CX,5			;Wait for i to load the constant
	CALL	DELAY
	DB	02EH, 0DBH, 01EH	;FISTP CS: BUFF_8087
	DW	BUFF_8087
	MOV	CX,5			;Get 1-millisecond delay constant
	CALL	DELAY			;Wait for the 8087 to catch up
	CMP	WORD PTR BUFF_8087,1234H;Is the 8087 home?
	MOV	AL,FALSE		;[Assume not...]
	JE	P87			;No, return FALSE
	MOV	AL,TRUE			;Yes - return TRUE!
P87:	POPREG	<CX>
	RET
POLL_8087 ENDP

	ENDIF



;**********************************************************************
;		  B A S E   P O R T   A D D R E S S E S
;**********************************************************************
		PUBLIC	PRINTER_BASE, SERIAL_BASE
PRINTER_BASE	DW	3BCH, 378H, 278H	;Base printer port addresses
SERIAL_BASE	DW	3F8H, 2F8H		;Base serial port addresses



MONITOR_SEGMENT ENDS

	END

