;**********************************************************************
; TERMINAL_DISPLAY_8025: (CHAR)
;
;	Terminal_Display_8025 is used as a special case routine for
; high-speed terminal-mode display of characters while in standard
; 80x25 mode. Terminal_Display_8025, like Terminal_Display, accepts 
; an ASCII character, and outputs it to the display.  The control 
; characters CR (Carriage Return), LF (Line Feed), BS (Backspace) 
; and BELL are processed as commands, rather than being displayed.
;
;	Terminal_Display_Mono is the alternate entry point used for
; high-speed display of information on the monochrome card.
;
;	One important note is that the structure of this routine is
; rather unorthodox - this was done to optimize execution time.  As
; a result of this structure, the routine assumes that page 0 is
; being written to.
;
; Input:
;	AL: Character to be displayed / control character
;**********************************************************************
	PUBLIC	TERMINAL_DISPLAY_8025, TERMINAL_DISPLAY_MONO
TERMINAL_DISPLAY_MONO:
	PUSH	SI
	MOV	SI,PAGE_BASE[8 * 2]	;Always have page 8 in monochrome mode
	JMP	SHORT TDT0

TERMINAL_DISPLAY_8025 PROC NEAR
	CMP	DISPLAYED_PAGE,0	;Displaying on page 0?
	JE	TD80			;Yes - can use fast routine
	JMP	TERMINAL_DISPLAY	;No, use the general-purpose routine
TD80:	PUSH	SI
	MOV	SI,PAGE_BASE[0 * 2]	;Assume page 0 for color card

TDT0:	PUSHREG	<AX,BX,DX,DI,ES>	;Save registers
	MOV	ES,VIDEO_SEGMENT	;Point ES to the screen segment
	MOV	DX,CURSOR_POSITION[0]	;Place the ROW in DH, COLUMN in DL
	MOV	BL,DH			;Get the current row number
	MOV	BH,0			;Extend it to a word offset
	SHL	BX,1
    	DEC	BH			;Make BH = FF if ROW<7F, 0 if >= 80H
    	NOT	BH			;Sign extend the cursor posn offset
	MOV	BX,CS:TEXT_LINE_OFFSET[BX] ;Pick up the offset to this line
	ADD	BX,SI			;Add in address of start of page
	ADD	BL,DL			;Now point to the current character
	ADC	BH,0
	ADD	BL,DL			;Must add twice to include attributes
	ADC	BH,0
	MOV	DI,BX			;Copy pointer into DI as a text ptr
	AND	DI,TEXT_SCREEN_MASK	;Ensure that the ptr is valid
	SHR	BX,1			;Make BX a hardware pointer
;
; Check for possible control character
;
TDT1:	CMP	AL,' '			;Might this be a control char?
	JB	TDT5			;Just might be - display control chars
;
; Display the character in 80x25 text mode
;
TDT2:	MOV	BYTE PTR ES:[DI],AL	;Display the character on the screen
	INC	BX			;Point to the next character
	INC	DL			;Advance cursor to next column
	CMP	DL,MAX_COLUMNS		;Beyond end of line?
	JAE	TDT9			;Yes - advance to next line
;
; Update the cursor pointer register, and cursor position
;
TDT3:	MOV	CURSOR_POSITION[0],DX	;Update this page's cursor position
	CMP	DISPLAYED_PAGE,0	;Is this page being displayed?
	JNE	TDT4			;No, skip update of hardware cursor
	MOV	DX,VIDEO_BASE		;Point DX to the CRT chip
	MOV	AL,CRT_CURSOR_START	;Point to the CRT cursor address reg
	OUT	DX,AL			;Select M.S. byte of cursor addr
	INC	DX			;Now, point to the parameter port
	MOV	AL,BH			;Get the M.S. cursor addr byte
	OUT	DX,AL			;Output it...
	DEC	DX			;Point back to the control port
	MOV	AL,CRT_CURSOR_START+1
	OUT	DX,AL			;Select L.S. byte of cursor address
	INC	DX			;Point to parameter port again
	MOV	AL,BL			;Output L.S. byte of address
	OUT	DX,AL
TDT4:	POPREG	<ES,DI,DX,BX,AX>	;Restore the user's registers
	POP	SI
	RET
;
; Process possible control characters
;
TDT5:	CMP	AL,CR			;Is the char a carriage return?
	JE	TDT7			;Yes - go to start of line
	CMP	AL,LF			;Is this a line-feed?
	JE	TDT10			;Yes - advance to next line
	CMP	AL,BELL			;Is the character a BELL?
	JE	TDT8			;Yes - beep!
	CMP	AL,BS			;Is the char a backspace?
	JNE	TDT2			;No, display character as usual
;
; Process Backspace
;
TDT6:	TEST	DL,DL			;At the start of the line already?
	JZ	TDT4			;Yes - return
	DEC	DL			;No, decrement the column number
	DEC	BX			;Point to the previous character
	JMP	SHORT TDT3		;Set cursor position and return
;
; Process Carriage Return
;
TDT7:	SUB	BL,DL			;Subtract horizontal offset...
	SBB	BH,0			;...to point to start of line
	MOV	DL,0			;Now, set column to 0
	JMP	SHORT TDT3		;Set cursor ptr to start of line
;
; Process Bell character (BEEP)
;
TDT8:	CALL	BEEP			;Beep!
	JMP	SHORT TDT4		;Return without cursor update
;
; Handle Automatic CR-LF at end of line
;
TDT9:	SUB	BL,DL			;Point back to start of line
	SBB	BH,0			;...first, subtract offset for chars
	MOV	DL,0			;Set position to start of line
;
; Process Line Feed
;
TDT10:	CMP	DH,LAST_ROW		;On the last line already?
	JNE	TDT11			;No, continue
	CALL	FAST_SCROLL_UP_8025	;Yes, scroll the display up a line
	CMP	SCROLL_MODE,0		;Did the display RAM advance?
	JE	TDT3			;No, continue with same screen addr
	JMP	SHORT TDT12		;Yes - advance to next line & return
TDT11:	INC	DH			;Go on to the next ROW
TDT12:	ADD	BX,80			;Advance cursor ptr as well as posn
	JMP	SHORT TDT3		;Set cursor position and return
TERMINAL_DISPLAY_8025 ENDP



;**********************************************************************
; FAST_SCROLL_UP_8025: (CHAR_PTR)
;
;	Fast_Scroll_Up_8025 scrolls the screen up a single line when 
; using the Terminal_Display_8025 routine.  Note that this routine
; provides a fast 'software' scroll mode routine - for other scrolling
; modes, it uses the general-purpose scrolling routine.  Note that
; this routine may be called either for color or monochrome displays.
;
; Input:
;	SI: Pointing to the base address of this page
;	DI: Pointing to the last character position prior to scroll.
;
; NOTE: Register AX is used, but NOT saved, for efficiency.
;**********************************************************************
FAST_SCROLL_UP_8025 PROC NEAR
	PUBLIC	FAST_SCROLL_UP_8025
	PUSHREG	<CX,SI,DI,DS>
	CMP	SCROLL_MODE,0		;In software (compatible) scroll mode?
	JE	FSU81			;Yes - perform special fast scroll
	MOV	AL,1			;No, set scroll count to 1
	MOV	CX,(0 SHL 8) OR 0	;Set upper-left hand position
	PUSH	BX			;Save register BX
	PUSH	DX			;Save register DX
	MOV	DH,LAST_ROW		;Set lower-right hand position
	MOV	DL,79
	MOV	BH,BYTE PTR ES:[DI+1]	;Get the last attribute for the scroll
	CALL	SCROLL_UP		;Scroll the screen up a line
	POP	DX			;Restore register DX
	POP	BX			;Restore register BX
	JMP	FSU83			;Return
FSU81:	MOV	DI,SI			;Point to the start of this page
	ADD	SI,MAX_COLUMNS * 2	;Point SI to the second line
	MOV	CX,24 * MAX_COLUMNS	;Get size of screen to scroll
	CMP	LAST_ROW,NUMBER_OF_ROWS-1;Is this a 'normal' page?
	JE	FSU82			;Yes - have screen size all set up
	MOV	AL,LAST_ROW		;No, point to the start of the last row
	MOV	CL,MAX_COLUMNS		;Get the length of each line
	MUL	CL			;Find number of chars to be scrolled
	MOV	CX,AX			;Place length in CX
FSU82:	PUSH	ES			;Save the pointer to the screen seg
	POP	DS			;Point DS to the screen sgment
	REP	MOVSW			;Scroll the screen up a line
	MOV	AH,BYTE PTR ES: [DI-1]	;Get the attribute from the screen
	MOV	AL,' '			;Fill last line with spaces
	MOV	CX,MAX_COLUMNS		;Get length of last line to erase
	REP	STOSW			;Clear last line of display
FSU83:	POPREG	<DS,DI,SI,CX>
	RET
FAST_SCROLL_UP_8025 ENDP




