;*********************************************************************
; READ_CHAR: (PAGE)
;
;	Read_Char is called to read characters from the
; screen RAM.  In alphanumeric screen modes, it reads the
; screen RAM directly;  in graphics modes, however, it com-
; pares the pixels on the screen with both the ROM font, and
; the user's font.  If a match is found, the Ascii value of
; the character is returned.
;
; Input:
;	BH: Page to be read
;
; Output:
;	AL: Character value read
;	AH: Attributes for the character
;*********************************************************************
READ_CHAR PROC NEAR
	PUBLIC	READ_CHAR
	PUSHREG	<CX,DX,SI,DI,DS,ES>
	CALL	SET_IO_PAGE		;Select the correct page
	CALL	SET_SCREEN_ADDRESS	;Point ES:DI to the right character
	CMP	VIDEO_MODE,7		;Is monochrome mode?
	JE	RC0			;Yes - read the character directly
	CMP	VIDEO_MODE,3		;In alphanumeric mode?
	JG	RC1			;No, have to scan through font
RC0:	MOV	AX,WORD PTR ES: [DI]	;Pick up character + attribute byte
	JMP	SHORT RC8		;Return with value in AX
RC1:	MOV	AX,MONITOR_SEGMENT	;Point DS:SI to font in ROM
	MOV	DS,AX
	MOV	SI,OFFSET FONT
	MOV	DH,2			;Number of passes (ROM+User-defined)
	MOV	DL,0			;Use DL as a character number
RC2:	PUSH	SI			;Save pointer to font
	PUSH	DI			;Save pointer to screen
	MOV	CX,FONT_HEIGHT		;Number of font bytes / char
RC3:	PUSH	DS			;Save font segment
	MOV	DS,DATA_SEGMENT		;Point to the current data segment
	CMP	VIDEO_MODE,5		;Are we in med-res graphics mode?
	POP	DS			;[Restore font segment]
	JA	RC4			;No, in high-resolution
	MOV	AX,WORD PTR ES: [DI]	;Yes - get the graphics word
	XCHG	AL,AH			;Place bytes in right positions
	CALL	COMPRESS_FONT		;Compress word to a byte
	JMP	RC5			;Continue
RC4:	MOV	AL,BYTE PTR ES: [DI]	;Get a graphics byte from the screen
RC5:	CMP	AL,BYTE PTR DS: [SI]	;Does it match this font byte?
	JNE	RC7			;No, try next font character
	XOR	DI,SCREEN_PARTITION_OFS	;Yes - point to the even/odd line
	INC	SI			;Point to the next font byte
	TEST	CX,1			;Did we just read an even scan line?
	JE	RC6			;Yes - continue
	ADD	DI,SCAN_LINE_INCREMENT	;No - point to the next even line
	CMP	DI,OFFSET END_OF_PART	;Beyond end of screen?
	JBE	RC6			;No, continue
	SUB	DI,SCREEN_BYTES	SHR 1	;Yes - wrap around to rest of screen
RC6:	LOOP	RC3			;Loop until match / not-match
	POP	DI			;Found Char! Restore ptr to char
	POP	SI			;Clean font ptr off of stack
	MOV	AH,BYTE PTR ES: [DI+1]	;Get the attribute (if any)
	MOV	AL,DL			;Put the char index in AL
	JMP	SHORT RC8		;Done!
RC7:	POP	DI			;Restore pointer to start of char
	POP	SI			;Restore pointer to font
	ADD	SI,FONT_HEIGHT		;Point to the next font char
	INC	DL			;Increment character index
	TEST	DL,7FH			;Reached end of a pass yet?
	JNE	RC2			;No, continue
	MOV	AL,USER_FONT_INTR	;Yes - get the user font int #
	CALL	GET_INTR_PTR		;Point to the user's font
	DEC	DH			;Tried both fonts yet?
	JNE	RC2			;No, loop until all tried
	MOV	AX,0			;Couldn't find char - return 0
RC8:	POPREG	<ES,DS,DI,SI,DX,CX>
	RET
READ_CHAR ENDP



;**********************************************************************
; READ_PIXEL: (VERT, HOR)
;
;	Read pixel is called in the graphics modes to
; read individual graphics pixels.
;
; Input:
;	DX: Vertical position (0-199)
;	CX: Horizontal position (0-319, or 0-639)
;
; Output:
;	AL: Pixel value (1 or 2 bits, depending on whether
;	    high or medium resolution graphics mode is 
;	    currently active).
;**********************************************************************
READ_PIXEL PROC NEAR
	PUBLIC	READ_PIXEL
	PUSHREG	<BX,CX,DI,ES>
	MOV	AH,FALSE		;Set graphics mode as parameter
	CALL	VIDEO_PTR		;Point to the correct byte location
	MOV	BL,AH			;Point to the correct pixel mask...
	MOV	BH,0			;...for this bit position
	DEC	AL			;Is there one bit/pixel?
	JE	RP1			;Yes - no need to adjust bit number
	SHL	AH,1			;No, get the starting bit number
	INC	AH			;Adjust for second pixel bit
RP1:	MOV	CL,7			;Make bit count from 0-7, into 7-0
	SUB	CL,AH
	MOV	AL,CS:HIGH_RES_MASKS[BX];Get the high-res mask for this bit
	CMP	VIDEO_MODE,6		;In high-resolution mode?
	JE	RP2			;Yes - continue
	MOV	AL,CS:MED_RES_MASKS[BX]	;No, get the medium-res bit mask
RP2:	AND	AL,BYTE PTR ES: [DI]	;Get the correct pixel of screen RAM
	SHR	AL,CL			;Right-align the pixel
RP3:	POPREG	<ES,DI,CX,BX>
	RET
READ_PIXEL ENDP




