page	60,132
title	MindSet Set Genlock Trap

;******************************************************************
; This program allows the user to interactively set
; the display sync features of the Mindset as well as
; to set/clear the key bit for any index in the palette.
;
; When this program is run, it traps the PRTSCN interrupt.
; Then, when PRTSCN is typed, if both shift keys are down,
; this program accepts a two-character
; command--otherwise the normal PRTSCN routine is run.
;
; The first character is the command and the
; second character is a single (ascii) hex digit '0'..'F'
;
; The commands are:
;	    B: Select border color index
;	    T: Toggle key bit of palette index 0..F
;	    S: Set key bit of palette index 0..F
;	    C: Clear key bit of palette index 0..F
;	    V: Set the Video display sync features to hex value 0..F
;	<ESC>: Exit
;
; Invalid commands nop
;******************************************************************

;Equates:
prtscint	equ	5	;PRTSCN SW interrupt number
backspace	equ	8	;ascii backspace
escape		equ	01bh	;ascii escape

; DOS interrupt function codes (use with interrupt 21h...move these to AH)
;	Console functions
dfc_set_vector	equ	25h		;set int vector (al=int #,ds:dx=vec)
dfc_term_stay	equ	31h		;terminate prog but stay resident
dfc_get_vector	equ	35h		;get int vector (in cs:ip)(al=int #)

;Mindset unique interrupts
mindset_video_int	equ	0efh	;graphics interrupt
ms_set_palette		equ	00ah
ms_get_palette		equ	00bh

mindset_io_int		equ	0eeh	;other unique
ms_set_syncs		equ	006h	;input in al



cseg	segment para public 'code'
	assume	cs:cseg,ds:cseg,ss:nothing,es:nothing

; PrtSc interrupt handler (interrupt 5.)

int_5	proc	far
	sti			;allow interrupts
	cmp	cs:[reent],0	;reentrant?
	je	not_reent	;no-jump
	iret			;yes-nop

not_reent:
	mov	cs:[reent],-1	;set reentrant flag
	push	ax		;save on previous stack
	push	di		;save on previous stack
	push	es		;save on previous stack

;Both shift keys depressed?
	mov	ax,0200h
	int	16h		;get shift status of keyboard
	and	al,3		;mask off all but shift key status
	cmp	al,3		;are both down?
	je	docmd		;yes-jump
	pop	es		;no-restore stack and jump to PRTSCN
	pop	di
	pop	ax
	mov	cs:[reent],0	;clear reentrant flag
	jmp	dword ptr cs:[old_vector]

docmd:
	cli			;off interrupts
	mov	di,ss		;temporary
	mov	ax,cs		;new stack is in code segment
	mov	ss,ax		;begin switch to new stack
	mov	ax,sp		;temporary
	sti			;allow interrupts after next instruction
	mov	sp,offset top_of_stack  ;complete the switch

	irp	x,<ax,bx,cx,dx,bp,si,di,ds>  ;save registers
	push	x
	endm

	mov	ax,cs
	mov	ds,ax		;data segment is in code segment
	mov	es,ax		;so is extra segment

;flush type ahead
flush:	mov	ah,1
	int	16h		;check kb buffer
	jz	getcmd		;jump if empty
	mov	ah,0
	int	16h		;read character
	jmp	short flush	;check for more characters

getcmd:
	mov	ax,0e07h
	int	10h		;beep
	mov	ah,0
	int	16h		;read command character
	cmp	al,escape	;escape?
	jne	tback1		;no-jump
	jmp	exit
tback1:	cmp	al,backspace	;backspace?
	je	getcmd		;yes-jump
	mov	dh,al		;dh = command character
	mov	ah,0
	int	16h		;read hex digit character
	cmp	al,escape	;escape?
	jne	tback2		;no-jump
	jmp	exit
tback2:	cmp	al,backspace	;backspace?
	je	getcmd		;yes-jump
	mov	dl,al		;dl = hex digit
	or	dh,20h		;force command to lowercase

;convert the ascii hex digit to binary
	cmp	dl,'0'		;below '0'?
	jnb	t9		;no-jump
	jmp	error
t9:	cmp	dl,'9'		;in range 0..9?
	ja	af		;no-jump
	sub	dl,'0'		;convert to binary
	jmp	short cmds	;go do command
af:	or	dh,20h		;force lowercase
	cmp	dl,'a'		;below 'a'?
	jnb	tf		;no-jump
	jmp	error
tf:	cmp	dl,'f'		;above 'f'?
	ja	error
	sub	dl,('a' - 10)	;convert to binary

;do the proper command
cmds:	cmp	dh,'v'		;set video sync?
	jne	readpal		;no-jump
	mov	ah,ms_set_syncs
	mov	al,dl
	int	mindset_io_int	;set the syncs
	jmp	short getcmd

;all other commands modify the palette
readpal:
	mov	cmd,dh		;save command byte
	xor	dh,dh		;dx = palette index to read
	mov	ah,ms_get_palette
	mov	bx,offset color
	mov	cx,1
	pushf			;for fake int ef
	call	dword ptr [ef_vector]

	cmp	cmd,'b'		;set border color?
	jne	tt		;no-jump
	mov	al,dl		;set border color
	jmp	short writepal

tt:	cmp	cmd,'t'		;toggle key bit?
	jne	ts		;no-jump
	xor	color,0200h	;toggle it
	jmp	short writepal

ts:	cmp	cmd,'s'		;set key bit?
	jne	tc		;no-jump
	or	color,0200h	;set it
	jmp	short writepal

tc:	cmp	cmd,'c'		;clear key bit?
	jne	error		;no-jump
	and	color,0fdffh	;clear it

writepal:
	mov	ah,ms_set_palette
	pushf			;for fake int ef
	call	dword ptr [ef_vector]
	jmp	getcmd

error:
	mov	cx,100
beeep:	mov	ax,0e07h
	int	10h		;beep
	loop	beeep
	jmp	getcmd

exit:
	mov	ax,0e07h
	int	10h		;beep
	irp	x,<ds,di,si,bp,dx,cx,bx,ax>  ;restore from my stack
	pop	x
	endm
	cli			;no interrupts
	mov	ss,di		;begin switch to previous stack
	sti			;interrupts ok after next instruction
	mov	sp,ax		; and complete it

	pop	es		;restore
	pop	di		;restore
	pop	ax		;restore
	mov	cs:[reent],0	;clear reentrant flag
	iret			;finished
int_5	endp

even

;**********
;Data in code segment
int_5_ptr		dd	int_5	;pointer to interrupt processor
old_vector		dd	?	;BIOS 'print screen' vector

ef_vector		dd	?	;store BIOS vector for compiled BASIC

reent			db	0	;reentrant flag
cmd			db	?	;command character
color			dw	?	;palette color


;**********
; Stack in code segment
my_stack	db	32 dup('stack   ')
top_of_stack	dw	0,0

	db	'End of SetGen'

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


; The one-time initialization code.

; Establish the correct data segment pointer, change the PrtSc vector
; to point to the new interrupt processing routine, and terminate
; but stay resident.

; *** Initialization...code beyond this point is executed once and
; the space is then returned to DOS for general usage.

init	proc	far

	mov	ax,cs
	mov	ds,ax		;map to data segment

;Get the 0efh vector before compiled basic has a chance to trash it.
;We can then fake the INT EFH via PUSHF/CALL [EF_VECTOR]
	mov	ah,dfc_get_vector
	mov	al,0efH
	int	21h
	mov	word ptr ef_vector,bx
	mov	word ptr ef_vector+2, es

;replace the current int 5 vector with our own
	mov	ah,dfc_get_vector
	mov	al,prtscint	;interrupt number
	int	21h		;returns es:bx=vector contents
	mov	word ptr old_vector,bx		;save offset
	mov	word ptr old_vector+2,es	; and segment

	mov	ah,dfc_set_vector	;set new vector
	mov	al,prtscint	;printer interrupt number
	lds	dx,int_5_ptr	;pointer to int processor
	int	21h

	mov	ah,09h		;dos print string
	mov	dx,offset msg	;point to message (ds=cs)
	int	21h		;print the message

	mov	dx,(((init-int_5) shr 4) + 17) ;# of paragraphs
	xor	al,al		;return code = 0
	mov	ah,dfc_term_stay	;terminate but stay resident
	int	21h

msg	db	'Genlock Trap V1.00 Installed.',13,10,'$'
init	endp

cseg	ends


STK	SEGMENT PARA STACK 'STACK'
	DB	200 dup (?)
STK	ENDS

	end	init
