/******************************************************************************
 *
 * FSTREAD - read RTE FST archive.
 * Ken Cornetet - ken@cornetet.org
 *
 * Usage: fstread [-V] [-v] [-b] Archive.fst
 *        -V = verbose - prints debug info
 *        -v = view contents, don't restore files
 *        -b = binary - don't convert RTE type 3 and type 4 files to
 *             standard newline ended files.
 *
 *
 * This is public domain code. You may use this file for any purpose you choose.
 *
 *
 ******************************************************************************/


#include <stdio.h>
#include <time.h>
#include <fcntl.h>
#include <memory.h>
#include <errno.h>
#include <ctype.h>
#include <string.h>

#ifndef O_BINARY
#define O_BINARY 0
#endif


struct archive_hdr *GetArchiveHeader( int fd );
struct file_hdr *GetFileHeader( int fd );
unsigned long MakeInt32( unsigned char *p );
unsigned MakeInt16( unsigned char *p );
int RestoreData(char *namr, int fd, long blocks );

unsigned MakeInt16( unsigned char *p );
void BadError( char *p );
long GetRecLen( int fd );


long pos = 0;

struct archive_hdr {
	char archive[101];			/* archive file (?) */
	unsigned chksum;			/* checksum of archive header */
	char version[65];			/* version of FST used to create the archive */
	unsigned long sz_comment;	/* Comment file size in "blocks" */			
	unsigned tapenum;			/* tape number */
	unsigned long firstfile;	/* "record" pointer into the FST directory file for first file */
	unsigned long startblock;	/* starting block number for the data of a file split across tape boundires */
	unsigned split;				/* flag - first file is split across tapes */
	char headertype[7];			/* header type ("FST") */
	time_t created;				/* creation time in standard unix epoch */
};

struct file_hdr {
	char desc[101];				/* file descriptor */
	unsigned mode;				/* protection bits */
	unsigned long bytes;		/* file size in bytes */
	unsigned chksum;			/* checksum */
	int lu;						/* source disk LU */
	unsigned sec_trk;			/* sectors/track */
	unsigned long blocks;		/* disk size in blocks */
	unsigned hier;				/* heirarchical file flag */
	unsigned dirent[32];		/* directory entry */
	unsigned extent;			/* extent number */
	unsigned moreext;			/* more extents flag */
	unsigned long offset;		/* block (256 bytes) past the archive header for file data */
	char headertype[7];			/* header type ("FST") */
	char owner[34];				/* Owner's name (directories only) */
};

int viewonly = 0;
int convert3 = 1;
int convert4 = 1;
int verbose = 0;

char *destdir = NULL;
	
main( int argc, char **argv )
{
	int i, fd;

	int ibuf[256];
	char type[2], *fname, *p;
	struct archive_hdr *ha;
	struct file_hdr *hf;

	int opt;
	extern char *optarg;
	extern int optind;

	while( (opt=getopt(argc, argv, "Vvd:")) != EOF )
	{
		switch(opt) {
		case 'v':
			viewonly = 1;
			break;
		case 'V':
			verbose = 1;
			break;
		case 'd':
			destdir = strdup(optarg);
			break;
		case 'b':
			convert3 = convert4 = 0;
			break;
		}
	}

	if( optind == argc )
	{
		printf("Try fstread FstArchive.fst\n");
		exit(1);
	}
		
	fname = argv[optind];

	if( (fd = open(fname, O_RDONLY|O_BINARY)) < 0 ) {
		perror(fname);
		exit(1);
	}

	ha = GetArchiveHeader(fd);
	
	printf("archive: %s\n", ha->archive);
	printf("checksum: %o\n", ha->chksum);
	printf("version: %s\n", ha->version);
	printf("Comment Size: %lu\n", ha->sz_comment);
	printf("Tape Number: %u\n", ha->tapenum);
	printf("First File: %lu\n", ha->firstfile);
	printf("Start Block: %lu\n", ha->startblock);
	printf("Split: %u\n", ha->split);
	printf("Header Type: %s\n", ha->headertype);
	printf("Created: %24.24s\n\n", ctime(&ha->created));

	// sanity clause

	p = ha->archive;
	while( *p ) {
		if( !isprint(*p) ) {
			printf("ERROR: Corrupt archive - non printable character in archive header\n");
			exit(1);
		}
		p++;
	}
	if( ha->sz_comment != 0 ) {
			printf("ERROR: non-zero comment size\n");
			exit(1);
	}

	if( strncmp(ha->headertype, "FST", 3) ) {
			printf("ERROR: header type != FST\n");
			exit(1);
	}
		
	hf = GetFileHeader(fd);						/* Skip the directory file entry */
	SkipData(fd, hf->blocks);

	if( destdir ) {
		if( chdir(destdir) ) {
			perror(destdir);
			exit(1);
		}
	}
	
	while( hf = GetFileHeader(fd) )
	{
		if(hf->desc[0] == '\000') break;
		if(hf->bytes == 0) break;
		printf("%8ld  ", hf->bytes);
		printf("0%3.3o  ", hf->mode);
		printf("%s\n", hf->desc);

		strcpy(type, "?");
		p = hf->desc;
		for( i = 0 ; i < 3 ; i++ ) {
			if( (p=strchr(p, ':')) == NULL ) break;
			p++;
		}

		if( p ) type[0] = *p;		

		if( verbose ) printf("type: %s  p:%s\n", type, p);
		
		
		if( (hf->blocks * 256) < hf->bytes ) {
			printf("warning! bytes (%d) and blocks (%d * 256 = %d) don't match, using byte value\n", hf->bytes, hf->blocks, hf->blocks * 256);
			hf->blocks = hf->bytes / 256;
		}

		if( (hf->blocks * 256) > hf->bytes ) {
			printf("warning! bytes (%d) and blocks (%d * 256 = %d) don't match, using block value\n", hf->bytes, hf->blocks, hf->blocks * 256);
		}
		
		if( viewonly ) {
			SkipData(fd, hf->blocks);
		} else {
			if( verbose ) printf("restoring %ld bytes\n", hf->blocks * 256);
			if( RestoreData(hf->desc, fd, hf->blocks) ) exit(2);
			if( convert3 && (type[0] == '3') ) UnPack(hf->desc);
			if( convert4 && (type[0] == '4') ) UnPack(hf->desc);
		}
	}
	close(fd);
}


struct archive_hdr *GetArchiveHeader( int fd )
{
	static struct archive_hdr h;
	unsigned char ibuf[512];
	unsigned char tmp[1024];

	pos = lseek(fd, 0L, SEEK_CUR);
	
	if( read(fd, ibuf, 512) < 0 ) {
		perror("ERROR reading archive header");
		exit(1);
	}

	memcpy(h.archive, ibuf, 100);
	h.archive[100] = '\000';
	
	memcpy(tmp, ibuf+148, 8);
	tmp[8] = '\000';
	sscanf(tmp, "%o", &h.chksum);

	memcpy(h.version, ibuf+158, 64);
	h.version[64] = '\000';

	h.sz_comment = MakeInt32( ibuf+298 );
	h.tapenum = MakeInt16(ibuf+380);
	h.firstfile = MakeInt32(ibuf+382);
	h.startblock = MakeInt32(ibuf+386);
	h.split = MakeInt16(ibuf+390);

	memcpy(h.headertype, ibuf+417, 6);
	h.headertype[6] = '\000';

	memcpy(tmp, ibuf+447, 12);
	tmp[12] = '\000';
	sscanf(tmp, "%o", &h.created);
	
	return(&h);
}

struct file_hdr *GetFileHeader( int fd )
{
	static struct file_hdr h;
	unsigned char ibuf[512];
	unsigned char tmp[1024];
	int i;

	memset(ibuf, 0, sizeof(ibuf));
	memset(&h, 0, sizeof(h));
	
	pos = lseek(fd, 0L, SEEK_CUR);
	if( read(fd, ibuf, 512) < 0 ) {
		perror("ERROR reading file header");
		exit(1);
	}

	memcpy(h.desc, ibuf, 100);
	h.desc[100] = '\000';

	for( i = 99 ; i >= 0 ; i-- )			/* trim trailing blanks */
	{
		if( h.desc[i] != ' ' ) break;
		h.desc[i] = '\000';
	}

	
	memcpy(tmp, ibuf+100, 8);
	tmp[8] = '\000';
	sscanf(tmp, "%o", &h.mode);

	memcpy(tmp, ibuf+124, 12);
	tmp[12] = '\000';
	sscanf(tmp, "%o", &h.bytes);

	memcpy(tmp, ibuf+148, 8);
	tmp[8] = '\000';
	sscanf(tmp, "%o", &h.chksum);

	h.lu = MakeInt16(ibuf+294);
	h.sec_trk = MakeInt16(ibuf+296);
	h.blocks = MakeInt32(ibuf+298);
	h.hier = MakeInt16(ibuf+302);
	h.extent = MakeInt16(ibuf+368);
	h.moreext = MakeInt16(ibuf+370);
	h.offset = MakeInt32(ibuf+372);

	memcpy(h.headertype, ibuf+417, 6);
	h.headertype[6] = '\000';

	memcpy(h.owner, ibuf+467, 33);
	h.owner[33] = '\000';

	return(&h);
}




unsigned long MakeInt32( unsigned char *p )
{
	unsigned long tmp = 0;
	
	tmp += ( *p << 24 ); p++;
	tmp += ( *p << 16 ); p++;
	tmp += ( *p << 8 );	p++;
	tmp +=  *p; 	

	return(tmp);
}

unsigned MakeInt16( unsigned char *p )
{
	unsigned long tmp = 0;
	
	tmp += ( *p << 8 );	p++;
	tmp +=  *p; 	

	return(tmp);
}


RestoreData(char *namr, int fd, long blocks )
{
	int ofd = 0;
	char *p, *n;
	char tmp[4096];
	long i;

	/* fixup file name (removing trailing FMP information) */

	if( (p=strchr(namr, ':')) ) *p = '\000';
	p = namr;

#if MSDOS
	while( *p ) {
		if( *p == '*' ) *p = '+';
		if( *p == '?' ) *p = '~';
		if( *p == '"' ) *p = '_';
		if( *p == '>' ) *p = '-';
		if( *p == '<' ) *p = '~';
		p++;
	}
#endif

	while( *namr && (*namr == '/') ) namr++;		/* strip leading "/" if present */
	strcpy(tmp, namr);

	if( (p=strrchr(tmp, '/')) )			/* hierarcical file name */
	{
		*p = '\000';					/* remove file name from end */
		if( MakeDirs(tmp) ) exit(1);	/* make intermediate directories */
	}		

	if( (ofd=open(namr, O_RDWR|O_BINARY|O_CREAT, 0666)) < 0 )
	{
		perror(namr);
		exit(1);
	}

	for( i = 0 ; i < blocks ; i++ )
	{
		if( read(fd, tmp, 256) < 0 ) {
			perror("ERROR: reading datablock");
			exit(1);
		}
		
		if( write(ofd, tmp, 256) < 0 ) {
			perror(namr);
			exit(1);
		}
	}

	close(ofd);

	if( blocks & 1 ) read(fd, tmp, 256);		/* data is padded if not even number of blocks */
	return(0);
}


SkipData(int fd, long blocks )
{
	char tmp[256];
	long i;
	
	for( i = 0 ; i < blocks ; i++ )
	{
		if( read(fd, tmp, 256) < 0 ) {
			perror("ERROR Reading data block");
			exit(1);
		}
	}
	if( blocks & 1 ) read(fd, tmp, 256);		/* data is padded if not even number of blocks */
	return(0);
}

MakeDirs( char *path )
{
	char *p = path;
	extern int errno;
	
	while( *p ) {
		if( *p == '/' ) {
			*p = '\000';
			if( mkdir(path) && (errno != EEXIST) ) {
				perror(path);
				exit(1);
			}
			*p = '/';
		}
		p++;
	}

	if( mkdir(path) && (errno != EEXIST) ) {
		perror(path);
		exit(1);
	}

	return(0);
}

#define RTE_EOF -1

UnPack(char *namr)
{
	char *p;
	int i;
	unsigned char ibuf[32768];
	int ifd, ofd;
	long reclen1, reclen2;
	char ofname[4096];
	char tmp[128];
	
	/* fixup file name (removing trailing FMP information) */

	if( (p=strchr(namr, ':')) ) *p = '\000';
	p = namr;

#if 0	

	while( *p ) {
		if( *p == '*' ) *p = '+';
		if( *p == '?' ) *p = '~';
		if( *p == '"' ) *p = '_';
		if( *p == '>' ) *p = '-';
		if( *p == '<' ) *p = '~';
		p++;
	}

#endif
	
	while( *namr && (*namr == '/') ) namr++;		/* strip leading "/" if present */

	sprintf(ofname, "%s~", namr);
	
	if( (ifd=open(namr, O_RDONLY|O_BINARY)) < 0) {
		perror(namr);
		exit(1);
	}
	
	if( (ofd=open(ofname, O_RDWR|O_CREAT, 0666)) < 0) {
		perror(ofname);
		exit(1);
	}

	while( (reclen1 = GetRecLen(ifd)) != RTE_EOF ) {
		if( verbose ) printf("UnPack: reading record of %d bytes\n", reclen1);
		if( ReadIt(ifd, ibuf, reclen1) ) {
			printf("ERROR: reading %d bytes at %ld in %s\n", reclen1, lseek(ifd, 0L, SEEK_CUR), namr);
			perror(namr);
			close(ifd); close(ofd);
			return(1);
		}
		if( reclen1 & 1 ) read(ifd, tmp, 1); 
		reclen2 = GetRecLen(ifd);
		if( reclen1 != reclen2 ) {
			fprintf(stderr, "Record lengths don't match! reclen1: %ld  reclen2: %ld\n", reclen1, reclen2);
			close(ifd); close(ofd);
			return(2);
		}
		if( write(ofd, ibuf, reclen1) != reclen1 ) BadError("writing");
		write(ofd, "\n", 1);
	}
	close(ifd);
	close(ofd);

	unlink(namr);
	rename(ofname, namr);

}

int ReadIt( int fd, char *p, long len )
{
	if( read(fd, p, len) != len ) return(1);
	return(0);
}

void BadError( char *p )
{
	fprintf(stderr, "Error occured!\n");
	perror(p);
	exit(1);
}

long GetRecLen( int fd )
{
	unsigned char ibuf[2];
	unsigned long tmp;
	
	if( read(fd, ibuf, 2) != 2 ) return(RTE_EOF);	

	if( verbose ) printf("GetRecLen read 0x%2.2x 0x%2.2x\n", ibuf[0], ibuf[1]);

	tmp = MakeInt16(ibuf);
	if( tmp == 0xffff ) return(RTE_EOF);
	if( tmp >= 0x8000 ) return( 2*((tmp & 0x7fff) + 1) - 1);	/* byte length */
	return(2 * tmp);
}


