  	.title 	'DSC/3, DSC/4 PROM Monitor'
	.sbttl	'PROM.ASM'
version	=	1
revision=	9
;----------
; DSC-3, DSC-4 PROM Monitor
;
; Features of the DSC/3:
;
;    Z-80 at 4MHz
;    64K Dynamic RAM
;    1K PROM
;    4 Ports - 3 RS-232
;              1 RS-422  (optionally RS-232)
;    Real time clock
;    Optional floppy disk controller
;    2 8-bit parallel ports
;
; Features of the DSC/4:
;
;    Same as DSC/3, except there is a 1K Dynamic RAM
;    on the CPU board, and 128K per memory board.  
;    There are 16 8-bit memory map registers on the
;    CPU board, which are used to map each 16-bit 
;    logical address into a 19-bit physical address.
;
; The PROM Monitor occupies the 1K PROM on the DSC/3
; and the DSC/4.  It can execute these commands:
;
; Set <addr>
;   set memory locations, starting at <addr>
;
; Dump [ <addr1> [<addr2>] ]
;   dump memory from <addr1> to <addr2>
;
; Fill <addr1> <addr2> <byte>
;   fill memory with <byte> from <addr1> to <addr2>
;
; Go <addr>
;   jump to <addr>
;
; In <port>
;   input from Z-80 <port>
;
; Out <port> <byte>
;   output <byte> to Z-80 <port>
;
; Map <reg> <byte>
;   set memory map register <reg> to <byte>
;
; Test <addr1> <addr2>
;   test memory from <addr1> to <addr2>
;
; Boot <device> [<combuf>]
;   read and execute program from floppy, network, 
;
	.page
	.pabs
	.phex
;----------
;    Data and address definitions:
;
RAM	==	09000h	; beginning of RAM
BOOTloc	==	RAM	; beginning of bootstrap
stack	==	RAM+400h; top of RAM
Daddr	==	RAM+3E0h; Dump address
resbuf	==	RAM+380h; Floppy result buffer
homeF	==	RAM+390h; home command
readF	==	RAM+393h; read command
cr	==	0Dh
lf	==	0Ah
RxRDY	==	0	; receiver ready status bit
TxRDY	==	2	; transmitter ready status bit
;
;    Port definitions:
;
DMA	==	38h	; DMA chip
CTC0	==	30h	; CTC channel 0 (port 0)
SIO1AC	==	2Ah	; SIO-1 channel A, control 
SIO1AD	==	28h	; SIO-1 channel A, data -port 0
SIO1B	=	2Bh	; SIO-1 channel B, control
SIO1BD	==	29h	; SIO-1 channel B, data -port 1
PIOAC	==	0Ah	; PIO channel A, control
PIOAD	==	08h	; PIO channel A, data
PIOBC	==	0Bh	; PIO channel B, control
PIOBD	==	09h	; PIO channel B, data
FLOPSR	==	10h	; Floppy status register
FLOPDR	==	11h	; Floppy data register
SETMAP	==	03h	; Set memory map register
STOPFLOP==	03h	; Stop floppy controller
;----------
; Come here on master reset
	.loc	00h	; RST 0
	JMP	DORELOC



;----------
; Update history:
;
; 1.07 (09/01/80) Initial release
; 1.08  (10/01)   Changed autoboot jumpers
; 1.09  (10/27)   Changed autoboot jumpers (AGAIN!)
;	          Boot from any floppy drive
;		  Boot from hard disk
; 1.09T 06/21/81  Boot from tape (option)
	.page
;----------
; Utility routines
;
; Callable by an RST instruction:
;
; RST 1 - in0:     Input a character from port 0
; RST 2 - out0:    Output a character to port 0
; RST 3 - outst0:  Output a string to port 0
; RST 4 - crlf:    Output <return>, <linefeed> to port0
; RST 5 - space:   Output <space> to port 0
; RST 6 - getbyt:  Input a byte from port 0
; RST 7 - restart: Restart the PROM monitor
;
; Callable through jump vector at top of ROM
;
; command: Output a command string to the floppy contlr
; result:  Input a result string from the floppy contlr
; ioerr:   Output an I/O error message on port 0
;
;JUMP VECTOR TABLE
 NUMVEC	=  7
        .LOC  3FCH-3*NUMVEC
	JMP	command
	jmp	result
	jmp	ioerr

;
; Version, revision, and serial number
	.byte	version
	.byte	revision
	.word	0FFFFh	; serial number
	.page
;----------
; Input a character from port 0
;  Regs in:   none
;  Regs out:  A = input character
	.loc	08h	; RST 1
in0:
waiti:	in	SIO1AC  ; read status
	bit	RxRDY,A ; check for receiver ready
	jrz	waiti	; wait until ready      
	in	SIO1AD	; read the char
;----------
; Output a character to port 0
;  Regs in:   A = output character
;  Regs out:  none
	.loc	10h	; RST 2
out0:
	ani	7Fh	; strip parity bit
	push	PSW	; save output char
	jmpr	outc	; continue elsewhere
;----------
; Flush an input character from port 0
;  (jumped to from 'out')
flush:	in	SIO1AD	; flush the input char
	rst	restart/8 ; restart the monitor
;----------
; Output a string to port 0
;  Regs in:   HL = address of string (ended by null)
;  Regs out:  none
;  Destroyed: A, HL
	.loc	18h	; RST 3
outst0:
	mov	A,M	; get next output char
	ora	A	; string terminated by null
	rz
	rst	out0/8	; output this char
	inx	H	; point to next char
	jmpr	outst0	; repeat until null found
;----------
; Output <return>, <linefeed> to port 0
;  Regs in:   none
;  Regs out:  none
;  Destroyed: A
	.loc	20h	; RST 4
crlf:
	mvi	A,cr
	rst	out0/8
	mvi	A,lf
	jmpr	out0	; return from out0
;----------
; Output <space> to port 0
;  Regs in:   none
;  Regs out:  none
;  Destroyed: A
	.loc	28h	; RST 5
space:
	mvi	A,' '
	jmpr	out0	; return from out0
;----------
; Handle an input error
;  Regs in:   none
;  Regs out:  none
error:	mvi	A,'?'
	rst	out0/8	; type '?' on input error
	rst	restart/8 ; restart the monitor
;--------
; Input a byte from port 0
;  Regs in:   none
;  Regs out:  A = byte read
;	      B = 1 if number was typed, 0 otherwise
;  Destroyed: DE
	.loc	30h	; RST 6
getbyt:
	push	H	; save HL (courtesy to caller)
	call	getadr	; let 'getadr' do the work
	mov	A,L	; result in low byte of HL
	pop	H	; restore HL
	ret
;----------
; Restart the main monitor loop
	.loc	38h	; RST 7
restart:
	jmp	prompt	; jump here to restart monitor
;----------
; Output a byte to port 0
;  (jumped to from 'out0')
outc:
wait0:	in	SIO1AC	; read status
	bit	TxRDY,A	; check for transmitter ready
	jrz	wait0	; wait until ready
	bit	RxRDY,A	; check for receiver ready
	jrnz	flush	; jump if interrupted by user
	pop	PSW	; restore output char
	out	SIO1AD	; write the char
;
; Convert lowercase to uppercase
	cpi	'a'	; return if less than 'a'
	rm
	cpi	'z'+1	; return if greater than 'z'
	rp
	sui	'a'-'A'	; convert lower to upper case
	ret
;----------
; Input an address from port 0
;  Regs in:   none
;  Regs out:  HL = address read
;	      B  = 1 if number was typed, 0 otherwise
;  Destroyed: A, DE
;
; Note:  An address contains zero or more spaces, 
;	 followed by one or more HEX digits, and
;	 terminated by a <space> or <return>
getadr:
	lxi	H,0	; HL = 0
	mov	B,H	; B = 0
	mov	D,H	; D = 0
get0:	rst	in0/8	; skip leading spaces
	cpi	' '
	jrz	get0
get1:	cpi	cr	; terminate if <return>
	rz
	cpi	'G'
	jrnc	error	; jump if digit > F
	cpi	'A'
	jrc	num	; jump if digit < A
	sui	'A'-('9'+1) ; partially convert A..F
	cpi	'9'+1	
	jrc	error	; jump if 9 < digit < A
num:	sui	'0'	; convert ASCII to HEX
	jrc	error	; jump if digit < 0
	dad	H	; shift HL left 4 bits
	dad	H
	dad	H
	dad	H
	mov	E,A	; DE = current HEX digit
	dad	D	; add current digit to addr
	mvi	B,1	; set 'number typed' flag
	rst	in0/8
	cpi	' '	; terminate if <space>
	rz
	jmpr	get1	; continue collecting chars
;----------
; Output an address to port 0
;  Regs in:   HL = address to be printed
;  Regs out:  none
;  Destroyed: A
prtadr:
	mov	A,H
	call	prtbyt
	mov	A,L
;----------
; Output a byte to port 0
;  Regs in:   A = byte to be printed
;  Regs out:  none
;  Destroyed: A
prtbyt:
	push	PSW
	rrc		; print high nibble of A
	rrc
	rrc
	rrc
	call	prtnbl	; print low nibble of A
	pop	PSW
prtnbl:	ani	0Fh	; mask lower nibble
	adi	'0'	; convert to ASCII
	cpi	'9'+1
	jc	out0	; jump if 0..9
	adi	'A'-('9'+1) ; convert A..F to ASCII
	jmp	out0	; use return from out0
;----------
; Send a command string to the floppy controller
;  Regs in:   HL = address of command string
;  Regs out:  none
;  Destroyed: A, HL
command:
	mov	A,M	; get a command byte
	cpi	endcom	; check if end-of-string
	rz		; return if all bytes sent
	call	waitdr	; wait for data register
	cc	ioerr	; I/O error if wrong direction
	mov	A,M	; get the command byte
	out	FLOPDR	; write to data register
	inx	H	; point to next command byte
	jmpr	command
;----------
; Collect result string from the floppy controller
;  Regs in:   none
;  Regs out:  none
;  Destroyed: A, HL
result:
	lxi	H,resbuf; point to result buffer
readDR:	call	waitdr	; wait from data register
	rnc		; return if all bytes received
	in	FLOPDR	; get the result byte
	mov	M,A	; store it in 'flopbuf'
	inx	H	; point to next result byte
	jmpr	readDR
;----------
; Wait until floppy data register is ready
;  Regs in:   none
;  Regs out:  A = status register, rotated left by 1
waitdr:
	in	FLOPSR	; read floppy status register
	rlc
	jrnc	waitdr	; bit 7 = 1 if ready
	rlc		; bit 6 = 1 if CPU receiving
	ret		; 	  0 if CPU sending
;----------
; Output an I/O error message
;  Regs in:   none
;  Regs out:  none
errmsg:	.asciz	[cr][lf]'I/O error '
ioerr:	lxi	H,errmsg
	rst	outst0/8 ; output a message
	pop	H	; get the caller's address
	call	prtadr	; print the caller's address
	rst	restart/8 ; restart the monitor
	.page
;----------
; Initialization code - executed on master reset
;
; Initialize the PIO chip
initial:
	mvi	A,0CFh	
	out	PIOAC	; reset PIO channel A
	mvi	A,0F8h	;  top    5 lines = input
	out	PIOAC	;  bottom 3 lines = output
	mvi	A,3
	out	PIOAC	; disable interrupts
	mvi	A,0CFh
	out	PIOBC	; reset PIO channel B
	mvi	A,0C0h	;  top    2 lines = input
	out	PIOBC	;  bottom 6 lines = output
	mvi	A,3
	out	PIOBC	; disable interrupts
;
; Do a little house cleaning
	mvi	A,0
	out	PIOBD	; unload the floppy head
	mvi	A,0C3h
	out	DMA	; reset the DMA chip
;
; Initialize port 0 to be a standard CRT at 9600 baud
	mvi	A,45h	; set counter mode
	out	CTC0	; next byte is time constant
	mvi	A,13	; 13 = 2Mhz/(16*baudrate)
	out	CTC0	; port 0 is 9600 baud
	lxi	H,rs232	; port 0 is RS-232
	lxi	B,rs232$<8+SIO1AC 
	outir
;
; Enter monitor if front-panel switch is on
	in	PIOAD
	bit	7,A	; check front-panel switch
	jrnz	Monitor
;
; Check for auto-boot
	in	SIO1AC	;  DCD  STS
	bit	5,A	;   0    0   Floppy boot
	jmp	DPRGM
;
; RS-232 SIO command - for complete explanation,
; 		       see Zilog Data Book, page 56
rs232:  .byte	18h	      ; channel reset
	.byte	14h,01001100b ; 2 stops bits, no parity
	.byte	 3 ,11000001b ; receiver enable
	.byte	 5 ,11101010b ; transmitter enable
	.byte	11h,00000000b ; no interrupts
rs232$	==	.-rs232	  ; length of SIO command string
;----------
; Main monitor loop
;
; Wait for a command character, then jump to the
; appropriate routine to handle the command
;
PROMmsg:.ascii  [cr][lf]'PROM Monitor '
	.byte	version+'0','.'
	.byte	revision/10+'0',revision@10+'0',0
Monitor:
	lxi	H,PROMmsg
	rst	outst0/8 ; print a boot-up message
prompt:
	lxi	SP,stack ; start with a fresh stack
	rst	crlf/8	; on a fresh line
	mvi	A,':'	; output the command prompt
	rst	out0/8
	rst	in0/8	; input a command
	cpi	'S'
	jrz	Set	; Set memory
	cpi	'D'
	jz	Dump	; Dump memory
	cpi	'G'
	jz	Go	; Go to a program
	cpi	'F'
	jz	Fill	; Fill memory
	cpi	'I'
	jz	In	; Input from a port
	cpi	'O'
	jz	Out	; Output to a port
	cpi	'M'
	jz	Map	; set Memory map register
	cpi	'T'
	jz	Test	; Test memory
	cpi	'B'
	jz	Boot	; Boot
	cpi	cr
	jrz	Dumpcr	; <cr> - dump memory
	jmp	error
	.page
;----------
; Set memory
;
; S <addr>
Set:
	call	getadr	; address is in HL
Set1:	rst	crlf/8
	call	prtadr	; print HL
	rst	space/8
	mov	A,M	; get old byte from memory
	call	prtbyt	; print it
	rst	space/8
	rst	getbyt/8 ; get new byte from CRT
	bit	0,B	; test if number entered
	jrz	Set2    ; jump if no number entered
	mov	M,A	; replace byte in memory
Set2:	inx	H	; point to next byte
	jmpr	Set1
;----------
; Dump memory
;
; D  [ <addr1> [<addr2] ]
Dumpcr:
	lxi	H,RAM	; default starting address
	shld	Daddr
	jmpr	default
Dump:
	call	getadr	; get starting address
	bit	0,B
	jrnz	Daddr1	; jump if number entered
	lhld	Daddr	; resume from previous dump
	jmpr	default
Daddr1: shld	Daddr	; store starting address
	cpi	cr	; see if <return> was typed
	jrz	default	; jump if no end addr entered
	call 	getadr	; get ending address
	jmpr	Daddr2
default:lxi	D,128	; default dump range
	dad	D
Daddr2:	xchg		; DE = ending address
;
; Top of dump loop - print dump address
Dump0:	lhld	Daddr
	rst	crlf/8
	call	prtadr
;
; Dump 16 bytes in hexadecimal format
Dump1:	rst	space/8
	mov	A,M	; get a byte
	call	prtbyt	; print it in hexadecimal
	inx	H	; point to next byte
	mov	A,L	; check if multiple of 16
	ani	0Fh
	jrnz	Dump1	; jump if not
;
; Dump 16 bytes in ASCII format
	rst	space/8
	lhld	Daddr
Dump2:	mov	A,M	; get a byte
	ani	7Fh	; mask out parity bit
	cpi	20h	; printable?
	jrnc	.+4	; skip next instr if printable
	mvi	A,'.'	; print <dot> if unprintable
	rst	out0/8	; print it in ASCII
	inx	H	; point to next byte
	mov	A,L	; check if multiple of 16
	ani	0Fh
	jrnz	Dump2	; jump if not
;
; Check if dump is finished
	shld	Daddr
	ora	A	; clear carry flag
	dsbc	D	; clear carry flag if done
	jrc	Dump0	; jump if not done
	rst	restart/8
;----------
; Go to a memory location
; 
; G <addr>
Go:
	call	getadr	; address is in HL
	pchl		; jump to address in HL
;----------
; Fill memory
;
; F <addr1> <addr2> <byte>
Fill:
	call	getadr	; starting address
	push	H
	call	getadr	; ending address
	rst	getbyt/8 ; fill char
	pop	D	; begin addr in DE
	ora	A	; clear carry
	dsbc	D	; HL = HL - DE
	mov	C,L	; BC = byte count
	mov	B,H	
	mov	L,E	; HL = source address
	mov	H,D
	mov	M,A	; put fill char in memory
	inx	D	; DE = destination address
	ldir
	rst	restart/8
;----------
; Input from a port
;
; I <port>
In:
	rst	getbyt/8
	mov	C,A	; C = port number
	rst	crlf/8
	inp	A	; input byte to A
	call	prtbyt	; print the byte
	rst	restart/8
;----------
; Output to a port
;
; O <port> <byte>
Out:
	rst	getbyt/8
	push	PSW	; save port number
	rst	getbyt/8 ; get output byte
	pop	B	; restore port number
	mov	C,B	; C = port number
	outp	A	; output byte from A
	rst	restart/8
;----------
; set memory Map register
;
; M <reg> <byte>
;
; Note:  <reg> = 0,1,..F
;	 <byte>: bit 7 = 0  local memory
;		 bit 7 = 1  external memory
;		 bits 6 to 0 = page number 
;		  (bits 6 to 4 are low-true)
;
; For example, the command "M 2 F3" will map logical
; addresses 2000-2FFF to physical addresses 3000-3FFF.
;
; To enable the memory map, run the command "O 0 8'.
Map:
	rst	getbyt/8
	push	PSW	; save register number
	rst	getbyt/8 ; get mm reg contents
	mov	H,A	; save contents of mm reg
	pop	PSW	; restore register number
	rlc		; put reg # in high nibble of B
	rlc
	rlc
	rlc
	mov	B,A
	mvi	C,SETMAP; C = port number
	outp	H	; load memory map register
	rst	restart/8
	.page
;----------
; Test memory
;
; T <addr1> <addr2>
;
; Repetitively write an increasing data pattern,
; and read it back for verification. 
;
; For each error detected , the address, byte written,
; and byte read are displayed.
pattern	=	0FFh	; initial pattern offset
Test:
	call	getadr	; read starting address
	push	H
	call	getadr	; read ending address
	pop	D	; DE = start, HL = end
	mvi	B,pattern
	rst	crlf/8
retry:	push	H	; save ending address
	push	H
fillmem:mov	A,L
	add	H
	add	B
	mov	M,A	; fill memory with H+L+pattern
	ora	A	; clear carry bit
	dsbc	D	; set Z flag if done
	dad	D	; restore HL
	dcx	H	; go to next location
	jrnz	fillmem
	pop	H	; restore ending address
testmem:mov	A,L
	add	H
	add	B
	cmp	M
	cnz	memerr	; call if memory error
	ora	A
	dsbc	D
	dad	D
	dcx	H
	jrnz	testmem
	dcr	B	; use another pattern
	mvi	A,'.'	; tell user we're still alive
	rst	out0/8
	pop	H
	jmp	retry
memerr:	push	PSW	; save what was written
	rst	crlf/8
	call	prtadr	; print address of error
	rst	space/8
	pop	PSW
	call	prtbyt	; byte written
	rst	space/8
	mov	A,M
	jmp	prtbyt	; byte read
	.page
;----------
; Boot from floppy, network, harddisk, or tape.
;
; B <device> [<combuf>]
;
; Read a bootstrap program from a device into RAM.
; If successful, jump to RAM.  If the device is 'T',
; then test the floppy disk controller.
Boot:
	rst	in0/8	; get device name
	cpi	'F'
	jrz	Floppy	; Boot from floppy unit 0
	cpi	'T'
	jrz	Testflop; Test floppy
;
; Check for boot from specified floppy unit
	sui	'0'	; unit must be >= 0
	jm	error
	cpi	7+1	; unit must be <= 7
	jp	error
	jmpr	goflop
;----------
; Floppy boot
;
; Read single density, track 0, sectors 1-2
; from drive 0, and jump to RAM
Floppy:
	sub	A
goflop:
	lxi	H,homeFLOP; setup home, read commands
	lxi	D,homeF	; must move to RAM so that
	lxi	B,13	; unitno can be selected
	ldir
	sta	homeF+1	; put unitno in home, read
	sta	readF+1
	set	2,A	; turn on drive select bit
	out	PIOBD	; select drive
	sub	A
	out	PIOAD	; multiplex Floppy to DMA
	call	result	; flush the floppy controller
	lxi	H,specFLOP
	call	command ; specify timing info
reseek: lxi	H,homeF
	call	command	; recalibrate drive 0
sense:	lxi	H,IsenseFLOP
	call	command	; sense interrupt status
	call	result	; 2 result bytes returned
	lda	resbuf	; get first status byte
	bit	3,A	
	jrnz	reseek	; retry if not ready
	bit	5,A
	jrz	sense	; retry until seek done
	ani	0D0h	; check if seek successful
	cnz	ioerr
;
; This code is very timing sensitive.
; For single density, each data byte must be read from
; the floppy controller within 27 microsecs, or else it
; will be lost, and the read operation will terminate
; prematurely.
	lxi	H,readF
	call	command	; read data
	lxi	H,BOOTloc ; boot address
       	lxi	B,256<8+FLOPDR
..read:	call	waitdr	; wait for next byte available
	ini		; get and store a byte
	jrnz	..read	; continue until 1 sector read
	in	STOPFLOP; stop the floppy controller
;
; Check result status
	call	result	; 7 result bytes returned
	lda	resbuf	; get first result byte
	ani	0C0h	; top 2 bits should be zero
	cnz	error	
	jmp	BOOTloc	; jump to boot code
;----------
; Test floppy
;
; Get command buffer address, and then
; perform a floppy command and result phase
Testflop:
	call	getadr	; get command buffer address
	call	command ; do a command phase
	call	result	; do a result phase
	rst	restart/8
 	.page
;----------
; Floppy command strings - see NEC765 chip description
;			   for explanation
drive	=	0	; boot drive number
track	=	0	; boot track number
sector	=	1	; first boot sector
endcom	=	0FFh	; end-of-command character
specFLOP:
	.byte	3	; 'specify' command
	.byte	88h	; seek rate = 8 ms
			; head unload time = 128ms
	.byte	81h	; head load time = 128ms
			; non-DMA mode
	.byte	endcom
IsenseFLOP:
	.byte	8	; 'sense interrupt status'
	.byte	endcom
homeFLOP:
	.byte	7	; 'recalibrate' command
	.byte	drive	; drive select
	.byte	endcom
readFLOP:
	.byte	6	; 'read' command
	.byte	drive	; head, drive select
	.byte	track	; track number
	.byte	00h	; head address
	.byte	sector	; sector number
	.byte	00h	; 128 byte sector
	.byte	1Ah	; last sector number
	.byte	07h	; gap length (in bytes)
	.byte	128	; sector size
	.byte	endcom
	.page

















;		    DMA SHORT PROM TEST


;DP2PM WITH EXTRA DMA RESETS  AT BEG OF  DMA XFER CMD AND AT
;AND END  OF READ 
;
;THIS PROM TRANSFERS DATA, THEN RETURNS TO PROM.
;USER MUST USE PROM `S`CMD TO SET BYTE COUNT,
;BEG SOURCE ADR AND BEG DESTINATION ADR 
;LOCATIONS IN MEMORY.(SEE EQUATES FOR 
;EXACT LOCATIONS
;
;-----------------------------------------
;                    DMA TRANSFER


PBYTSA	==	17H	;BYTES TO PROGRAM DMA CTLR
PBYTSB	==	04H	;PRGRM BYTES TO READ REGS
DMA	==	38H	;DMA PORT ADR
MEM	==	8000H	;BEG PARAM LOC
SADR	==	MEM	;BEG SOURCE ADR
DADR	==	9000H   ;BEG DESTINATION ADR
BYCNT	==	000FH	;BYTE COUNT

DSTAT	==	0A080H	;DMA STATUS REG LOC
BYCNTL  ==	0A081H	;DMA LOWER BYTE COUNT
BYCNTH	==	0A082H	;DMA UPPER BYTE COUNT 
DSADRL	==	0A083H	;LOW BYTE DMA SOUCE ADR
DSADRH	==	0A084H	;HIGH BYTE DMA SOURCE ADR
DDADRL	==	0A085H	;LOW BYTE  DEST END
DDADRH	==	0A086H	;HIGH BYTE DEST END

DPRGM:	LXI	H,DCMD	;LOAD CODE TO PROGRAM DMA
	LXI	B,PBYTSA;BYTES TO TRANSFER
	MVI	C,DMA	;SET DMA PORT ADR
	OUTIR
	CALL	DELAY
	RST	7	;RETURN TO PROM


;------------------------------------
;	READ DMA REGISTERS	


RDSTAT: LXI	H,RDCMD	;LOAD READ CMD
	LXI	B,PBYTSB
	MVI	C,DMA
	OUTIR		;OUTPUT CMD TO DMA
	CALL	DELAY	;DELAY TO SYNC DMA AND CPU
	MVI	D,07H	;SET BYTES COUNT	
	LXI	H,DSTAT ;LOAD DMA REG STORAGE LOC	
SVDMA:	IN	DMA	;INPUT DMA REGISTER
	MOV	M,A	;STORE DMA REGISTER	
	INX	H	
	DCR	D	;DECREASE BYTE COUNT
	JRNZ	SVDMA
        MVI     A,0C3H  ;RESET DAM
        OUT     DMA
	RST	7	;RETURN TO PROM	

DELAY:	LXI	H,0FFFFH ;WAIT LOOP
DLY:	DCX	H
	MOV	A,H
	ORA	L
	JRNZ	DLY
        RET

;------------------------------
;	COMMAND BYTES

;DMA TRANSFER COMMAND
	
DCMD:	.BYTE	0C3H	;MASTER RESET   2D
        .BYTE   0C3H
        .BYTE   0C3H
        .BYTE   0C3H
        .BYTE   0C3H
        .BYTE   0C3H      
     	.BYTE	0C7H	;RESET SRCE  PORT (2D)
	.BYTE	0CBH	;RESET DEST PORT (2D)
	.BYTE   0B3H	;INTERAL RDY (2D)
	.BYTE	079H	;LOAD DEST PARM (1A)
	.WORD	DADR	;BEG DEST ADR(1A)
	.WORD	BYCNT	;BYTE COUNT(1A)
	.BYTE	0CFH	;LOAD ADR (2D)
	.BYTE	014H	;PORT A INCR MEM (1B)
	.BYTE	010H	;PORT B INCR (1B)
	.BYTE	0CDH	;BURST MODE/NO INT(2B)
	.WORD	SADR	;SET BEG SOURCE (1A)			
	.BYTE	092H	;STOP E.O.BLK(2C)
	.BYTE	0CFH	;LOAD BEG ADR(2D)
	.BYTE	087H	;ENABLE DMA(2D)

;READ DMA INTERNAL REGS CMD


RDCMD:	.BYTE	0BBH	;READ MASK(2D)
	.BYTE	07FH	;READ 7 REGS(2D)
	.BYTE	0A7H	;INIT READ(2D)
;---------------------------------------
;THIS CODE DOES MOVES THE PROM MONITOR
;PROGRAM TO RAM AT LOC 0H. 
;(A)LOADS CODE TO F000H, THEN EXECUTES IT. 
;(B)THE CODE AT F000H 
;    (1)MOVES THE ROM CODE TO RAM STARTING 
;       AT E000H.
;    (2)IT THE TURNS OFF THE PROM 
;    (3)THEN  MOVES THE MONITOR CODE 
;       TO RAM STARTING AT 0000H.
;
;THE USER ACCESSES THE PROM MONITOR CODE AS IF IT WERE IN ROM
;THE ADVANTAGE IS THAT THE LOW MEM LOCATIONS CAN BE MODIFIED.
;
DORELOC:LXI	H,RELOCDE
	LXI	D,0F000H
	LXI	B,LENRELC
	LDIR
	JMP	0F000H
	
RELOCDE:LXI	H,0000H	;LOAD ROM CODE
	LXI	D,0E000H;LOAD DEST OF CODE 
	LXI	B,400H	;BYTE LENGTH
	LDIR
	IN	02	;TURN OFF PROM
	LXI	H,0E000H;LOAD PROMCODE
	LXI	D,0000H ;LOAD DEST LOC
	LXI	B,400H
	LDIR            ;MOVE CODE 
	JMP	BEGMON	;JMP TO RAM MONITOR CODE
LENRELC=. - RELOCDE

BEGMON:lxi	SP,stack; you always need a stack
	jmp	initial	; intialize the PIO and SIO1

	
;----------
	.end
