/* 
 * Mach Operating System
 * Copyright (c) 1988, 1989, 1990 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */

/*
 *	Apple Macintosh II Mach (macmach)
 *
 *	File: mac2emul/macintr.c
 *	Author: David E. Bohman II (CMU macmach)
 */

/*
 * RCS documentation
 *
 * $Header$
 *
 * $Log$
 */

#include <sys/param.h>
#include <sys/proc.h>
#include <sys/user.h>

#include <mac2/pcb.h>
#include <mac2/psl.h>

#include <mac2emul/macdefs.h>
#include <mac2emul/macuser.h>

/*
 * mac software interrupt handling
 */

check_mac_interrupts()
{
    register struct pcb *pcb = current_thread_pcb();
    register struct mac_internal *mac = pcb->pcb_emul;
    register struct softintr *si;
    register ipl, s;

    if (mac == 0)
	return;

    s = splhigh();

    ipl = ((struct status_reg *)&mac->sr)->sr_ipl;
    
    for (si = mac->softintr;
	 si < &mac->softintr[sizeof (mac->softintr)/sizeof (struct softintr)];
	 si++)
	if (si->ipl > ipl &&
	    ((si->flags&(MAC_IF_PEND|MAC_IF_ENB)) ==
	     (MAC_IF_PEND|MAC_IF_ENB))) {
	    pcb->pcb_ast |= AST_SCHED;
	    break;
	}

    splx(s);
}

perform_mac_interrupts()
{
    register struct pcb *pcb = current_thread_pcb();
    register struct normal_exception_frame *frame = pcb->pcb_frame;
    register struct mac_internal *mac = pcb->pcb_emul;
    register struct softintr *si;
    register ipl, s;

    if (frame->f_fmt != STKFMT_NORMAL && frame->f_fmt != STKFMT_SPECIAL) {
	check_mac_interrupts();
	return;
    }

    s = splhigh();

    ipl = ((struct status_reg *)&mac->sr)->sr_ipl;
    
    for (si = mac->softintr;
	 si < &mac->softintr[sizeof (mac->softintr)/sizeof (struct softintr)];
	 si++)
	if (si->ipl > ipl &&
	    ((si->flags&(MAC_IF_PEND|MAC_IF_ENB)) ==
	     (MAC_IF_PEND|MAC_IF_ENB)) && take_mac_interrupt(si))
	    break;

    splx(s);
}

take_mac_interrupt(si)
struct softintr *si;
{
    register struct pcb *pcb = current_thread_pcb();
    register struct normal_exception_frame *frame = pcb->pcb_frame;
    register struct regs *regs = &pcb->pcb_user;
    register struct mac_internal *mac = pcb->pcb_emul;
    register unsigned long sp;
    register frame_size;

    /*
     * determine format and size of exception
     * frame.
     */
    switch (frame->f_fmt) {
      case STKFMT_NORMAL:
	frame_size = sizeof (struct normal_exception_frame);
	break;

      case STKFMT_SPECIAL:
	frame_size = sizeof (struct special_exception_frame);
	break;

      default:
	return (0);
    }

    /*
     * set system byte of frame status reg
     * to software copy.
     */
    ((struct status_reg *)&frame->f_sr)->sr_sys =
	((struct status_reg *)&mac->sr)->sr_sys;

    /*
     * if tracing, clear trace mode for
     * exception handler.
     */
    mac->sr &= ~(SR_T1|SR_T0);

    /*
     * clear interrupt pending
     * flag.
     */
    si->flags &= ~MAC_IF_PEND;

    /*
     * save exception frame on stack.
     */
    sp = regs->r_sp - frame_size;
    if (copyout(frame, sp, frame_size)) {
	frame->f_sr &= ~SR_USERCLR;
	pcb->pcb_ast &= ~TRACE_USER;
	si->flags |= MAC_IF_PEND;
	return (0);
    }
    regs->r_sp = sp;

    frame->f_sr &= ~SR_USERCLR;
    pcb->pcb_ast &= ~TRACE_USER;

    /*
     * set the ipl in the software
     * status register to the ipl of
     * this interrrupt.
     */
    ((struct status_reg *)&mac->sr)->sr_ipl = si->ipl;

    /*
     * since we know that we have a normal or special
     * return frame, only need to set the
     * return pc.
     */
    if (si->vector >= &((unsigned long *)0)[256])
	FETCH(si->vector, &frame->f_pc, long);
	
    else
	FETCH(mac->ctrl[R_VBR]+si->vector, &frame->f_pc, long);

    return (1);
}
