/* 
 * Mach Operating System
 * Copyright (c) 1988, 1989, 1990 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */

/*
 *	Apple Macintosh II Mach (macmach)
 *
 *	File: mac2emul/macinit.c
 *	Author: David E. Bohman II (CMU macmach)
 */

/*
 * RCS documentation
 *
 * $Header$
 *
 * $Log$
 */

#include <sys/types.h>

#include <kern/thread.h>
#include <kern/zalloc.h>

#include <mach/vm_param.h>

#include <mac2/pcb.h>
#include <mac2/clock.h>
#include <mac2/act.h>

#include <mac2dev/via.h>

#include <mac2slotmgr/Types.h>
#include <mac2slotmgr/Slots.h>

#include <mac2slotmgr/slotmgr.h>

#include <mac2emul/macdefs.h>
#include <mac2emul/macvia.h>

int	mac_emul = (MAC_EMUL_BUS_ERR|MAC_EMUL_PRIV_INST|MAC_EMUL_LINE_1010);

zone_t	maczone;

#define INTR_CLOCK_LIST		0
#define	INTR_TIMER1_LIST	1
#define INTR_TIMER2_LIST	2
#define INTR_SOUND_LIST		3
#define INTR_KB_LIST		4
#define INTR_INTRSW_LIST	5

struct actlist	acttimer1, acttimer2, actsound, actintrsw;
extern struct actlist	actkb;

mac_intr(pcb, list, ap)
struct pcb *pcb;
register int list;
struct act *ap;
{
    register struct mac_internal *mac = pcb->pcb_emul;
    register struct softintr *intr;

    switch (list) {
      case INTR_INTRSW_LIST:
	intr = &mac->softintr[MAC_INTR_INTR];
	if (intr->flags&MAC_IF_ENB) {
	    intr->flags |= MAC_IF_PEND;
	    pcb->pcb_ast |= AST_SCHED;
	}
	break;

      case INTR_CLOCK_LIST:
	intr = &mac->softintr[MAC_INTR_CLK];
	if (intr->flags&MAC_IF_ENB && mac->via1.IER&VIA_IE_CA1) {
	    intr->flags |= MAC_IF_PEND;
	    pcb->pcb_ast |= AST_SCHED;
	}
	break;

      case INTR_KB_LIST:
	intr = &mac->softintr[MAC_INTR_INP];
	if (intr->flags&MAC_IF_ENB) {
	    intr->flags |= MAC_IF_PEND;
	    pcb->pcb_ast |= AST_SCHED;
	}
	break;

      case INTR_SOUND_LIST:
	intr = &mac->softintr[MAC_INTR_ASC];
	if (intr->flags&MAC_IF_ENB) {
	    mac->via2.IFR |= VIA_IF_CB1;
	    if (mac->via2.IER&VIA_IE_CB1) {
		intr->flags |= MAC_IF_PEND;
		pcb->pcb_ast |= AST_SCHED;
	    }
	}
	break;

      case INTR_TIMER1_LIST:
	intr = &mac->softintr[MAC_INTR_TIMER1];
	if (intr->flags&MAC_IF_ENB) {
	    mac->via1.IFR |= VIA_IF_TIMER1;
	    if (mac->via1.IER&VIA_IE_TIMER1) {
		intr->flags |= MAC_IF_PEND;
		pcb->pcb_ast |= AST_SCHED;
	    }
	}
	
      case INTR_TIMER2_LIST:
	intr = &mac->softintr[MAC_INTR_TIMER2];
	if (intr->flags&MAC_IF_ENB) {
	    mac->via1.IFR |= VIA_IF_TIMER2;
	    if (mac->via1.IER&VIA_IE_TIMER2) {
		intr->flags |= MAC_IF_PEND;
		pcb->pcb_ast |= AST_SCHED;
	    }
	}
	break;
    }

    runact(list, ap, pcb, 0);
}

mactimer1_intr()
{
    DOACTLIST(acttimer1);

    VIA1_ADDR->ier = VIA_IE_TIMER1;
}

mactimer2_intr()
{
    unsigned char x;

    DOACTLIST(acttimer2);

    x = VIA1_ADDR->t2cl;
}

macsound_intr()
{
    DOACTLIST(actsound);

    VIA2_ADDR->ier = VIA_IE_CB1;
}

macintrsw_intr()
{
    DOACTLIST(actintrsw);
}

int mac_sched_pri_incr = 5;

mac_begin(pcb)
register struct pcb *pcb;
{
    register struct mac_internal *mac;

    if (maczone == ZONE_NULL) {
	maczone = zinit(sizeof (struct mac_internal),
			sizeof (struct mac_internal),
			MAC2_PGBYTES, FALSE, "mac_internal");

	if (maczone == ZONE_NULL)
	    panic("mac_begin: can't create mac zone");
    }
    mac = (struct mac_internal *)zalloc(maczone);
    if (mac) {
	current_thread()->priority -= mac_sched_pri_incr;	/* XXX */
	bzero(mac, sizeof (*mac));
	mac->sr = INIT_R_SR;
	CTRL_INIT(R_SFC);
	CTRL_INIT(R_DFC);
	CTRL_INIT(R_CACR);
	CTRL_INIT(R_USP);
	CTRL_INIT(R_VBR);
	CTRL_INIT(R_CAAR);
	CTRL_INIT(R_MSP);
	CTRL_INIT(R_ISP);
	mac->via1.portB.dat = 0x8;	/* XXX */
	mac->via2.portA.dat = 0x7f;	/* XXX */
	mac->intr = makeact(mac_intr, SR_IPL7, 6);
	if (mac->intr) {
	    addact(INTR_CLOCK_LIST, mac->intr, &actclock);
	    runact(INTR_CLOCK_LIST, mac->intr, pcb, 0);
	    addact(INTR_KB_LIST, mac->intr, &actkb);
	    runact(INTR_KB_LIST, mac->intr, pcb, 0);
	    addact(INTR_TIMER1_LIST, mac->intr, &acttimer1);
	    runact(INTR_TIMER1_LIST, mac->intr, pcb, 0);
	    addact(INTR_TIMER2_LIST, mac->intr, &acttimer2);
	    runact(INTR_TIMER2_LIST, mac->intr, pcb, 0);
	    addact(INTR_SOUND_LIST, mac->intr, &actsound);
	    runact(INTR_SOUND_LIST, mac->intr, pcb, 0);
	    addact(INTR_INTRSW_LIST, mac->intr, &actintrsw);
	    runact(INTR_INTRSW_LIST, mac->intr, pcb, 0);
	}
	else {
	    zfree(maczone, mac);
	    return;
	}
    }
    else
	return;

    pcb->pcb_emul = mac;
    pcb->pcb_flags |= MAC_EMULATION;
}

mac_end(pcb)
register struct pcb *pcb;
{
    register struct mac_internal *mac = pcb->pcb_emul;

    if (mac->intr)
	endact(mac->intr);
    macscsi_end(&mac->scsi);
    pcb->pcb_emul = 0;
    zfree(maczone, mac);
}

mac_reset(pcb)
register struct pcb *pcb;
{
    register struct mac_internal *mac = pcb->pcb_emul;
    register char sint;

    mac->softintr[MAC_INTR_CLK].flags = 0;
    mac->softintr[MAC_INTR_INP].flags = 0;
    mac->adb_trans_state = 0;
    bzero(&mac->via1, sizeof (struct via_device));
    mac->via1.portB.dat = 0x8;
    sint = mac->via2.portA.dat;
    bzero(&mac->via2, sizeof (struct via_device));
    mac->via2.portA.dat = sint;
}

mac_slot_enb(pcb, slot)
register struct pcb *pcb;
register slot;
{
    register struct mac_internal *mac = pcb->pcb_emul;

    if (slot_in_range(slot))
	mac->via2.portA.dat &= ~(1 << slot_to_index(slot));
}
mac_set_intr_vect(pcb, addr)
register struct pcb *pcb;
register addr;
{
    register struct mac_internal *mac = pcb->pcb_emul;
    register s = splhigh();

    copyin(addr, mac->softintr, sizeof (mac->softintr));

    splx(s);
}

mac_set_intr_pend(pcb, intr)
register struct pcb *pcb;
register intr;
{
    register struct mac_internal *mac = pcb->pcb_emul;
    register s = splhigh();

    if (intr >= 0 && intr < sizeof (mac->softintr)/sizeof (struct softintr))
	mac->softintr[intr].flags |= MAC_IF_PEND;

    splx(s);
}

mac_set_env_ptr(pcb, addr)
register struct pcb *pcb;
{
    register struct mac_internal *mac = pcb->pcb_emul;

    mac->envp = addr;
}

mac_get_env_ptr(pcb)
register struct pcb *pcb;
{
    register struct mac_internal *mac = pcb->pcb_emul;

    return (mac->envp);
}

