/* 
 * Mach Operating System
 * Copyright (c) 1989 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
/*
 * HISTORY
 * $Log:	afs_gateway.c,v $
 * Revision 2.7  89/06/12  14:50:27  jsb
 * 	Turned lingering on again, since it does indeed seem to work.
 * 	[89/06/12  10:32:48  jsb]
 * 
 * Revision 2.6  89/06/04  09:56:06  jsb
 * 	Turned afs_LingeringOK off (sigh).
 * 	[89/06/04  09:25:30  jsb]
 * 
 * 	Total rewrite. We no longer map <hostaddr,pag> pairs into pags, but
 * 	instead have added hostaddr fields to the two or three afs structures
 * 	that use uids. Vnode lingering code has been rewritten. The mode bit
 * 	mutilating routine has been changed so that it can be called from
 * 	afs_getattr in afs/afs_vnodeops.c instead of from VOP_GETATTR in
 * 	nfs/nfs_server.c.
 * 	[89/06/02  14:23:09  jsb]
 * 
 * Revision 2.3  89/05/01  18:00:03  rpd
 * 	Disabled vnode lingering since it doesn't work yet.
 * 	[89/05/01  17:35:36  jsb]
 * 
 * Revision 2.2  89/04/22  15:13:42  gm0w
 * 	Created for RX version.
 * 	[89/04/14            gm0w]
 * 
 */
/*
 *	File:	afs_gateway.c
 *	Author:	Joseph S. Barrera III
 *
 *	Copyright (C) 1989, Joseph S. Barrera III
 *
 *	Support for afs service to nfs-only machines.
 *
 */

#include <afs/param.h>
#ifdef	AFS_GATEWAY
#include <sys/user.h>
#include <sys/file.h>
#ifdef	AFS_GFS_ENV
#include <afs/gfs_vnode.h>
#else	AFS_GFS_ENV
#ifdef	AFS_MACH_ENV
#include <vfs/vnode.h>
#include <sys/inode.h>
#else	AFS_MACH_ENV
#include <sys/vnode.h>
#include <ufs/inode.h>
#endif	AFS_MACH_ENV
#endif	AFS_GFS_ENV

#define	AFS_CREDTABLESIZE	64	/* should be > no. of nfsds */
#define	AFS_SYSNAMETABLESIZE	16	/* must be power of 2 */
#define	AFS_LINGERTABLESIZE	16	/* max num vnodes allowed to linger */

/*
 *  definitions for implementing @sys
 */

#define	AFS_SYSNAMEHASH(hostaddr)	((hostaddr) & (AFS_SYSNAMETABLESIZE-1))

struct afs_sysname {
	unsigned long hostaddr;
	char *sysname;
	struct afs_sysname *next;
} *afs_sysnametable[AFS_SYSNAMETABLESIZE] = {0};

char *afs_sysname = 0;

/*
 *  definitions for implementing remote creds/uids/pags
 */

struct ucred *remote_cred_table[AFS_CREDTABLESIZE];
unsigned long remote_hostaddr_table[AFS_CREDTABLESIZE];
int remote_cred_count = 0;

#define	AFS_CREDINDEXOFFSET	0x111
#define	AFS_CREDINDEX(cred)	((cred)->cr_ruid - AFS_CREDINDEXOFFSET)
#define	AFS_CREDISLOCAL(cred) 					\
	((AFS_CREDINDEX(cred) < 0) ||				\
	 (AFS_CREDINDEX(cred) >= remote_cred_count) ||		\
	 (remote_cred_table[AFS_CREDINDEX(cred)] != (cred)))
#define	AFS_SETCREDINDEX(cred, val) \
	((cred)->cr_ruid = (val) + AFS_CREDINDEXOFFSET)

/*
 *  definitions for implementing lingering vnodes
 */

struct vnode *linger_vnodes[AFS_LINGERTABLESIZE];
struct ucred *linger_creds[AFS_LINGERTABLESIZE];

int linger_table_count = 0;
int afs_LingeringOK = 1;


/*
 *  For use outside this module, eg. by afs_write
 */
int
afs_CredIsLocal(cred)
struct ucred *cred;
{
	return AFS_CREDISLOCAL(cred);
}

/*
 *  Called by afs_gateway_set_hostaddr, every so often or whenever
 *  there aren't any free slots.
 */
afs_GCRemoteCreds()
{
	int i;
	struct ucred *cred;

	for (i = 0; i < remote_cred_count - 1; ) {
		cred = remote_cred_table[i];
		if (cred->cr_ref > 1) {
			i++;
		} else {
			afs_gateway_unset_hostaddr(cred);
		}
	}
}

/*
 *  Called by afs_rfs_dispatch and PSetClientContext.
 *
 *  Since PSetClientContext doesn't call afs_gateway_unset_hostaddr,
 *  we do a crhold on the cred that we are given; afs_GCRemoteCreds
 *  then looks for credentials with only one reference (ours) when
 *  looking for remote_cred_table slots to free.
 */
afs_gateway_set_hostaddr(cred, hostaddr)
struct ucred *cred;
unsigned long hostaddr;
{
	int index;
	static int calls_since_GC = 0;

	if (remote_cred_count == AFS_CREDTABLESIZE || calls_since_GC++ > 8) {
		afs_GCRemoteCreds();
		calls_since_GC = 0;
	}
	while (remote_cred_count == AFS_CREDTABLESIZE) {
		tablefull("remote cred table");
		sleep((caddr_t)remote_cred_table, PZERO);
	}
	crhold(cred);
	index = remote_cred_count++;
	remote_cred_table[index] = cred;
	remote_hostaddr_table[index] = hostaddr;
	AFS_SETCREDINDEX(cred, index);
}

/*
 *  Called by afs_rfs_dispatch and afs_GCRemoteCreds.
 *
 *  The crfree corresponds to the crhold in afs_gateway_set_hostaddr.
 *  If this routine is called directly from a live process (eg from
 *  afs_rfs_dispatch), the crfree just drops the reference count
 *  to its proper value. If this routine is called by afs_GCRemoteCreds,
 *  then the corresponding process already exited (and called crfree once),
 *  and thus this crfree really will free the cred.
 *  Finally, if the reference count is more that two, then someone else
 *  (eg afs_LingerOpen) did a crhold on cred, and we should not unset hostaddr.
 */
afs_gateway_unset_hostaddr(cred)
struct ucred *cred;
{
	int index;
	int last;

	if (cred->cr_ref > 2) {
		return;
	}
	index = AFS_CREDINDEX(cred);
	last = remote_cred_count - 1;
	if (index != last) {
		remote_cred_table[index] = remote_cred_table[last];
		remote_hostaddr_table[index] = remote_hostaddr_table[last];
		AFS_SETCREDINDEX(remote_cred_table[index], index);
	}
	if (remote_cred_count == AFS_CREDTABLESIZE) {
		wakeup((caddr_t)remote_cred_table);
	}
	remote_cred_count--;
	crfree(cred);
}

/*
 *  Get host addr associated with cred; a host addr of 0 is local host.
 */
unsigned long
afs_gateway_get_hostaddr(cred)
struct ucred *cred;
{
	if (AFS_CREDISLOCAL(cred)) {
		return 0;
	}
	return remote_hostaddr_table[AFS_CREDINDEX(cred)];
}

/*
 *  Set sysname for given hostaddr.
 *  Some special case code is required for local case.
 */
afs_gateway_set_sysname(hostaddr, sysname)
unsigned long hostaddr;
char *sysname;
{
	char **sysnameptr;
	struct afs_sysname **sp, *s;

	if (hostaddr == 0) {
		sysnameptr = &afs_sysname;
	} else {
		int i = AFS_SYSNAMEHASH(hostaddr);
		for (sp = &afs_sysnametable[i];; sp = &s->next) {
			if ((s = *sp) == 0) {
				s = *sp = (struct afs_sysname *) osi_Alloc(sizeof(*s));
				s->hostaddr = hostaddr;
				s->sysname = 0;
				s->next = 0;
				break;
			}
			if (s->hostaddr == hostaddr) {
				break;
			}
		}
		sysnameptr = &s->sysname;
	}
	if (*sysnameptr) {
		osi_Free(*sysnameptr, strlen(*sysnameptr) + 1);
	}
	*sysnameptr = (char *) osi_Alloc(strlen(sysname)+1);
	strcpy(*sysnameptr, sysname);
}

/*
 *  Get sysname for hostaddr of current user.
 */
char *afs_gateway_get_sysname(cred)
struct ucred *cred;
{
	struct afs_sysname *s;
	unsigned long hostaddr;
	int i;

	if (AFS_CREDISLOCAL(cred)) {
		if (afs_sysname == 0) {
			afs_gateway_set_sysname(0, SYS_NAME);
		}
		return afs_sysname;
	}
	hostaddr = afs_gateway_get_hostaddr(cred);
	i = AFS_SYSNAMEHASH(hostaddr);
	for (s = afs_sysnametable[i]; s; s = s->next) {
		if (s->hostaddr == hostaddr) {
			return s->sysname;
		}
	}
	return "@sys";
}

/*
 *  Get the afs uid -- be it a pag or a unix uid -- from a remote cred.
 *  We set the high bit on for pags, and strip it off from uids.
 *
 *  Note that we only assume that a group holds 16 bits of info.
 *  This is because an nfs client can't assume that the server
 *  will use 32 bit groups, even if in fact we do.
 */
unsigned long
afs_gateway_get_auid(cred)
struct ucred *cred;
{
	unsigned long g0, g1, h, l;

#ifdef	AFS_AIX_ENV
	if (cred->cr_ngrps < 2) {
		goto nopag;
	}
	g0 = (unsigned short) cred->grplst[0];
	g1 = (unsigned short) cred->grplst[1];
#else	AFS_AIX_ENV
	g0 = (unsigned short) cred->cr_groups[0];
	g1 = (unsigned short) cred->cr_groups[1];
#endif	AFS_AIX_ENV
	g0 -= 0x3f00;
	g1 -= 0x3f00;
	if (g0 < 0xc000 && g1 < 0xc000) {
		l = ((g0 & 0x3fff) << 14) | (g1 & 0x3fff);
		h = (g0 >> 14);
		h = (g1 >> 14) + h + h + h;
		return 0x80000000 | (h << 28) | l;
	}
nopag:
	return 0x7fffffff & ((unsigned long) cred->cr_uid);
}

/*
 *  When an afs vnode which was written to gets released, the whole
 *  file gets written out to the real afs server. This is bad if
 *  an nfs client of ours is trying to write a big file out to afs,
 *  because for each chuck that we get, we flush the file received so
 *  far, which gives O(n*n) performance (for example a 700KB file
 *  write takes 5 minutes, vs. 15secs for a read). We get around this
 *  problem by faking an afs_open for write on a vnode whenever we see
 *  an afs_write without an afs_open from a remote cred. On a sync
 *  (eg every 30 secs) we then do an afs_close on all such vnodes.
 *  Unfortunately we must remember both vnodes and creds.
 *
 *  We actually hook into the afs_Daemon loop for the 30 second timeout
 *  since actual sync calls can be far more frequent, which is bad.
 *  TO DO: a pioctl to call a routine which flushes all vnodes lingering
 *  for a given hostaddr (client machine).
 */

/*
 *  Mark vp as open; grab references to vp and cred
 */
int
afs_LingerOpen(vp, cred)
struct vnode *vp;
struct ucred *cred;
{
	int error;

	error = afs_open(&vp, FWRITE, cred);
	if (error) {
		return error;
	}
	VN_HOLD(vp);
	crhold(cred);
	return 0;
}

/*
 *  Close vp (letting it flush); release references to vp and cred
 */
afs_LingerClose(vp, cred)
struct vnode *vp;
struct ucred *cred;
{
	struct ucred *saveucred;

	saveucred = u.u_cred;
	u.u_cred = cred; /* since afs_BQueue looks at u.u_cred */
#ifdef	AFS_VFS40
	afs_close(vp, FWRITE, 0, cred); /* 0 is (ignored) count arg */
#else
	afs_close(vp, FWRITE, cred);
#endif
	u.u_cred = saveucred;
	afs_gateway_unset_hostaddr(cred); /* since nfsd's was ignored */
	crfree(cred);
	VN_RELE(vp);
}

/*
 *  LingerOpen vp, but remember vp and cred so that we can LingerClose
 *  them in afs_gateway_sync.
 */
int
afs_gateway_linger(vp, cred)
struct vnode *vp;
struct ucred *cred;
{
	int j, error;

	if (!afs_LingeringOK) {
		return 0;
	}
	error = afs_LingerOpen(vp, cred);
	if (error) {
		return error;
	}
	if (linger_table_count < AFS_LINGERTABLESIZE) {
		linger_table_count++;
	} else {
		afs_LingerClose(linger_vnodes[0], linger_creds[0]); /* LRU */
		for (j = 0; j < AFS_LINGERTABLESIZE - 1; j++) {
			linger_vnodes[j] = linger_vnodes[j + 1];
			linger_creds[j] = linger_creds[j + 1];
		}
	}
	linger_vnodes[linger_table_count - 1] = vp;
	linger_creds[linger_table_count - 1] = cred;
	return 0;
}

/*
 *  We copy the table because it might change from beneath us
 *  when we are afs_closing in afs_LingerClose.
 */
afs_gateway_sync()
{
	int j, count;
	struct vnode *vnodes[AFS_LINGERTABLESIZE];
	struct ucred *creds[AFS_LINGERTABLESIZE];

	count = linger_table_count;
	linger_table_count = 0;
	bcopy(linger_vnodes, vnodes, sizeof(vnodes));
	bcopy(linger_creds, creds, sizeof(creds));
	for (j = 0; j < count; j++) {
		afs_LingerClose(vnodes[j], creds[j]);
	}
}

/*
 *  One last bit of miscellany...
 *
 *  If the user is an nfs gateway client, we OR in the user bits into
 *  the group and other bits. We need to do this because there is
 *  no RFS_ACCESS call and thus nfs clients implement nfs_access by
 *  interpreting the mode bits in the traditional way, which of course
 *  loses with afs.
 *
 *  We might want to have some way for the client to tell us that it
 *  is not brain dead and that we can return bits unmutilated.
 */
afs_gateway_mutilate_mode(cred, attrs)
	struct ucred *cred;
	register struct vattr *attrs;
{
	unsigned int ubits;

	if (! AFS_CREDISLOCAL(cred)) {
		ubits = (attrs->va_mode & 0700) >> 6;
		attrs->va_mode = attrs->va_mode | ubits | (ubits << 3);
	}
}
#endif	AFS_GATEWAY
