/* 
 * Mach Operating System
 * Copyright (c) 1989 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
/*
 * HISTORY
 * $Log:	afs_buffer.c,v $
 * Revision 2.8  89/08/02  07:58:01  jsb
 * 	Added afs_num_buffers to allow allocation of far fewer buffers.
 * 	There is no good reason to keep a large buffer cache here
 * 	since cache misses here go through the real buffer cache anyway.
 * 	[89/07/31  18:36:08  jsb]
 * 
 * Revision 2.7  89/06/03  15:26:12  jsb
 * 	Merged with newer ITC sources.
 * 	[89/06/02  01:08:07  jsb]
 * 
 * Revision 2.6  89/04/22  15:12:49  gm0w
 * 	Updated to RX version.  Added MACH format includes.
 * 	[89/04/14            gm0w]
 * 
 */

#ifndef lint
#endif

/*
 * P_R_P_Q_# (C) COPYRIGHT IBM CORPORATION 1987
 * LICENSED MATERIALS - PROPERTY OF IBM
 * REFER TO COPYRIGHT INSTRUCTIONS FORM NUMBER G120-2083
 */
/*
File				buffer.c
Author			Mike Kazar
Date				Now
*/

#include <afs/param.h>
#include <sys/param.h>
#include <sys/time.h>
#ifndef	AFS_AIX_ENV
#include <sys/kernel.h>    /* Doesn't needed, so it should go */
#endif

#include <afs/osi.h>
#include <afs/lock.h>

#include <sys/buf.h>

/* number of pages per Unix buffer, when we're using Unix buffer pool */
#define	NPB 4
/* page size */
#define AFS_PAGESIZE 2048
/* log page size */
#define LOGPS 11
/* page hash table size */
#define PHSIZE 32
/* the pHash macro */
#define pHash(fid) ((fid)[0] & (PHSIZE-1))

struct buffer {
    long fid[5];	/* Unique cache key + i/o addressing */
    long page;
    long accesstime;
    struct buffer *hashNext;
    char *data;
    char lockers;
    char dirty;
    char hashIndex;
#if AFS_USEBUFFERS
    struct buf *bufp;
#endif
} *Buffers;

char *BufferData;

#ifdef	AFS_AIX_ENV
extern struct buf *geteblk();
#endif
static struct afs_lock afs_bufferLock;
static struct buffer *phTable[PHSIZE];	/* page hash table */
static struct buffer *LastBuffer;
int nbuffers;
int timecounter;
static int calls=0, ios=0;

struct buffer *afs_newslot();

/* if non-zero, overrides abuffers in DInit */
int afs_num_buffers = 4;

int DInit (abuffers)
    int abuffers; {
    /* Initialize the venus buffer system. */
    register int i;
    register struct buffer *tb;
#if AFS_USEBUFFERS
    struct buf *tub;	    /* unix buffer for allocation */
#endif

    if (afs_num_buffers) abuffers = afs_num_buffers;
#if AFS_USEBUFFERS
    /* round up to next multiple of NPB, since we allocate multiple pages per chunk */
    abuffers = ((abuffers-1) | (NPB-1)) + 1;
#endif
    Lock_Init(&afs_bufferLock);
    Buffers = (struct buffer *) osi_Alloc(abuffers * sizeof(struct buffer));
#if !AFS_USEBUFFERS
    BufferData = (char *) osi_Alloc(abuffers * AFS_PAGESIZE);
#endif
    timecounter = 0;
    LastBuffer = Buffers;
    nbuffers = abuffers;
    for(i=0;i<PHSIZE;i++) phTable[i] = 0;
    for (i=0;i<abuffers;i++) {
#if AFS_USEBUFFERS
	if ((i & (NPB-1)) == 0) {
	    /* time to allocate a fresh buffer */
	    tub = geteblk(AFS_PAGESIZE*NPB);
	    BufferData = (char *) tub->b_un.b_addr;
	}
#endif
        /* Fill in each buffer with an empty indication. */
	tb = &Buffers[i];
        dirp_Zap(tb->fid);
        tb->accesstime = tb->lockers = 0;
#if AFS_USEBUFFERS
	if ((i & (NPB-1)) == 0) 
	    tb->bufp = tub;
	else
	    tb->bufp = 0;
	tb->data = &BufferData[AFS_PAGESIZE * (i&(NPB-1))];
#else
        tb->data = &BufferData[AFS_PAGESIZE*i];
#endif
	tb->hashIndex = 0;
        tb->dirty = 0;
    }
    return 0;
}

char *DRead(fid,page)
    register long *fid;
    register int page; {
    /* Read a page from the disk. */
    register struct buffer *tb;

    ObtainWriteLock(&afs_bufferLock);
    calls++;
    if (LastBuffer->page == page && dirp_Eq(LastBuffer->fid, fid)) {
	tb = LastBuffer;
	tb->accesstime = ++timecounter;
	tb->lockers++;
	ReleaseWriteLock(&afs_bufferLock);
	return tb->data;
    }
    for(tb=phTable[pHash(fid)]; tb; tb=tb->hashNext) {
	if (tb->page == page && dirp_Eq(tb->fid, fid)) {
	    tb->lockers++;
	    tb->accesstime = ++timecounter;
	    LastBuffer = tb;
	    ReleaseWriteLock(&afs_bufferLock);
	    return tb->data;
	}
    }
    /* can't find it */
    tb = afs_newslot(fid, page);
    tb->lockers++;
    if (!dirp_Read(fid,tb->page,tb->data)) {
        dirp_Zap(tb->fid);	/* disaster */
	tb->lockers--;
	ReleaseWriteLock(&afs_bufferLock);
        return 0;
    }
    ios++;
    /* Note that findslot sets the page field in the buffer equal to what it is searching for. */
    ReleaseWriteLock(&afs_bufferLock);
    return tb->data;
}

static FixupBucket(ap)
    register struct buffer *ap; {
    register struct buffer **lp, *tp;
    register int i;
    /* first try to get it out of its current hash bucket, in which it might not be */
    i = ap->hashIndex;
    lp = &phTable[i];
    for(tp = *lp; tp; tp=tp->hashNext) {
	if (tp == ap) {
	    *lp = tp->hashNext;
	    break;
	}
	lp = &tp->hashNext;
    }
    /* now figure the new hash bucket */
    i = pHash(ap->fid);
    ap->hashIndex = i;		/* remember where we are for deletion */
    ap->hashNext = phTable[i];	/* add us to the list */
    phTable[i] = ap;
}

struct buffer *afs_newslot (afid,apage)
    long *afid, apage; {
    /* Find a usable buffer slot */
    register long i;
    long lt,pt;
    register struct buffer *lp, *pp, *tp;

    lp = 0;		/* last non-pure */
    pp = 0;		/* last pure */
    lt = 999999999;
    pt = 999999999;
    tp = Buffers;
    for (i=0;i<nbuffers;i++,tp++) {
	if (tp->lockers == 0) {
	    if (tp->dirty) {
		if (tp->accesstime < lt) {
		    lp = tp;
		    lt = tp->accesstime;
		}
	    }
	    else if (tp->accesstime < pt) {
		pp = tp;
		pt = tp->accesstime;
	    }
	}
    }
    /* If we make it here, the buffer is not in memory.  Find an already-used buffer and trash it.
	If the buffer is dirty, try not to use it.  If it must be used, don't forget to write it out first. */

    if (pp == 0) {
        /* There are no unlocked buffers that don't need to be written to the disk.
	    The variable lx gives the index of the buffer to write out to the disk. */
        if (lp == 0) Die ("all buffers locked");
        if (!dirp_Write(lp->fid,lp->page,lp->data)) Die("writing bogus buffer");
        lp->dirty = 0;
        pp = lp;		/* The buffer to use from now on. */
    }

    /* Now fill in the header. */
    dirp_Cpy(pp->fid, afid);	/* set this */
    pp->page = apage;
    pp->accesstime = ++timecounter;

    FixupBucket(pp);		/* move to the right hash bucket */

    LastBuffer = pp;
    return pp;
}

DRelease (bp,flag)
    register struct buffer *bp;
    int flag; {
    /* Release a buffer, specifying whether or not the buffer has been modified by the locker. */
    register int index;
#if AFS_USEBUFFERS
    register struct buffer *tp;
#endif

    if (!bp) return;
#if AFS_USEBUFFERS
    /* look for buffer by scanning Unix buffers for appropriate address */
    tp = Buffers;
    for(index = 0; index < nbuffers; index += NPB, tp += NPB) {
	if ((long)bp >= (long)tp->data && (long)bp < (long)tp->data + AFS_PAGESIZE*NPB) {
	    /* we found the right range */
	    index += ((long)bp - (long)tp->data) >> LOGPS;
	    break;
	}
    }
#else
    index = (((char *)bp)-((char *)BufferData))>>LOGPS;
#endif
    bp = &(Buffers[index]);
    ObtainWriteLock(&afs_bufferLock);
    bp->lockers--;
    if (flag) bp->dirty=1;
    ReleaseWriteLock(&afs_bufferLock);
}

DVOffset (ap)
    register struct buffer *ap; {
    /* Return the byte within a file represented by a buffer pointer. */
    register struct buffer *bp;
    register int index;
#if AFS_USEBUFFERS
    register struct buffer *tp;
#endif
    bp=ap;
#if AFS_USEBUFFERS
    /* look for buffer by scanning Unix buffers for appropriate address */
    tp = Buffers;
    for(index = 0; index < nbuffers; index += NPB, tp += NPB) {
	if ((long)bp >= (long)tp->data && (long)bp < (long)tp->data + AFS_PAGESIZE*NPB) {
	    /* we found the right range */
	    index += ((long)bp - (long)tp->data) >> LOGPS;
	    break;
	}
    }
#else
    index = (((char *)bp)-((char *)BufferData))>>LOGPS;
#endif
    if (index<0 || index >= nbuffers) return -1;
    bp = &(Buffers[index]);
    return AFS_PAGESIZE*bp->page+((char *)ap)-bp->data;
}

DZap (fid)
    register long *fid; {
    /* Destroy all buffers pertaining to a particular fid. */
    register struct buffer *tb;
    ObtainWriteLock(&afs_bufferLock);
    for(tb=phTable[pHash(fid)]; tb; tb=tb->hashNext)
        if (dirp_Eq(tb->fid,fid)) {
            dirp_Zap(tb->fid);
            tb->dirty = 0;
	}
    ReleaseWriteLock(&afs_bufferLock);
}

#ifdef notdef
/* we don't seem to use this one */
DFlushEntry (fid)
    register long *fid; {
    /* Flush pages modified by one entry. */
    register struct buffer *tb;
    ObtainWriteLock(&afs_bufferLock);
    for(tb = phTable[pHash(fid)]; tb; tb=tb->hashNext)
        if (tb->dirty && dirp_Eq(tb->fid, fid)) {
            if (dirp_Write(tb->fid, tb->page, tb->data)) tb->dirty = 0;
	}
    ReleaseWriteLock(&afs_bufferLock);
}
#endif

DFlush () {
    /* Flush all the modified buffers. */
    register int i;
    register struct buffer *tb;

    tb = Buffers;
    ObtainWriteLock(&afs_bufferLock);
    for(i=0;i<nbuffers;i++,tb++) {
        if (tb->dirty && dirp_Write(tb->fid, tb->page, tb->data))
            tb->dirty = 0;	/* Clear the dirty flag */
    }
    ReleaseWriteLock(&afs_bufferLock);
}

char *DNew (fid,page)
    register int page;
    register long *fid; {
    /* Same as read, only do *not* even try to read the page, since it probably doesn't exist. */
    register struct buffer *tb;
    ObtainWriteLock(&afs_bufferLock);
    if ((tb = afs_newslot(fid,page)) == 0) {
	ReleaseWriteLock(&afs_bufferLock);
	return 0;
    }
    tb->lockers++;
    ReleaseWriteLock(&afs_bufferLock);
    return tb->data;
}

shutdown_bufferpackage() {
#if AFS_USEBUFFERS
    register struct buffer *tp;
#endif
    int i;

    /* Free all allocated Buffers and associated buffer pages */
    DFlush();
    osi_Free(Buffers, nbuffers * sizeof(struct buffer));
#if !AFS_USEBUFFERS
    osi_Free(BufferData, nbuffers * AFS_PAGESIZE);
#else
    tp = Buffers;
    for (i=0; i < nbuffers; i+= NPB, tp += NPB) {
	/* The following check shouldn't be necessary and it will be removed soon */
	if (!tp->bufp) 
	    printf("shutdown_bufferpackage: bufp == 0!! Shouldn't happen\n");
	else {
	    brelse(tp->bufp);
	    tp->bufp = 0;
	}
    }
#endif
    LastBuffer = 0;
    nbuffers = timecounter = calls = ios = 0;
    for(i=0;i<PHSIZE;i++) phTable[i] = 0;
    bzero(&afs_bufferLock, sizeof(struct afs_lock));
}  
