/*
 * Remote Laboratory Serial Server
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * (c) 2014 Timothy Pearson
 * Raptor Engineering
 * http://www.raptorengineeringinc.com
 */

#include <stdio.h>                /* perror() */
#include <stdlib.h>               /* atoi() */
#include <sys/types.h>
#include <sys/socket.h>
#include <unistd.h>               /* read() */
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <fcntl.h>
#include <errno.h>
#include <termios.h>
#include <unistd.h>
#include <sys/signal.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <math.h>

#include <tqtimer.h>

#include <tdelocale.h>

#include "serial_server.h"

#define FLUSH_IN	0
#define FLUSH_OUT	1
#define FLUSH_BOTH	2

#define ABORT_SOCKET(s) 		s->close();						\
					s->disconnect();					\
					delete s;						\
					s = NULL;

/* exception handling */
struct exit_exception {
	int c;
	exit_exception(int c):c(c) { }
};

/*
  The SerialSocket class provides a socket that is connected with a client.
  For every client that connects to the server, the server creates a new
  instance of this class.
*/
SerialSocket::SerialSocket(int sock, TQObject *parent, const char *name) :
	TDEKerberosServerSocket(parent, name), m_criticalSection(0), m_pollInterval(10), enableDebug(false), m_loopTimer(NULL), m_config(static_cast<SerialServer*>(parent)->m_config) {

	// Read settings
        m_config->setGroup("Tuning");
        m_pollInterval = m_config->readNumEntry("pollInterval", m_pollInterval);
	enableDebug = m_config->readBoolEntry("enableDebug", enableDebug);

	// Initialize timers
	m_kerberosInitTimer = new TQTimer();
	connect(m_kerberosInitTimer, SIGNAL(timeout()), this, SLOT(finishKerberosHandshake()));

	setServiceName("ulab");

	line = 0;
	connect(this, SIGNAL(connectionClosed()), SLOT(connectionClosedHandler()));
	connect(this, SIGNAL(connectionClosed()), parent, SLOT(remoteConnectionClosed()));
	setSocket(sock);
}

SerialSocket::~SerialSocket() {
	if (m_kerberosInitTimer) {
		m_kerberosInitTimer->stop();
		delete m_kerberosInitTimer;
		m_kerberosInitTimer = NULL;
	}
	if (m_loopTimer) {
		m_loopTimer->stop();
		delete m_loopTimer;
		m_loopTimer = NULL;
	}
}

void SerialSocket::close() {
	if (state() == TQSocket::Connected) {
		TDEKerberosServerSocket::close();
		connectionClosedHandler();
		TQTimer::singleShot(0, parent(), SLOT(remoteConnectionClosed()));
	}
}

void SerialSocket::connectionClosedHandler() {
	if (enableDebug) {
		printf("[DEBUG] Connection from %s closed\n\r", m_remoteHost.ascii()); fflush(stdout);
	}

	if (m_criticalSection > 0) {
		throw exit_exception(-1);
	}
}

void SerialSocket::initiateKerberosHandshake() {
	setUsingKerberos(true);
	m_kerberosInitTimer->start(100, TRUE);
}

void SerialSocket::finishKerberosHandshake() {
	if (kerberosStatus() == TDEKerberosServerSocket::KerberosInitializing) {
		m_kerberosInitTimer->start(100, TRUE);
		return;
	}
	if (kerberosStatus() == TDEKerberosServerSocket::KerberosInUse) {
		m_config->setGroup("Security");
		TQString masterUser = m_config->readEntry("masteruser");
		TQString masterRealm = m_config->readEntry("masterrealm");
		if (masterRealm == "") {
			masterRealm = "(NULL)";
		}
		if ((m_authenticatedUserName != masterUser) || (m_authenticatedRealmName != masterRealm)) {
			if (enableDebug) {
				printf("[DEBUG] Connection from %s closed due to authentication failure (attempted connection as user %s@%s)\n\r", m_remoteHost.ascii(), m_authenticatedUserName.ascii(), m_authenticatedRealmName.ascii()); fflush(stdout);
			}
			close();
			return;
		}
		if (setupSerial() != 0) {
			if (enableDebug) {
				printf("[DEBUG] Connection from %s closed due to serial port initialization failure\n\r", m_remoteHost.ascii()); fflush(stdout);
			}
			close();
			return;
		}

		TQDataStream ds(this);
		ds.setPrintableData(true);
		ds << TQString("OK");
		writeEndOfFrame();

		// Inform user of baudrate
		TQString infoString;
		infoString = TQString("Serial connection established with a baudrate of %1\rReady for data").arg(m_baudRate);
		TQByteArray data;
		data.duplicate(infoString.ascii(), strlen(infoString.ascii()));
		ds << TQString("DATA");
		ds << data;
		writeEndOfFrame();

		enterCommandLoop();
		return;
	}
	else {
		if (enableDebug) {
			printf("[DEBUG] Connection from %s closed due to Kerberos failure\n\r", m_remoteHost.ascii()); fflush(stdout);
		}
		close();
		return;
	}
}

int SerialSocket::setupSerial() {
	struct termios oldtio, newtio;

	m_config->setGroup("Serial");
	TQString serialDevice = m_config->readEntry("serialdevice", "/dev/ttyS0");
	m_baudRate = m_config->readEntry("baudrate", "9600");

	m_fd_tty = ::open(serialDevice.ascii(), O_RDWR | O_NOCTTY | O_NONBLOCK | O_APPEND);
	if (m_fd_tty < 0) {
		printf("[FAIL] Unable to open serial device %s\n\r", serialDevice.ascii()); fflush(stdout);
		return 1;
	}

	tcgetattr(m_fd_tty, &oldtio);	// Save current port settings

	long serialBaud;
	if (m_baudRate == "1200") {
		serialBaud = B1200;
	}
	else if (m_baudRate == "9600") {
		serialBaud = B9600;
	}
	else if (m_baudRate == "19200") {
		serialBaud = B19200;
	}
	else if (m_baudRate == "115200") {
		serialBaud = B115200;
	}
	else {
		printf("[WARNING] Invalid baudrate %s specified, selecting 9600 instead\n\r", m_baudRate.ascii()); fflush(stdout);
		serialBaud = B9600;
	}

	bzero(&newtio, sizeof(newtio));
	newtio.c_cflag = serialBaud | CS8 | CLOCAL | CREAD;
	newtio.c_iflag = IGNPAR;
	newtio.c_oflag = 0;

	// Set input mode (non-canonical, no echo,...)
	newtio.c_lflag = 0;

	newtio.c_cc[VTIME] = 0;	// Inter-character timer unused
	newtio.c_cc[VMIN]  = 0;	// Blocking read unused
	
	tcflush(m_fd_tty, TCIFLUSH);
	tcsetattr(m_fd_tty, TCSANOW, &newtio);

	return 0;
}

void SerialSocket::commandLoop() {
	int cc;
	int ret;
	char buffer[1024];
	bool transferred_data;
	TQString instrumentCommand;

	m_criticalSection++;
	try {
		transferred_data = false;
		if (state() == TQSocket::Connected) {
			if (canReadFrame()) {
				TQDataStream ds(this);
				ds.setPrintableData(true);
				ds >> instrumentCommand;

				if (instrumentCommand != "") {
					if ((instrumentCommand == "SEND")) {		// Send data to serial port
						TQByteArray data;
						ds >> data;
						ret = write(m_fd_tty, data.data(), data.size());
						while ((ret < 0) && (errno == EAGAIN)) {
							usleep(1000);
							ret = write(m_fd_tty, data.data(), data.size());
						}
						ioctl(m_fd_tty, TCFLSH, FLUSH_OUT);
						if (ret >= 0) {
							ds << TQString("ACK");
							if (enableDebug) {
								printf("[DEBUG] Wrote %d bytes to the serial port\n\r", ret); fflush(stdout);
							}
						}
						else {
							ds << TQString("NCK");
						}
						writeEndOfFrame();
					}
					else if ((instrumentCommand == "PING")) {
						ds << TQString("PONG");
						writeEndOfFrame();
					}
					else {
						printf("[WARNING] Received unknown command %s from host %s\n\r", instrumentCommand.ascii(), m_remoteHost.ascii()); fflush(stdout);
						ds << TQString("NCK");
						writeEndOfFrame();
					}
				}

				transferred_data = true;
			}
			cc = read(m_fd_tty, buffer, 1024);
			if (cc > 0) {
				TQDataStream ds(this);
				ds.setPrintableData(true);
				ds >> instrumentCommand;

				TQByteArray data;
				data.duplicate(buffer, cc);
				ds << TQString("DATA");
				ds << data;
				writeEndOfFrame();

				transferred_data = true;
				if (enableDebug) {
					printf("[DEBUG] Got %d bytes from the serial port\n\r", cc); fflush(stdout);
				}
			}
		}
		m_criticalSection--;
		if (transferred_data) {
			if (m_loopTimer) m_loopTimer->start(0, TRUE);
		}
		else {
			if (m_loopTimer) m_loopTimer->start(m_pollInterval, TRUE);
		}
		return;
	}
	catch (...) {
		m_criticalSection--;
		return;
	}
}

int SerialSocket::enterCommandLoop() {
	if (!m_loopTimer) {
		m_loopTimer = new TQTimer();
		connect(m_loopTimer, SIGNAL(timeout()), this, SLOT(commandLoop()));
	}
	if (m_loopTimer) m_loopTimer->start(0, TRUE);
	return 0;
}

/*
  The SerialServer class handles new connections to the server. For every
  client that connects, it creates a new SerialSocket -- that instance is now
  responsible for the communication with that client.
*/
SerialServer::SerialServer(TQObject* parent, int port, KSimpleConfig* config) :
	TQServerSocket( port, 1, parent ), m_config(config), m_numberOfConnections(0) {

	if ( !ok() ) {
		printf("[ERROR] Failed to bind to port %d\n\r", port);
		exit(1);
	}

	printf("[INFO] Server started on port %d\n\r", port); fflush(stdout);
}

SerialServer::~SerialServer() {
	//
}

void SerialServer::newConnection(int socket) {
	SerialSocket *s = new SerialSocket(socket, this);
	s->m_remoteHost = s->peerAddress().toString();
	printf("[DEBUG] New connection from %s\n\r", s->m_remoteHost.ascii()); fflush(stdout);
	if (m_numberOfConnections > 0) {
		printf("[DEBUG] Connection from %s closed due to multiple access attempt\n\r", s->m_remoteHost.ascii()); fflush(stdout);
		ABORT_SOCKET(s)
		return;
	}
	connect(s, SIGNAL(connectionClosed()), s, SLOT(deleteLater()));
	s->initiateKerberosHandshake();
	emit newConnect(s);
}

void SerialServer::remoteConnectionClosed() {
	m_numberOfConnections--;
}
