/* This file is part of the KDE project
   Copyright (C) 2005-2006 Jaroslaw Staniek <js@iidea.pl>

   This program is free software; you can redistribute it and,or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#include "kexitableviewheader.h"

#include <tqapplication.h>
#include <tqtooltip.h>
#include <tqstyle.h>

#include <kexiutils/utils.h>
#include <kexiutils/styleproxy.h>

//! @internal A style that allows to temporary change background color while
//!           drawing header section primitive. Used in KexiTableViewHeader.
class KexiTableViewHeaderStyle : public KexiUtils::StyleProxy
{
	public:
		KexiTableViewHeaderStyle(TQStyle *parentStyle, TQWidget *widget)
		 : KexiUtils::StyleProxy(parentStyle)
		{
			setBackgroundColor( widget->palette().active().background() );
		}
		~KexiTableViewHeaderStyle() {}

		virtual void drawPrimitive( PrimitiveElement pe,
			TQPainter *p, const TQRect &r, const TQColorGroup &cg, SFlags flags = Style_Default,
			const TQStyleOption& option = TQStyleOption::Default ) const
		{
			if (pe==TQStyle::PE_HeaderSection) {
				TQColorGroup newCg(cg);
				newCg.setColor(TQColorGroup::Button, m_backgroundColor);
				newCg.setColor(TQColorGroup::Background, m_backgroundColor); //set background color as well (e.g. for thinkeramik)
				m_style->drawPrimitive( pe, p, r, newCg, flags, option );
				return;
			}
			m_style->drawPrimitive( pe, p, r, cg, flags, option );
		}

		void setBackgroundColor( const TQColor& color ) { m_backgroundColor = color; }

	protected:
		TQColor m_backgroundColor;
};

KexiTableViewHeader::KexiTableViewHeader(TQWidget * parent, const char * name) 
	: TQHeader(parent, name)
	, m_lastToolTipSection(-1)
	, m_selectionBackgroundColor(tqApp->palette().active().highlight())
	, m_selectedSection(-1)
	, m_styleChangeEnabled(true)
{
	styleChange( style() );
	installEventFilter(this);
	connect(this, TQ_SIGNAL(sizeChange(int,int,int)), 
		this, TQ_SLOT(slotSizeChange(int,int,int)));
}

KexiTableViewHeader::~KexiTableViewHeader()
{
}

void KexiTableViewHeader::styleChange( TQStyle& oldStyle )
{
	TQHeader::styleChange( oldStyle );
	if (!m_styleChangeEnabled)
		return;
	m_styleChangeEnabled = false;
	setStyle( new KexiTableViewHeaderStyle(&tqApp->style(), this) );
	m_styleChangeEnabled = true;
}

int KexiTableViewHeader::addLabel ( const TQString & s, int size )
{
	m_toolTips += "";
	slotSizeChange(0,0,0);//refresh
	return TQHeader::addLabel(s, size);
}

int KexiTableViewHeader::addLabel ( const TQIconSet & iconset, const TQString & s, int size )
{
	m_toolTips += "";
	slotSizeChange(0,0,0);//refresh
	return TQHeader::addLabel(iconset, s, size);
}

void KexiTableViewHeader::removeLabel( int section )
{
	if (section < 0 || section >= count())
		return;
	TQStringList::Iterator it = m_toolTips.begin();
	it += section;
	m_toolTips.remove(it);
	slotSizeChange(0,0,0);//refresh
	TQHeader::removeLabel(section);
}

void KexiTableViewHeader::setToolTip( int section, const TQString & toolTip )
{
	if (section < 0 || section >= (int)m_toolTips.count())
		return;
	m_toolTips[ section ] = toolTip;
}

bool KexiTableViewHeader::eventFilter(TQObject * watched, TQEvent * e)
{
	if (e->type()==TQEvent::MouseMove) {
		const int section = sectionAt( static_cast<TQMouseEvent*>(e)->x() );
		if (section != m_lastToolTipSection && section >= 0 && section < (int)m_toolTips.count()) {
			TQToolTip::remove(this, m_toolTipRect);
			TQString tip = m_toolTips[ section ];
			if (tip.isEmpty()) { //try label
				TQFontMetrics fm(font());
				int minWidth = fm.width( label( section ) ) + style().pixelMetric( TQStyle::PM_HeaderMargin );
				TQIconSet *iset = iconSet( section );
				if (iset)
					minWidth += (2+iset->pixmap( TQIconSet::Small, TQIconSet::Normal ).width()); //taken from TQHeader::sectionSizeHint()
				if (minWidth > sectionSize( section ))
					tip = label( section );
			}
			if (tip.isEmpty()) {
				m_lastToolTipSection = -1;
			}
			else {
				TQToolTip::add(this, m_toolTipRect = sectionRect(section), tip);
				m_lastToolTipSection = section;
			}
		}
	}
//			if (e->type()==TQEvent::MouseButtonPress) {
//	todo
//			}
	return TQHeader::eventFilter(watched, e);
}

void KexiTableViewHeader::slotSizeChange(int /*section*/, int /*oldSize*/, int /*newSize*/ )
{
	if (m_lastToolTipSection>0)
		TQToolTip::remove(this, m_toolTipRect);
	m_lastToolTipSection = -1; //tooltip's rect is now invalid
}

void KexiTableViewHeader::setSelectionBackgroundColor(const TQColor &color)
{
	m_selectionBackgroundColor = color;
}

TQColor KexiTableViewHeader::selectionBackgroundColor() const
{
	return m_selectionBackgroundColor;
}

void KexiTableViewHeader::setSelectedSection(int section)
{
	if (m_selectedSection==section || (section!=-1 && section>=count()))
		return;
	const int oldSection = m_selectedSection;
	m_selectedSection = section;
	if (oldSection!=-1)
		update(sRect(oldSection));
	if (m_selectedSection!=-1)
		update(sRect(m_selectedSection));
}

int KexiTableViewHeader::selectedSection() const
{
	return m_selectedSection;
}

void KexiTableViewHeader::paintSection( TQPainter * p, int index, const TQRect & fr )
{
	const bool paintSelection = index==m_selectedSection && index != -1;
	if (paintSelection) {
		static_cast<KexiTableViewHeaderStyle&>(style()).setBackgroundColor(
			KexiUtils::blendedColors( 
				palette().active().background(), m_selectionBackgroundColor, 2, 1) );
	}

	TQHeader::paintSection( p, index, fr );

	if (paintSelection) { //revert the color for subsequent paints
		static_cast<KexiTableViewHeaderStyle&>(style()).setBackgroundColor(
			palette().active().background());
	}
}

#include "kexitableviewheader.moc"
