/* ============================================================
 * File  : caltemplate.cpp
 * Author: Renchi Raju <renchi@pooh.tam.uiuc.edu>
 * Date  : 2003-11-04
 * Description :
 *
 * Copyright 2003 by Renchi Raju <renchi@pooh.tam.uiuc.edu>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

// TQt includes.

#include <tqvariant.h>
#include <tqframe.h>
#include <tqgroupbox.h>
#include <tqlabel.h>
#include <tqcombobox.h>
#include <tqbuttongroup.h>
#include <tqvbuttongroup.h>
#include <tqradiobutton.h>
#include <tqcheckbox.h>
#include <tqslider.h>
#include <tqlayout.h>
#include <tqtimer.h>
#include <tqfontdatabase.h>
#include <tqpaintdevicemetrics.h>
#include <tqpixmap.h>
#include <tqpainter.h>
#include <tqprinter.h>

// KDE includes.

#include <tdelocale.h>
#include <tdestandarddirs.h>

// Local includes.

#include "caltemplate.h"
#include "calwidget.h"
#include "calsettings.h"
#include "calpainter.h"

namespace KIPICalendarPlugin
{

CalTemplate::CalTemplate(TQWidget* parent, const char* name)
           : TQWidget(parent, name)
{
    TQGridLayout *mainLayout = new TQGridLayout(this, 1, 1, 5, 5);

    // ----------------------------------------------------------------

    previewSize_ = 300;

    TQGroupBox *boxPreview_ = new TQGroupBox( i18n("Preview"), this );
    boxPreview_->setColumnLayout(0, TQt::Vertical);
    boxPreview_->layout()->setMargin( 5 );

    TQVBoxLayout *previewLayout = new TQVBoxLayout(boxPreview_->layout());
    calWidget_ = new CalWidget(boxPreview_);
    previewLayout->addWidget(calWidget_, 0, TQt::AlignCenter);

    mainLayout->addWidget( boxPreview_, 0, 0 );

    // ---------------------------------------------------------------

    TQGroupBox *gbox = new TQGroupBox( i18n("Settings"), this );
    gbox->setColumnLayout(0, TQt::Vertical );
    gbox->layout()->setSpacing( 11 );
    gbox->layout()->setMargin( 6 );
    TQVBoxLayout* gboxLayout = new TQVBoxLayout( gbox->layout() );

    // ---------------------------------------------------------------

    TQHBoxLayout *hlayout = new TQHBoxLayout( 0, 0, 5 );
    hlayout->addWidget( new TQLabel(i18n("Paper size:"), gbox) );

    comboPaperSize_ = new TQComboBox(false, gbox);
    hlayout->addWidget(comboPaperSize_);
    gboxLayout->addLayout( hlayout );

    TQStringList paperSizes;
    paperSizes << "A4";
    paperSizes << "US Letter";
    comboPaperSize_->insertStringList(paperSizes);

    connect(comboPaperSize_, TQ_SIGNAL(activated(int)),
            TQ_SLOT(slotParamsChanged()));

    // ---------------------------------------------------------------

    btnGroupImagePos_ = new TQVButtonGroup(i18n("Image Position"), gbox);
    btnGroupImagePos_->setRadioButtonExclusive(true);

    // Insert the buttons in the order Top, Left, Right so that they
    // get the corresponding ids
    TQRadioButton* radioBtn = new TQRadioButton(i18n("Top"), btnGroupImagePos_);
    radioBtn->setChecked(true);
    Q_ASSERT(btnGroupImagePos_->id( radioBtn ) == CalParams::Top);

    radioBtn = new TQRadioButton(i18n("Left"), btnGroupImagePos_);
    Q_ASSERT(btnGroupImagePos_->id( radioBtn ) == CalParams::Left);

    radioBtn = new TQRadioButton(i18n("Right"), btnGroupImagePos_);
    Q_ASSERT(btnGroupImagePos_->id( radioBtn ) == CalParams::Right);

    gboxLayout->addWidget( btnGroupImagePos_ );

    connect(btnGroupImagePos_, TQ_SIGNAL(clicked(int)),
            TQ_SLOT(slotParamsChanged()));

    // ---------------------------------------------------------------

    checkBoxDrawLines_ = new TQCheckBox(i18n("Draw lines in calendar"), gbox);
    gboxLayout->addWidget( checkBoxDrawLines_ );
    checkBoxDrawLines_->setChecked(true);

    connect(checkBoxDrawLines_, TQ_SIGNAL(toggled(bool)),
            TQ_SLOT(slotParamsChanged()));

    // ---------------------------------------------------------------

    hlayout = new TQHBoxLayout( 0, 0, 5 );

    hlayout->addWidget(new TQLabel(i18n("Image to text ratio:"), gbox));

    sliderRatio_ = new TQSlider(50,300,5,100,TQt::Horizontal,gbox);
    hlayout->addWidget( sliderRatio_ );

    gboxLayout->addLayout( hlayout );

    connect(sliderRatio_, TQ_SIGNAL(valueChanged(int)),
            TQ_SLOT(slotParamsChanged()));

    // ---------------------------------------------------------------

    hlayout = new TQHBoxLayout( 0, 0, 5 );

    hlayout->addWidget(new TQLabel(i18n("Font:"), gbox));

    comboFont_ = new TQComboBox(false, gbox);
    hlayout->addWidget( comboFont_ );

    TQFontDatabase fontDB;
    TQStringList families(fontDB.families());
    TQStringList smoothScalableFamilies;
    for (TQStringList::iterator it=families.begin(); it != families.end();
         ++it)
    {
        smoothScalableFamilies.append(*it);
    }
    comboFont_->insertStringList(smoothScalableFamilies);

    // fetch and set the default font selected in the combo.
    TQFont f;
    comboFont_->setCurrentText( f.family() );


    gboxLayout->addLayout( hlayout );

    connect(comboFont_, TQ_SIGNAL(activated(int)),
            TQ_SLOT(slotParamsChanged()));


    // ---------------------------------------------------------------

    gboxLayout->addItem(new TQSpacerItem(5,10,TQSizePolicy::Minimum,
                                        TQSizePolicy::Expanding));

    mainLayout->addWidget( gbox, 0, 1 );

    // ---------------------------------------------------------------

    timer_ = new TQTimer(this);

    connect(timer_, TQ_SIGNAL(timeout()),
            TQ_SLOT(slotUpdatePreview()));

    timer_->start(0,true);
}

CalTemplate::~CalTemplate()
{
    delete timer_;
}


void CalTemplate::slotParamsChanged()
{
    timer_->start(10, true);
}

void CalTemplate::slotUpdatePreview()
{
    if (timer_->isActive()) return;

    CalParams& params = CalSettings::instance()->calParams;

    TQString paperSize = comboPaperSize_->currentText();
    if (paperSize == "A4") {
        params.paperWidth  = 210;
        params.paperHeight = 297;
        params.pageSize    = KPrinter::A4;
    }
    else if (paperSize == "US Letter") {
        params.paperWidth  = 216;
        params.paperHeight = 279;
        params.pageSize    = KPrinter::Letter;
    }

    const int imgPos = btnGroupImagePos_->selectedId();

    if (imgPos == CalParams::Top) {
        params.imgPos = CalParams::Top;

        float zoom = TQMIN((float)previewSize_/params.paperWidth,
                          (float)previewSize_/params.paperHeight);
        params.width  = (int)(params.paperWidth  * zoom);
        params.height = (int)(params.paperHeight * zoom);

    }
    else if (imgPos == CalParams::Left) {
        params.imgPos = CalParams::Left;

        float zoom = TQMIN((float)previewSize_/params.paperWidth,
                          (float)previewSize_/params.paperHeight);
        params.width  = (int)(params.paperHeight  * zoom);
        params.height = (int)(params.paperWidth   * zoom);

    }
    else {
        params.imgPos = CalParams::Right;

        float zoom = TQMIN((float)previewSize_/params.paperWidth,
                          (float)previewSize_/params.paperHeight);
        params.width  = (int)(params.paperHeight  * zoom);
        params.height = (int)(params.paperWidth   * zoom);
    }

    params.drawLines = checkBoxDrawLines_->isChecked();
    params.ratio     = sliderRatio_->value();
    params.baseFont  = TQFont(comboFont_->currentText());
    calWidget_->recreate();
}

}  // NameSpace KIPICalendarPlugin

#include "caltemplate.moc"
