/*
 * This file is part of the Polkit-tqt project
 * Copyright (C) 2009 Lukas Tinkl <ltinkl@redhat.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB. If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef POLKIT_TQT_IDENTITY_H
#define POLKIT_TQT_IDENTITY_H

#include "polkit-tqt-export.h"

#include <unistd.h>

typedef struct _PolkitIdentity  PolkitIdentity;
typedef struct _PolkitUnixUser  PolkitUnixUser;
typedef struct _PolkitUnixGroup PolkitUnixGroup;

class TQString;
template<typename> class TQValueList;


/**
 * \namespace PolkitTQt PolkitTQt
 *
 * \brief Namespace wrapping Polkit-TQt classes
 *
 * This namespace wraps all Polkit-TQt classes.
 */
namespace PolkitTQt
{

/**
 * \class Identity polkit-tqt-identity.h Identity
 * \author Lukas Tinkl <ltinkl@redhat.com>
 *
 * This class encapsulates the PolkitIdentity interface.
 *
 * \brief Abstract class representing identities
 *
 * \see UnixGroup
 * \see UnixUser
 */
class POLKIT_TQT_EXPORT Identity
{
  public:
    typedef TQValueList<Identity> List;

    Identity();
    Identity(const Identity &other);
    ~Identity();

    /**
     * Creates an Identity object from PolkitIdentity
     *
     * \warning It shouldn't be used directly unless you are completely aware of what are you doing
     *
     * \param pkIdentity PolkitIdentity object
     */
    explicit Identity(PolkitIdentity *pkIdentity);

    Identity& operator=(const Identity &other);

    /**
     * Check whether an identity is valid
     *
     * \return true is the identity is valid
     */
    bool isValid() const;

    /**
     * Serialization of object to the string
     *
     * \return Serialized Identity object
     */
    TQString toString() const;

    /**
     * Creates the Identity object from string representation
     *
     * \param string string representation of the object
     *
     * \return the new Identity instance
     */
    static Identity fromString(const TQString &string);

    /**
     * Gets PolkitIdentity object.
     *
     * \warning It shouldn't be used directly unless you are completely aware of what are you doing
     *
     * \return Pointer to PolkitIdentity instance
     */
    PolkitIdentity* identity() const;

    void setIdentity(PolkitIdentity *identity);

  private:
    class Data;
    Data *d;
};

/**
  * \class UnixUserIdentity polkit-tqt-identity.h Identity
  *
  * An object representing a user identity on a UNIX system.
  *
  * \brief UNIX user identity
  * \sa Identity
  */
class POLKIT_TQT_EXPORT UnixUserIdentity : public Identity
{
  public:
    /**
     * Creates UnixUser object by UID of the user
     *
     * \param uid user id
     */
    explicit UnixUserIdentity(uid_t uid);

    /**
     * Creates UnixUser object by unix name of the user
     *
     * \param name Unix name
     */
    explicit UnixUserIdentity(const TQString &name);

    /**
     * Creates UnixUser object from PolkitUnixUser object
     *
     * \warning Use this only if you are completely aware of what are you doing!
     *
     * \param pkUnixUser The PolkitUnixUser object
     */
    explicit UnixUserIdentity(PolkitUnixUser *pkUnixUser);

    /**
     * Gets an user id
     *
     * \return user id
     */
    uid_t uid() const;

    /**
     * Sets the id of user
     *
     * \param uid user id
     */
    void setUid(uid_t uid);
};

/**
  * \class UnixGroupIdentity polkit-tqt-identity.h Identity
  *
  * An object representing a group identity on a UNIX system.
  *
  * \brief UNIX group identity
  * \sa Identity
  */
class POLKIT_TQT_EXPORT UnixGroupIdentity : public Identity
{
  public:
    /**
     * Creates UnixGroup object by GID of the group
     *
     * \param gid group id
     */
    explicit UnixGroupIdentity(gid_t gid);

    /**
     * Creates UnixGroup object by unix name of the group
     *
     * \param name group name
     */
    explicit UnixGroupIdentity(const TQString &name);

    /**
     * Creates UnixGroup object from PolkitUnixGroup object
     *
     * \warning Use this only if you are completely aware of what are you doing!
     *
     * \param pkUnixGroup The PolkitUnixGroup object
     */
    explicit UnixGroupIdentity(PolkitUnixGroup *pkUnixGroup);

    /**
     * Gets a group id
     *
     * \return group id
     */
    gid_t gid() const;

    /**
     * Sets the id of group
     *
     * \param gid group id
     */
    void setGid(gid_t gid);
};

}

#endif
