;;; kaomel-utils.el --- Development utilities for kaomel -*- lexical-binding: t; -*-
;;
;; Author: Giovanni Crisalfi <giovanni.crisalfi@protonmail.com>
;; Maintainer: Giovanni Crisalfi <giovanni.crisalfi@protonmail.com>
;; Keywords: tools, development
;; Homepage: https://github.com/gicrisf/kaomel
;;
;; This file is not part of GNU Emacs.
;;
;;; Commentary:
;;
;; This file contains development utilities for the Kaomel package.
;; It provides functions to regenerate kaomel-data.el from kaomel-data.json.
;; 
;; This is a development-only tool and is not shipped with the MELPA release.
;;
;; Usage:
;;   Called programmatically to regenerate kaomel-data.el from JSON
;;
;; Functions:
;;   kaomel-dev-retrieve-kaomojis-from-json - Load JSON data from file
;;
;;; Code:

(require 'json)

(defun kaomel-dev-retrieve-kaomojis-from-json (path)
  "Retrieve kaomojis from JSON file at PATH.
Returns a vector of hash tables representing the kaomoji data."
  (with-temp-buffer
    (set-buffer-multibyte t)
    (insert-file-contents path)
    (goto-char (point-min))
    (let ((json-object-type 'hash-table))
      (json-read))))

(defun kaomel-dev--generate-data-file (json-path output-path)
  "Generate kaomel-data.el from JSON-PATH, writing to OUTPUT-PATH."
  (let* ((kaomoji-data (kaomel-dev-retrieve-kaomojis-from-json json-path))
         (current-time (current-time-string)))
    
    (with-temp-buffer
      (insert ";;; kaomel-data.el --- Vectorized kaomoji data -*- lexical-binding: t; -*-\n")
      (insert ";;\n")
      (insert ";; (ɔ) Copyleft 2023 Giovanni Crisalfi (gicrisf)\n")
      (insert ";;\n")
      (insert ";; Author: Giovanni Crisalfi <giovanni.crisalfi@protonmail.com>\n")
      (insert ";; Maintainer: Giovanni Crisalfi <giovanni.crisalfi@protonmail.com>\n")
      (insert (format ";; Generated: %s\n" current-time))
      (insert ";; Keywords: convenience extensions faces tools\n")
      (insert ";; Homepage: https://github.com/gicrisf/kaomel\n")
      (insert ";; Package-Requires: ((emacs \"27.1\"))\n")
      (insert ";;\n")
      (insert ";; This file is not part of GNU Emacs.\n")
      (insert ";;\n")
      (insert ";;; Commentary:\n")
      (insert ";;\n")
      (insert ";; This file contains the vectorized kaomoji data for the Kaomel package.\n")
      (insert ";; The data is stored as a Lisp vector for optimal performance.\n")
      (insert ";;\n")
      (insert ";; This file is automatically generated from kaomel-data.json.\n")
      (insert ";; Do not edit manually - use kaomel-dev-regenerate-data instead.\n")
      (insert ";;\n")
      (insert ";;; Code:\n\n")
      
      ;; Insert the data variable
      (insert "(defvar kaomel-data\n")
      (prin1 kaomoji-data (current-buffer))
      (insert "\n  \"Vector containing kaomoji data with multilingual tags.\")\n\n")
      
      (insert "(provide 'kaomel-data)\n")
      (insert ";;; kaomel-data.el ends here\n")
      
      ;; Write to file
      (write-file output-path))
    
    (message "Generated kaomel-data.el with %d kaomoji entries" (length kaomoji-data))))

(provide 'kaomel-utils)
;;; kaomel-utils.el ends here
