/*
    TiMidity++ -- MIDI to WAVE converter and player
    Copyright (C) 1999-2002 Masanao Izumo <mo@goice.co.jp>
    Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>
    ALSA 0.[56] support by Katsuhiro Ueno <katsu@blue.sky.or.jp>
                rewritten by Takashi Iwai <tiwai@suse.de>
    Alsa 4.0/5.1 channel output by Greg Lee <greg@ling.lll.hawaii.edu>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    alsa_a.c

    Functions to play sound on the ALSA audio driver

*/

#include "config.h"
#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>

#ifndef NO_STRING_H
#    include <string.h>
#else
#    include <strings.h>
#endif

#include <alsa/asoundlib.h>


#include "gtim.h"
#include "output.h"
#include "controls.h"

static int open_output (void);	/* 0=success, 1=warning, -1=fatal error */
static void close_output (void);

static void output_data (int32 *buf, uint32 nbytes);
static int driver_output_data (unsigned char *buf, uint32 count);
static void flush_output (void);
static void purge_output (void);
static int output_count (uint32 ct);
static int output_pause (int do_pause);



#if ALSA_LIB >= 5
static int detect (void);
#endif


/* export the playback mode */

#undef dpm
#define dpm alsa_play_mode

PlayMode dpm = {
    DEFAULT_RATE, PE_16BIT | PE_SIGNED,
    -1,
    {0},			/* default: get all the buffer fragments you can */
    "Alsa dsp device", 'a',
    "default",
    open_output,
    close_output,
    output_data,
    driver_output_data,
    flush_output,
    purge_output,
    output_count,
    output_pause
};


/*************************************************************************/
/* We currently only honor the PE_MONO bit, the sample rate, and the
   number of buffer fragments. We try 16-bit signed data first, and
   then 8-bit unsigned if it fails. If you have a sound device that
   can't handle either, let me know. */


/*ALSA PCM handler*/
static snd_pcm_t *handle = NULL;
static int total_bytes = -1;
static unsigned int frag_size = 0;
static int sample_shift = 0;
static unsigned int frame_size = 0;
static int output_counter;
static signed short *multichannel_buf = NULL;

static const char *
alsa_device_name (void)
{
    if (dpm.name && *dpm.name)
	return dpm.name;
    else
	return "default";
}

static void
error_report (int snd_error)
{
    ctl->cmsg (CMSG_ERROR, VERB_NORMAL, "%s: %s",
	       alsa_device_name (), snd_strerror (snd_error));
}


static void check_queue (void);

/*================================================================
 * ALSA API version 0.9.x
 *================================================================*/

/*
static void error_handle(const char *file, int line, const char *func, int err, const char *fmt, ...)
{
}
*/

static snd_pcm_uframes_t buffer_frame_size = 0;

static int ok_to_pause = 0;
static int output_is_paused = 0;

/*return value == 0 sucess
               == 1 warning
               == -1 fails
 */
static int
open_output (void)
{
    unsigned int orig_rate = dpm.rate;
    int     ret_val = 0;
    int     tmp, frags, pfds;
    unsigned int r;
    snd_pcm_hw_params_t *pinfo;
    snd_pcm_sw_params_t *swpinfo;

    snd_pcm_uframes_t buffer_size_min = 0;
    snd_pcm_uframes_t buffer_size_max = 0;
    snd_pcm_uframes_t period_size = 0;
    snd_pcm_uframes_t period_size_min;
    snd_pcm_uframes_t period_size_max;
    int     dir, err;

    /*dpm.name = get_pcm_name(); */

    snd_lib_error_set_handler (NULL);

    tmp = -1;
    if (!requested_ochannels || requested_ochannels > MAX_OUT_CHANNELS)
	requested_ochannels = MAX_OUT_CHANNELS;

    num_ochannels = 2;

    if ((dpm.encoding & PE_16BIT) && !(dpm.encoding & PE_24BIT) &&
	!(dpm.encoding & PE_MONO) && (dpm.encoding & PE_SIGNED)) {
	if (requested_ochannels == 6)
	    tmp =
		snd_pcm_open (&handle, "surround51", SND_PCM_STREAM_PLAYBACK,
			      SND_PCM_NONBLOCK);
	if (tmp >= 0)
	    num_ochannels = 6;
	else {
	    if (requested_ochannels >= 4)
		tmp =
		    snd_pcm_open (&handle, "surround40",
				  SND_PCM_STREAM_PLAYBACK, SND_PCM_NONBLOCK);
	    if (tmp >= 0)
		num_ochannels = 4;
	    else {
		tmp =
		    snd_pcm_open (&handle, dpm.name, SND_PCM_STREAM_PLAYBACK,
				  SND_PCM_NONBLOCK);
		if (tmp >= 0)
		    num_ochannels = 2;
	    }
	}
    }
    else
	tmp = snd_pcm_open (&handle, dpm.name, SND_PCM_STREAM_PLAYBACK, SND_PCM_NONBLOCK);	/* avoid blocking by open */

    if (tmp < 0) {
	ctl->cmsg (CMSG_ERROR, VERB_NORMAL, "Can't open pcm device '%s'.",
		   dpm.name);
	return -1;
    }

    if (num_ochannels == 6)
	dpm.name = "surround51";
    else if (num_ochannels == 4)
	dpm.name = "surround40";

    snd_pcm_nonblock (handle, 0);	/* set back to blocking mode */

    snd_pcm_hw_params_alloca (&pinfo);
    snd_pcm_sw_params_alloca (&swpinfo);

    if (snd_pcm_hw_params_any (handle, pinfo) < 0) {
	ctl->cmsg (CMSG_ERROR, VERB_NORMAL,
		   "ALSA pcm '%s' can't initialize hw_params",
		   alsa_device_name ());
	snd_pcm_close (handle);
	return -1;
    }

#ifdef LITTLE_ENDIAN
#    define S24_FORMAT	SND_PCM_FORMAT_S24_3LE
#    define S16_FORMAT	SND_PCM_FORMAT_S16_LE
#    define U16_FORMAT	SND_PCM_FORMAT_U16_LE
#else
#    define S24_FORMAT	SND_PCM_FORMAT_S24_3BE
#    define S16_FORMAT	SND_PCM_FORMAT_S16_BE
#    define U16_FORMAT	SND_PCM_FORMAT_U16_BE
#endif

    dpm.encoding &= ~(PE_ULAW | PE_ALAW | PE_BYTESWAP);

    /*check sample bit */
    if (snd_pcm_hw_params_test_format (handle, pinfo, S24_FORMAT) >= 0 &&
	snd_pcm_hw_params_set_format (handle, pinfo, S24_FORMAT) == 0) {
	dpm.encoding &= ~PE_16BIT;
	dpm.encoding |= PE_24BIT;
	dpm.encoding |= PE_SIGNED;
    }
    else {
	if (snd_pcm_hw_params_test_format (handle, pinfo, S16_FORMAT) < 0 &&
	    snd_pcm_hw_params_test_format (handle, pinfo, U16_FORMAT) < 0)
	    dpm.encoding &= ~PE_16BIT;	/*force 8bit samples */
	if (snd_pcm_hw_params_test_format (handle, pinfo, SND_PCM_FORMAT_U8) <
	    0
	    && snd_pcm_hw_params_test_format (handle, pinfo,
					      SND_PCM_FORMAT_S8) < 0)
	    dpm.encoding |= PE_16BIT;	/*force 16bit samples */
    }

    /*check format */
    if (dpm.encoding & PE_16BIT) {
	/*16bit */
	if (snd_pcm_hw_params_set_format (handle, pinfo, S16_FORMAT) == 0)
	    dpm.encoding |= PE_SIGNED;
	else if (snd_pcm_hw_params_set_format (handle, pinfo, U16_FORMAT) ==
		 0)
	    dpm.encoding &= ~PE_SIGNED;
	else {
	    ctl->cmsg (CMSG_ERROR, VERB_NORMAL,
		       "ALSA pcm '%s' doesn't support 16 bit sample width",
		       alsa_device_name ());
	    snd_pcm_close (handle);
	    return -1;
	}
    }
    else if (!(dpm.encoding & PE_24BIT)) {
	/*8bit */
	if (snd_pcm_hw_params_set_format (handle, pinfo, SND_PCM_FORMAT_U8) ==
	    0)
	    dpm.encoding &= ~PE_SIGNED;
	else if (snd_pcm_hw_params_set_format
		 (handle, pinfo, SND_PCM_FORMAT_S8) == 0)
	    dpm.encoding |= PE_SIGNED;
	else {
	    ctl->cmsg (CMSG_ERROR, VERB_NORMAL,
		       "ALSA pcm '%s' doesn't support 8 bit sample width",
		       alsa_device_name ());
	    snd_pcm_close (handle);
	    return -1;
	}
    }

    if (num_ochannels > 2 && !(dpm.encoding & PE_SIGNED))
	ctl->cmsg (CMSG_ERROR, VERB_NORMAL,
		   "ALSA 5.1/4.0 pcm '%s' needs signed format",
		   alsa_device_name ());

    if (snd_pcm_hw_params_set_access (handle, pinfo,
				      SND_PCM_ACCESS_RW_INTERLEAVED) < 0) {
	ctl->cmsg (CMSG_ERROR, VERB_NORMAL,
		   "ALSA pcm '%s' doesn't support interleaved data",
		   alsa_device_name ());
	snd_pcm_close (handle);
	return -1;
    }

    /*check rate */
    dir = 0;
    err = snd_pcm_hw_params_get_rate_min (pinfo, &r, &dir);
    dir = 0;
    if ( /*r >= 0 && */ r > dpm.rate) {
	dpm.rate = r;
	ret_val = 1;
    }
    err = snd_pcm_hw_params_get_rate_max (pinfo, &r, &dir);
    dir = 0;
    if ( /* r >= 0 && */ r < dpm.rate) {
	dpm.rate = r;
	ret_val = 1;
    }
    err = snd_pcm_hw_params_set_rate_near (handle, pinfo, &dpm.rate, &dir);
    dir = 0;
    if (err < 0) {
	ctl->cmsg (CMSG_ERROR, VERB_NORMAL,
		   "ALSA pcm '%s' can't set rate %d",
		   alsa_device_name (), dpm.rate);
	snd_pcm_close (handle);
	return -1;
    }

    /*check channels */
    if (dpm.encoding & PE_MONO) {
	if (snd_pcm_hw_params_test_channels (handle, pinfo, 1) < 0)
	    dpm.encoding &= ~PE_MONO;
    }
    else {
	if (snd_pcm_hw_params_test_channels (handle, pinfo, num_ochannels) <
	    0)
	    dpm.encoding |= PE_MONO;
    }

    if (dpm.encoding & PE_MONO) {
	num_ochannels = 1;
	if (snd_pcm_hw_params_set_channels (handle, pinfo, 1) < 0) {
	    ctl->cmsg (CMSG_ERROR, VERB_NORMAL,
		       "ALSA pcm '%s' can't set mono channel",
		       alsa_device_name ());
	    snd_pcm_close (handle);
	    return -1;
	}
    }
    else {
	if (snd_pcm_hw_params_set_channels (handle, pinfo, num_ochannels) < 0) {
	    ctl->cmsg (CMSG_ERROR, VERB_NORMAL,
		       "ALSA pcm '%s' can't set %d channel output",
		       alsa_device_name (), num_ochannels);
	    snd_pcm_close (handle);
	    return -1;
	}
    }

    frame_size = num_ochannels;
    if (dpm.encoding & PE_16BIT)
	frame_size = 2 * num_ochannels;
    if (dpm.encoding & PE_24BIT)
	frame_size = 3 * num_ochannels;

    /* Set buffer fragment size (in extra_param[1]) */
    if (dpm.extra_param[1] != 0)
	frag_size = dpm.extra_param[1];
    else
	frag_size = AUDIO_BUFFER_SIZE * frame_size;

    err = snd_pcm_hw_params_get_buffer_size_min (pinfo, &buffer_size_min);
    err = snd_pcm_hw_params_get_buffer_size_max (pinfo, &buffer_size_max);

    /* Set buffer fragments (in extra_param[0]) */
    if (dpm.extra_param[0] == 0)
	frags = buffer_size_max / AUDIO_BUFFER_SIZE;
    else
	frags = dpm.extra_param[0];

    /* (not real total for 4/6 channels) */
    total_bytes = frag_size * frags;
    buffer_frame_size = total_bytes / frame_size;

    dir = 0;
    err =
	snd_pcm_hw_params_get_period_size_min (pinfo, &period_size_min, &dir);
    dir = 0;
    err =
	snd_pcm_hw_params_get_period_size_max (pinfo, &period_size_max, &dir);

    if (buffer_size_max < buffer_frame_size)
	buffer_frame_size = buffer_size_max;
    if (buffer_size_min > buffer_frame_size)
	buffer_frame_size = buffer_size_min;

    period_size = buffer_frame_size / frags;
    if (period_size_max < period_size)
	period_size = period_size_max;
    if (period_size_min > period_size)
	period_size = period_size_min;
    buffer_frame_size = period_size * frags;
/*
  ctl->cmsg(CMSG_INFO, VERB_VERBOSE,
	    "Requested buffer size %d (%d frames X %d frame size), fragment size %d",
	    buffer_frame_size * frame_size,
	    buffer_frame_size, frame_size, frag_size);
*/
    if ((tmp =
	 snd_pcm_hw_params_set_buffer_size_near (handle, pinfo,
						 &buffer_frame_size)) < 0) {
	ctl->cmsg (CMSG_WARNING, VERB_NORMAL,
		   "ALSA pcm '%s' can't set buffer size %d",
		   alsa_device_name (), total_bytes);
	snd_pcm_close (handle);
	return -1;
    }

    if ((tmp =
	 snd_pcm_hw_params_set_period_size_near (handle, pinfo, &period_size,
						 &dir)) < 0) {
	ctl->cmsg (CMSG_WARNING, VERB_NORMAL,
		   "ALSA pcm '%s' can't set period size %d",
		   alsa_device_name (), period_size);
	snd_pcm_close (handle);
	return -1;
    }
    dir = 0;

    if (snd_pcm_hw_params (handle, pinfo) < 0) {
	snd_output_t *alsa_log;
	ctl->cmsg (CMSG_WARNING, VERB_NORMAL,
		   "ALSA pcm '%s' can't set hw_params", alsa_device_name ());
	snd_output_stdio_attach (&alsa_log, stderr, 0);
	snd_pcm_hw_params_dump (pinfo, alsa_log);
	snd_pcm_close (handle);
	return -1;
    }

    err = snd_pcm_hw_params_get_buffer_size (pinfo, &buffer_frame_size);
    total_bytes = buffer_frame_size * frame_size;
    err = snd_pcm_hw_params_get_period_size (pinfo, &period_size, &dir);
    dir = 0;
    frag_size = period_size * frame_size;
    ctl->cmsg (CMSG_INFO, VERB_VERBOSE,
	       "ALSA pcm '%s' buffer size %d, %d fragments, period size %d frames, rate %d",
	       alsa_device_name (), buffer_frame_size * frame_size,
	       frags, period_size, dpm.rate);
    snd_pcm_hw_params_get_rate (pinfo, &tmp, &dir);
    dir = 0;
    if (tmp > 0 && tmp != (int) dpm.rate) {
	dpm.rate = (unsigned int) tmp;
	ret_val = 1;
    }
    if (orig_rate != dpm.rate) {
	ctl->cmsg (CMSG_WARNING, VERB_VERBOSE,
		   "Output rate adjusted to %d Hz (requested %d Hz)",
		   dpm.rate, orig_rate);
    }

    ok_to_pause = snd_pcm_hw_params_can_pause(pinfo);
/*
 int snd_pcm_pause  	(   	snd_pcm_t *   	  pcm,
		int  	  enable
	)  	
  	

Pause/resume PCM.

Parameters:
    pcm 	PCM handle
    pause 	0 = resume, 1 = pause

*/

    snd_pcm_sw_params_current (handle, swpinfo);
    //snd_pcm_sw_params_set_start_threshold (handle, swpinfo, frag_size);
    snd_pcm_sw_params_set_start_threshold (handle, swpinfo, buffer_frame_size);
    snd_pcm_sw_params_set_stop_threshold (handle, swpinfo, frag_size);

    tmp = snd_pcm_prepare (handle);
    if (tmp < 0) {
	ctl->cmsg (CMSG_WARNING, VERB_NORMAL, "unable to prepare channel\n");
	snd_pcm_close (handle);
	return -1;
    }

    pfds = snd_pcm_poll_descriptors_count (handle);
    if (pfds > 1) {
	ctl->cmsg (CMSG_ERROR, VERB_NORMAL, "too many poll descriptors: %s",
		   alsa_device_name ());
	close_output ();
	return -1;
    }
    else if (pfds == 1) {
	struct pollfd pfd;
	if (snd_pcm_poll_descriptors (handle, &pfd, 1) >= 0)
	    dpm.fd = pfd.fd;
	else
	    dpm.fd = -1;
    }
    else
	dpm.fd = -1;

    output_counter = 0;

/*
  ctl->cmsg(CMSG_INFO, VERB_VERBOSE,
	      "Alsa %d channel output",
	      num_ochannels);
*/
    return ret_val;
}

static void
close_output (void)
{
    int     ret;

    if (handle) {
	snd_pcm_drain (handle);
	ret = snd_pcm_close (handle);
	if (ret < 0)
	    error_report (ret);
	handle = NULL;
    }
    if (multichannel_buf)
	free (multichannel_buf);
    multichannel_buf = NULL;
    dpm.fd = -1;
}

#define C_FLEFT 0
#define C_FRIGHT 1
#define C_RLEFT 2
#define C_RRIGHT 3
#define C_CNTR 4
#define C_LFE 5

static signed short *
more_channels (signed short *buf, unsigned int nframes)
{
    unsigned int     i, frame, s_frame;
    signed short l, r, c;
    frame = s_frame = 0;

    if (!multichannel_buf)
	multichannel_buf =
	    (signed short *) malloc (AUDIO_BUFFER_SIZE * frame_size);

    for (i = 0; i < nframes; i++, frame += num_ochannels, s_frame += 2) {
	l = LE_SHORT (buf[s_frame + C_FLEFT]);
	r = LE_SHORT (buf[s_frame + C_FRIGHT]);
	c = (l / 2) + (r / 2);

	multichannel_buf[frame + C_FLEFT] = LE_SHORT (l);
	multichannel_buf[frame + C_FRIGHT] = LE_SHORT (r);

	if (num_ochannels >= 4) {
	    multichannel_buf[frame + C_RLEFT] = LE_SHORT ((l - c) / 2);
	    multichannel_buf[frame + C_RRIGHT] = LE_SHORT ((r - c) / 2);
	}
	if (num_ochannels == 6) {
	    multichannel_buf[frame + C_CNTR] = LE_SHORT (c);
	    multichannel_buf[frame + C_LFE] = 0;
	}
    }
    return (multichannel_buf);
}

static int
driver_output_data (unsigned char *buf, uint32 nbytes)
{
    int     n;
    unsigned int     nframes;
    if (!handle)
	return -1;

    nframes = nbytes / frame_size;

    if (num_ochannels != num_ochannels_in_buf)
	fprintf (stderr, "output mismatch\n");

    if (num_ochannels > 2 && num_ochannels_in_buf == 2
	&& (dpm.encoding & PE_16BIT))
	buf = (unsigned char *) more_channels ((signed short *) buf, nframes);
    else if (num_ochannels != num_ochannels_in_buf)
	return -1;

    while (nframes > 0) {
	n = snd_pcm_writei (handle, buf, nframes);
	if (n == -EAGAIN || (n >= 0 && n < (int)nframes)) {
	    check_for_rc();
	    snd_pcm_wait (handle, 1000);
	}
	else if (n == -EPIPE) {
	    snd_pcm_status_t *status;
	    snd_pcm_status_alloca (&status);
	    if (snd_pcm_status (handle, status) < 0) {
		ctl->cmsg (CMSG_WARNING, VERB_DEBUG, "%s: cannot get status",
			   alsa_device_name ());
		return -1;
	    }
	    danger += 2000;
	    ctl->cmsg (CMSG_INFO, VERB_DEBUG,
		       "%s: underrun at %ld", alsa_device_name (),
		       output_counter << sample_shift);
	    snd_pcm_prepare (handle);
	}
	else if (n < 0) {
	    ctl->cmsg (CMSG_WARNING, VERB_DEBUG,
		       "%s: %s", alsa_device_name (),
		       (n < 0) ? snd_strerror (n) : "write error");
	    return -1;
	}
	if (n > 0) {
	    nframes -= n;
	    /* buf += n << shift; */
	    buf += n * frame_size;
	    output_counter += n;
	}
    }

    return 0;
}


static void
output_data (int32 *buf, uint32 count)
{
    int     ocount;

    //check_queue ();

    count *= num_ochannels;	/* multichannel samples */
    ocount = (int) count;

    if (ocount) {
	if (dpm.encoding & PE_16BIT) {
	    /* Convert data to signed 16-bit PCM */
	    s32tos16 (buf, count);
	    ocount *= 2;
	}
	else if (dpm.encoding & PE_24BIT) {
	    /* Convert data to signed 24-bit PCM */
	    s32tos24 (buf, count);
	    ocount *= 3;
	}
	else {
	    /* Convert to 8-bit unsigned and write out. */
	    s32tou8 (buf, count);
	}
    }

    driver_output_data ((unsigned char *) buf, (uint32) ocount);

    check_queue ();
}

/*
 delay in samples - snd_pcm_status_get_delay()

The function snd_pcm_avail_update() updates the current available count of
   samples for writing (playback)

The function snd_pcm_delay() returns the delay in samples. For playback, it means
   count of samples in the ring buffer before the next sample will be sent to DAC.
    Note that this function does not update the
   current r/w pointer for applications, so the function snd_pcm_avail_update() must
   be called afterwards before any read/write begin+commit operations.
*/

static int
alsa_delay (void)
{
    int     ret;
    snd_pcm_sframes_t delayp;
    ret = snd_pcm_delay (handle, &delayp);
    if (ret >= 0)
	return delayp;
    else
	return 0;
}

#define PRESUMED_FULLNESS 20

#ifdef WATCH_BUFFER
static int watch_buffer = 20;
#endif

static int starting_up = 6;

static void
check_queue (void)
{
    int     samples_queued;

    if (starting_up > 0) {
	starting_up--;
	output_buffer_full = PRESUMED_FULLNESS;
	return;
    }
    samples_queued = alsa_delay ();
    /* samples_queued *= 4; */

    if (!samples_queued)
	output_buffer_full = PRESUMED_FULLNESS;
    else
	output_buffer_full =
	    samples_queued * 100 / buffer_frame_size;

#ifdef WATCH_BUFFER
    watch_buffer--;
    if (watch_buffer < 0) {
	fprintf (stderr, "-%d", output_buffer_full);
	watch_buffer = 20;
    }
#endif
}



static void
flush_output (void)
{
    int     tmp;
    snd_pcm_drain (handle);
    tmp = snd_pcm_prepare (handle);
    if (tmp < 0) {
	ctl->cmsg (CMSG_WARNING, VERB_NORMAL,
		   "flush: unable to prepare channel (%d)\n", tmp);
	snd_pcm_close (handle);
	dpm.fd = -1;
	exit (1);
    }
    snd_pcm_reset (handle);
    output_counter = 0;
    output_buffer_full = PRESUMED_FULLNESS;
    danger = 0;
    starting_up = 6;
}
static void
purge_output (void)
{
    int     tmp;
    snd_pcm_drain (handle);
    tmp = snd_pcm_prepare (handle);
    if (tmp < 0) {
	ctl->cmsg (CMSG_WARNING, VERB_NORMAL,
		   "purge: unable to prepare channel (%d)\n", tmp);
	snd_pcm_close (handle);
	dpm.fd = -1;
	exit (1);
    }
    snd_pcm_reset (handle);
    output_counter = 0;
    output_buffer_full = PRESUMED_FULLNESS;
    danger = 0;
    starting_up = 6;
}
static int
output_count (uint32 ct)
{
    int ret_count = ct;
    snd_pcm_sframes_t delayp;

    ret_count = output_counter;

/*int snd_pcm_delay(snd_pcm_t *pcm, snd_pcm_sframes_t *delayp) */
    if ( snd_pcm_delay(handle, &delayp) == 0 ) {
	    ret_count -= delayp;
    }

    return ret_count;
}
static int
output_pause (int do_pause)
{
    int err;

    if (!ok_to_pause) return 0;
    if (do_pause && output_is_paused) return 1;
    if (!do_pause && !output_is_paused) return 1;
    if (do_pause) {
    	err = snd_pcm_pause(handle, 1);
	ctl->cmsg (CMSG_ERROR, VERB_NORMAL, "Paused ret = %d.",
		   err);
	if (!err) return 1;
	return 0;
    }
    if (!do_pause) {
    err = snd_pcm_prepare (handle);
    if (err < 0) {
	ctl->cmsg (CMSG_WARNING, VERB_NORMAL, "unable to prepare channel\n");
	return 0;
    }
    	err = snd_pcm_pause(handle, 0);
	ctl->cmsg (CMSG_ERROR, VERB_NORMAL, "Resumed ret = %d.",
		   err);
	if (!err) return 1;
	return 0;
    }
}
