/*=============================================================================

    This file is part of FLINT.

    FLINT is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    FLINT is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with FLINT; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

=============================================================================*/
/******************************************************************************

    Copyright (C) 2009 William Hart
    Copyright (C) 2010 Sebastian Pancratz

******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <gmp.h>
#include "flint.h"
#include "fmpz.h"
#include "fmpq_poly.h"
#include "ulong_extras.h"

int
main(void)
{
    int i, result;
    ulong cflags = UWORD(0);

    FLINT_TEST_INIT(state);

    flint_printf("rem_powers_precomp....");
    fflush(stdout);

    /* Check aliasing of q and a */
    for (i = 0; i < 50 * flint_test_multiplier(); i++)
    {
        fmpq_poly_t a, b, r;
        fmpq_poly_powers_precomp_t binv;

        fmpq_poly_init(a);
        fmpq_poly_init(b);
        fmpq_poly_init(r);
        fmpq_poly_randtest(a, state, n_randint(state, 50), 100);
        fmpq_poly_randtest_not_zero(b, state, n_randint(state, 50) + 1, 100);

        fmpq_poly_powers_precompute(binv, b);
        
        fmpq_poly_rem_powers_precomp(r, a, b, binv);
        fmpq_poly_rem_powers_precomp(a, a, b, binv);

        result = (fmpq_poly_equal(r, a));
        if (!result)
        {
            flint_printf("FAIL:\n");
            flint_printf("r = "), fmpq_poly_debug(r), flint_printf("\n\n");
            flint_printf("a = "), fmpq_poly_debug(a), flint_printf("\n\n");
            flint_printf("b = "), fmpq_poly_debug(b), flint_printf("\n\n");
            flint_printf("cflags = %wu\n\n", cflags);
            abort();
        }

        fmpq_poly_powers_clear(binv);

        fmpq_poly_clear(a);
        fmpq_poly_clear(b);
        fmpq_poly_clear(r);
    }
    
    /* Check aliasing of q and b */
    for (i = 0; i < 50 * flint_test_multiplier(); i++)
    {
        fmpq_poly_t a, b, r;
        fmpq_poly_powers_precomp_t binv;

        fmpq_poly_init(a);
        fmpq_poly_init(b);
        fmpq_poly_init(r);
        fmpq_poly_randtest(a, state, n_randint(state, 50), 100);
        fmpq_poly_randtest_not_zero(b, state, n_randint(state, 50) + 1, 100);

        fmpq_poly_powers_precompute(binv, b);
        
        fmpq_poly_rem_powers_precomp(r, a, b, binv);
        fmpq_poly_rem_powers_precomp(b, a, b, binv);

        result = (fmpq_poly_equal(r, b));
        if (!result)
        {
            flint_printf("FAIL:\n");
            flint_printf("r = "), fmpq_poly_debug(r), flint_printf("\n\n");
            flint_printf("a = "), fmpq_poly_debug(a), flint_printf("\n\n");
            flint_printf("b = "), fmpq_poly_debug(b), flint_printf("\n\n");
            flint_printf("cflags = %wu\n\n", cflags);
            abort();
        }

        fmpq_poly_powers_clear(binv);

        fmpq_poly_clear(a);
        fmpq_poly_clear(b);
        fmpq_poly_clear(r);
    }

    /* Compare with divrem */
    for (i = 0; i < 50 * flint_test_multiplier(); i++)
    {
        fmpq_poly_t a, b, q, r2, r;
        fmpq_poly_powers_precomp_t binv;

        fmpq_poly_init(a);
        fmpq_poly_init(b);
        fmpq_poly_init(q);
        fmpq_poly_init(r2);
        fmpq_poly_init(r);
        fmpq_poly_randtest(a, state, n_randint(state, 50), 100);
        fmpq_poly_randtest_not_zero(b, state, n_randint(state, 50) + 1, 100);

        fmpq_poly_powers_precompute(binv, b);
        
        fmpq_poly_divrem(q, r, a, b);
        fmpq_poly_rem_powers_precomp(r2, a, b, binv);
        fmpq_poly_canonicalise(r2);

        result = (fmpq_poly_equal(r, r2));
        if (!result)
        {
            flint_printf("FAIL:\n");
            flint_printf("a  = "), fmpq_poly_debug(a), flint_printf("\n\n");
            flint_printf("b  = "), fmpq_poly_debug(b), flint_printf("\n\n");
            flint_printf("q  = "), fmpq_poly_debug(q), flint_printf("\n\n");
            flint_printf("r  = "), fmpq_poly_debug(r), flint_printf("\n\n");
            flint_printf("r2 = "), fmpq_poly_debug(r2), flint_printf("\n\n");
            flint_printf("cflags = %wu\n\n", cflags);
            abort();
        }

        fmpq_poly_powers_clear(binv);

        fmpq_poly_clear(a);
        fmpq_poly_clear(b);
        fmpq_poly_clear(q);
        fmpq_poly_clear(r2);
        fmpq_poly_clear(r);
    }

    FLINT_TEST_CLEANUP(state);
    
    flint_printf("PASS\n");
    return 0;
}
