;;; slurpbarf.el --- Commands for slurping and barfing  -*- lexical-binding: t; -*-

;; Copyright (C) 2023-2025 Vili Aapro

;; Author: Vili Aapro
;; Keywords: convenience, lisp, XML
;; Package-Requires: ((emacs "29.1"))
;; Package-Version: 20251018.2247
;; Package-Revision: 20f209035ab0
;; URL: https://codeberg.org/vilij/slurpbarf-elcute

;; This file is part of Slurpbarf.
;;
;; Slurpbarf is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published
;; by the Free Software Foundation; either version 3 of the License,
;; or (at your option) any later version.
;;
;; Slurpbarf is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with Slurpbarf.  If not, see <https://www.gnu.org/licenses/>.

;;; Commentary:

;; Slurpbarf is a library defining the minor mode `slurpbarf-mode' for
;; slurping and barfing, or ingesting and emitting, expressions in the
;; spirit of Paredit.

;; Supported major modes are Lisp Data mode and nXML mode; however,
;; the commands should mostly work also in C mode and others: a global
;; minor mode, `global-slurpbarf-mode', is provided.

;; A command for splicing expressions, `slurpbarf-splice', is included
;; but not bound to any key.

;;; Code:

(require 'cl-lib)
(require 'nxml-mode)

(defmacro slurpbarf--excurse (&rest body)
  (declare (indent 0))
  `(save-excursion
     ,@body
     (point)))

(defmacro slurpbarf--silence-errors (interactive &rest body)
  (declare (indent 1))
  `(if ,interactive
       (condition-case nil
	   (progn ,@body)
	 (error nil))
     ,@body))

(defmacro slurpbarf--nxml-motion (interactive &rest body)
  "Tame nXML-mode motion by skipping spaces and translating errors.
If INTERACTIVE is non-nil, translate errors raised inside BODY
into user errors."
  (declare (indent 1))
  `(progn
     (if ,interactive
	 (condition-case err (progn ,@body)
	   (error (signal 'user-error (cdr err))))
       ,@body)
     (slurpbarf--skip-blanks-and-newline)))

(defvar slurpbarf-init-alist
  `((nxml-mode ,#'slurpbarf--nxml-init))
  "Associates major modes with lists of initialization functions.")

(defun slurpbarf--init (alist)
  (dolist (pair alist)
    (cl-destructuring-bind (mode . functions) pair
      (when (derived-mode-p mode)
	(dolist (f functions)
	  (funcall f))))))

(defun slurpbarf--nxml-init ()
  (setq-local
   slurpbarf-insert-space-flag nil
   slurpbarf-skip-comments-flag nil
   slurpbarf-up-function #'slurpbarf--nxml-up
   slurpbarf-down-function #'slurpbarf--nxml-down
   slurpbarf-forward-function #'slurpbarf--nxml-forward))

;;;###autoload
(define-minor-mode slurpbarf-mode
  "Toggle slurping and barfing (Slurpbarf mode) in the current buffer.

When Slurpbarf mode is enabled, various buffer-local variables
are set according to major mode, affecting the commands bound in
the mode map, and the command `slurpbarf-splice'.

Supported major modes are Lisp Data mode and nXML mode; in
addition, the default Lisp Data behavior may be useful globally:
see `global-slurpbarf-mode'."
  :keymap (define-keymap
	    "C-)" #'slurpbarf-slurp-forward
	    "C-(" #'slurpbarf-slurp-backward
	    "C-}" #'slurpbarf-barf-forward
	    "C-{" #'slurpbarf-barf-backward)
  (slurpbarf--init slurpbarf-init-alist))

;;;###autoload
(define-globalized-minor-mode global-slurpbarf-mode
  slurpbarf-mode slurpbarf-mode
  :group 'convenience)

(defvar slurpbarf-insert-space-flag t
  "Non-nil means insert a space between words and symbols.")
(defvar slurpbarf-skip-comments-flag t
  "Non-nil means skip comments when barfing.")
(defvar slurpbarf-up-function #'slurpbarf--lisp-up
  "Goes up expression hierarchy.")
(defvar slurpbarf-down-function #'down-list
  "Goes down expression hierarchy.")
(defvar slurpbarf-forward-function #'slurpbarf--lisp-forward
  "Moves forward expressions laterally.")

(defun slurpbarf--up (n &optional interactive)
  "Go up |N| levels of expression hierarchy.
With positive argument, move forward; with negative, backward.
If INTERACTIVE is non-nil, report errors as appropriate for
interactive usage."
  (funcall slurpbarf-up-function n interactive))

(defun slurpbarf--down (n &optional interactive)
  "Go down |N| levels of expression hierarchy.
With positive argument, move forward; with negative, backward.
If INTERACTIVE is non-nil, report errors as appropriate for
interactive usage."
  (funcall slurpbarf-down-function n interactive))

(defun slurpbarf--beware-unterminated-comment (interactive)
  "Check that point is not at end of buffer inside a comment.
Return t if it is not.  Otherwise, if INTERACTIVE is non-nil,
return nil; if nil, signal an error."
  (if (and (eq (point) (point-max))
	   (eq (syntax-ppss-context (syntax-ppss)) 'comment))
      (if interactive
	  nil
	(error "Unterminated comment"))
    t))

(defun slurpbarf--forward (n &optional interactive)
  "Move forward N expressions or as far as we can.
With negative argument, move backward.  If INTERACTIVE is
non-nil, handle errors silently.
Return signed number of expressions moved: positive forward,
negative backward."
  (let ((sign (cl-signum n))
	(i 0)
	(pos (point)))
    (while (and
	    (/= i n)
	    (progn
	      (slurpbarf--silence-errors interactive
		(funcall slurpbarf-forward-function sign))
	      (/= pos (point)))
	    (slurpbarf--beware-unterminated-comment interactive))
      (cl-incf i sign)
      (setq pos (point)))
    i))

(defun slurpbarf--break-out-string ()
  (let ((syntax (syntax-ppss)))
    (when (eq (syntax-ppss-context syntax) 'string)
      (let ((string-start (nth 8 syntax)))
	(goto-char string-start)))))

(defun slurpbarf--lisp-up (n interactive)
  "Move up N levels of expressions in Lisp Data.
Treat strings as atoms and include prefix characters.  If
INTERACTIVE is non-nil, report errors as appropriate for
interactive usage."
  (when (/= n 0)
    (let ((pos
	   (slurpbarf--excurse
	     (slurpbarf--break-out-string)
	     (up-list n nil interactive)
	     (backward-prefix-chars))))
      (goto-char pos))))

(defun slurpbarf--lisp-forward (n)
  "Move forward N expressions in Lisp Data.
Behave better than `forward-sexp' at beginning and end of buffer
by not jumping there over comments even while there's no sexp to
be found."
  (let ((pos (scan-sexps (point) n)))
    (if (not pos) pos
      (goto-char pos))))

(defun slurpbarf--skip-blanks-and-newline ()
  "Skip blanks and a newline.
For stylistic reasons, we consider such trailing whitespace an
inseparable part of an XML tag."
  (skip-chars-forward "[:blank:]")
  (skip-chars-forward "\n" (1+ (point))))

(defun slurpbarf--nxml-up (n interactive)
  (slurpbarf--nxml-motion interactive
    (nxml-up-element n)))

(defun slurpbarf--nxml-down (n interactive)
  (slurpbarf--nxml-motion interactive
    (nxml-down-element n)))

(defun slurpbarf--nxml-forward (n)
  (nxml-forward-balanced-item n)
  (slurpbarf--skip-blanks-and-newline))

(defun slurpbarf--indent-p ()
  (and electric-indent-mode
       (not electric-indent-inhibit)))

(defun slurpbarf--indent (n)
  "Indent N levels of containing expressions.
Restrict indentation to field at point.  Do nothing if Electric
Indent is disabled."
  (when (slurpbarf--indent-p)
    (with-restriction (field-beginning) (field-end)
      (cl-labels
	  ((excurse (n extreme)
	     (slurpbarf--excurse
	       (condition-case nil
		   (slurpbarf--up n)
		 (error (goto-char extreme))))))
	(let ((left (excurse (- n) (point-min)))
	      (right (excurse n (point-max))))
	  (indent-region left right))))))

(defun slurpbarf--unindent (&optional pos)
  "Delete horizontal whitespace after POS if POS starts a line.
If POS is nil, use point instead.  Do nothing if Electric Indent is
disabled."
  (when (slurpbarf--indent-p)
    (save-excursion
      (when pos
	(goto-char pos))
      (when (bolp)
	(let ((origin (point)))
	  (skip-chars-forward "[:blank:]")
	  (delete-region origin (point)))))))

(defconst slurpbarf--word-or-symbol-class (list 2 3))

(defun slurpbarf--insert-space ()
  "Insert a space between words and symbols."
  (when
      (and
       slurpbarf-insert-space-flag
       (let ((before (syntax-class (syntax-after (1- (point)))))
	     (after (syntax-class (syntax-after (point)))))
	 (and (memq before slurpbarf--word-or-symbol-class)
	      (memq after slurpbarf--word-or-symbol-class))))
    (insert " ")))

(defun slurpbarf--insert (string)
  (slurpbarf--unindent (point))
  (save-excursion
    (let ((origin (point)))
      (insert string)
      (slurpbarf--insert-space)
      (goto-char origin)
      (slurpbarf--insert-space))))

(defun slurpbarf--extract-region (beg end)
  (let ((beg (min beg end))
	(end (max beg end)))
    (slurpbarf--unindent end)
    (let ((end-margin (- (point-max) end)))
      (slurpbarf--unindent beg)
      (let ((end (- (point-max) end-margin)))
	(delete-and-extract-region beg end)))))

(defun slurpbarf--skip-comments (sign)
  (when slurpbarf-skip-comments-flag
    (forward-comment (* sign (buffer-size)))))

(defun slurpbarf-slurp-forward (n &optional interactive)
  "Slurp or ingest forward N expressions.
With negative argument, slurp backward.  If INTERACTIVE is
non-nil, as it is interactively, report errors as appropriate for
this kind of usage.

 ((foo | bar) baz)
->
 ((foo | bar baz))"
  (interactive "p\nd")
  (let ((sign (cl-signum n)))
    (save-excursion
      (slurpbarf--up sign interactive)
      (let ((origin (point))
	    (offset
	     (- (slurpbarf--excurse
		  (slurpbarf--down (- sign) interactive))
		(point))))
	(when (/= (slurpbarf--forward n interactive) 0)
	  (let ((substring (slurpbarf--extract-region origin (point))))
	    (forward-char offset)
	    (slurpbarf--insert substring))
	  (slurpbarf--indent 2))))))

(defun slurpbarf-slurp-backward (n &optional interactive)
  "Slurp or ingest backward N expressions.
With negative argument, slurp forward.  If INTERACTIVE is
non-nil, as it is interactively, report errors as appropriate for
this kind of usage.

 (foo (bar | baz))
->
 ((foo bar | baz))"
  (interactive "p\nd")
  (slurpbarf-slurp-forward (- n) interactive))

(defun slurpbarf-barf-forward (n &optional interactive)
  "Barf or emit forward N expressions.
With negative argument, barf backward.  If INTERACTIVE is
non-nil, as it is interactively, report errors as appropriate for
this kind of usage.

 ((foo | bar baz))
->
 ((foo | bar) baz)"
  (interactive "p\nd")
  (let ((sign (cl-signum n)))
    (save-excursion
      (slurpbarf--up sign interactive)
      (let ((offset
	     (- (slurpbarf--excurse
		  (slurpbarf--down (- sign) interactive))
		(point))))
	(forward-char offset)
	(let ((origin (point)))
	  (when (/= (slurpbarf--forward (- n) interactive) 0)
	    (slurpbarf--skip-comments (- sign))
	    (let ((substring (slurpbarf--extract-region origin (point))))
	      (backward-char offset)
	      (slurpbarf--insert substring))
	    (slurpbarf--indent 2)))))))

(defun slurpbarf-barf-backward (n &optional interactive)
  "Barf or emit backward N expressions.
With negative argument, barf forward.  If INTERACTIVE is non-nil,
as it is interactively, report errors as appropriate for this
kind of usage.

 ((foo bar | baz))
->
 (foo (bar | baz))"
  (interactive "p\nd")
  (slurpbarf-barf-forward (- n) interactive))

(defun slurpbarf-splice (&optional interactive)
  "Splice expression at point into containing expression.
If INTERACTIVE is non-nil, as it is interactively, report errors
as appropriate for this kind of usage.

 ((foo | bar))
->
 (foo | bar)"
  (interactive "d")
  (save-excursion
    (let* ((out0 (slurpbarf--excurse (slurpbarf--up -1 interactive)))
	   (out1 (slurpbarf--excurse (slurpbarf--up +1 interactive)))
	   (in0 (slurpbarf--excurse
		  (goto-char out0)
		  (slurpbarf--down +1 interactive)))
	   (in1 (slurpbarf--excurse
		  (goto-char out1)
		  (slurpbarf--down -1 interactive))))
      (goto-char out1)
      (slurpbarf--unindent)
      (delete-region in1 out1)
      (slurpbarf--insert-space)
      (goto-char in0)
      (slurpbarf--unindent)
      (delete-region in0 out0)
      (slurpbarf--insert-space)))
  (slurpbarf--indent 1))

(provide 'slurpbarf)

;;; slurpbarf.el ends here
