/* ncurses-gui.c
 * Implementation for ncurses GUI
 *
 * Yersinia
 * By David Barroso <tomac@wasahero.org> and Alfredo Andres <slay@wasahero.org>
 * Copyright 2005 Alfredo Andres and David Barroso
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#ifndef lint
static const char rcsid[] =
"$Id: ncurses-gui.c,v 1.10 2005/08/01 15:00:58 t0mac Exp $";
#endif

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#define _REENTRANT

#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>       

#ifdef HAVE_NETINET_IN_SYSTM_H
#include <netinet/in_systm.h>
#else
#ifdef HAVE_NETINET_IN_SYSTEM_H
#include <netinet/in_system.h>
#endif
#endif

#include <netinet/in.h>
#include <arpa/inet.h>
#include <signal.h>
#include <time.h>

#ifdef TIME_WITH_SYS_TIME
#include <sys/time.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif

#ifdef HAVE_BSTRING_H
#include <bstring.h>
#endif

#ifdef STDC_HEADERS
#include <stdlib.h>
#endif

#include <stdarg.h>

#include <netinet/tcp.h>

#ifdef SOLARIS
#include <pthread.h>
#include <thread.h>
#else
#include <pthread.h>
#endif

#include "ncurses-gui.h"



/*
 * Initialization routines for the GUI
 */
void 
ncurses_gui(void *args)
{
    int tmp;
    WINDOW *my_wins[NCURSES_MAX_WINDOWS];
    PANEL  *my_panels[NCURSES_MAX_WINDOWS];
    struct term_node *term_node = NULL;
    time_t this_time;
    sigset_t mask;

write_log(0,"\n ncurses_gui_th = %d\n",(int)pthread_self());

    pthread_mutex_lock(&terms->gui_th.finished);

    sigfillset(&mask);

    if (pthread_sigmask(SIG_BLOCK, &mask, NULL))
    {
       thread_error("ncurses_gui_th pthread_sigmask()",errno);
       ncurses_gui_th_exit(NULL);
    }
 
    if (pthread_mutex_lock(&terms->mutex) != 0) {
        thread_error("ncurses_gui_th pthread_mutex_lock",errno);
        ncurses_gui_th_exit(NULL);
    }

    if (term_add_node(&term_node, TERM_CON, (int)NULL, pthread_self()) < 0)
    {
       if (pthread_mutex_unlock(&terms->mutex) != 0)
          thread_error("ncurses_gui_th pthread_mutex_unlock",errno);
       ncurses_gui_th_exit(NULL);
    }
    
    if (term_node == NULL)
    {
       write_log(0, "Ouch!! No more than %d %s accepted!!\n", 
               term_type[TERM_CON].max, term_type[TERM_CON].name);
       if (pthread_mutex_unlock(&terms->mutex) != 0)
          thread_error("ncurses_gui_th pthread_mutex_unlock",errno);
       ncurses_gui_th_exit(NULL);
    }

    this_time = time(NULL);

#ifdef HAVE_CTIME_R
#ifdef SOLARIS
    ctime_r(&this_time,term_node->since, sizeof(term_node->since));
#else
    ctime_r(&this_time,term_node->since);
#endif
#else
    pthread_mutex_lock(&mutex_ctime);
    strncpy(term_node->since, ctime(&this_time), sizeof(term_node->since));
    pthread_mutex_unlock(&mutex_ctime);
#endif

    /* Just to remove the cr+lf...*/
    term_node->since[sizeof(term_node->since)-2] = 0;

    /* This is a console so, man... ;) */
    strncpy(term_node->from_ip, "127.0.0.1", sizeof(term_node->from_ip));
   
    if (pthread_mutex_unlock(&terms->mutex) != 0) {
        thread_error("ncurses_gui_th pthread_mutex_unlock",errno);
        ncurses_gui_th_exit(term_node);
    }

    if (ncurses_gui_init(my_wins, my_panels, term_node) < 0)
       ncurses_gui_th_exit(term_node);
       
    /* take the first valid interface */
    if (strlen(interfaces[0].ifname)) {
        if (ncurses_gui_error_window(0,
                    "Warning: interface %s selected as the default one", 
                    interfaces[0].ifname) < 0)
            ncurses_gui_th_exit(term_node);
        if ((tmp = interfaces_add(interfaces[0].ifname)) == -1) {
            if (ncurses_gui_error_window(1,
                        "Unable to use interface %s!! (Maybe nonexistent?)\n\n", 
                        interfaces[0].ifname) < 0)
                ncurses_gui_th_exit(term_node);
         } else
            term_node->used_ints[tmp] = 1;
    } else {
        if (ncurses_gui_error_window(1,
                    "Hmm... you don't have any valid interface. \
                    %s is useless. Go and get a life!", PACKAGE) < 0)
            ncurses_gui_th_exit(term_node);
    }

    ncurses_gui_engine(my_wins, my_panels, term_node);

    ncurses_gui_th_exit(term_node);
}


/* 
 * GUI destroy. End
 */
void
ncurses_gui_th_exit(struct term_node *term_node)
{
    u_int8_t i;
    
    if (endwin() == ERR)
        thread_error("Error in endwin", errno);

    write_log(0, "\n ncurses_gui_th_exit start...\n");
   
   if (term_node)
   {
      for (i=0; i< MAX_INTERFACES; i++)
      {
         if (term_node->used_ints[i])
            interfaces_del(i);
      }   

      attack_kill_th(term_node,ALL_ATTACK_THREADS);

      if (pthread_mutex_lock(&terms->mutex) != 0)
         thread_error("ncurses_gui_th pthread_mutex_lock",errno);
      
      term_delete_node(term_node, NOKILL_THREAD);               

      if (pthread_mutex_unlock(&terms->mutex) != 0)
         thread_error("ncurses_gui_th pthread_mutex_unlock",errno);
   }

write_log(0," ncurses_gui_th_exit finish...\n");   
   
   terms->gui_th.id = 0;

   if (pthread_mutex_unlock(&terms->gui_th.finished) != 0)
      thread_error("ncurses_gui_th pthread_mutex_unlock",errno);
   
   pthread_exit(NULL);
}




/*
 * Ncurses init
 */
int8_t
ncurses_gui_init(WINDOW *my_wins[], PANEL *my_panels[], struct term_node *node)
{
    int32_t row, col;

    initscr();

    if (has_colors() == FALSE)
        write_log(0, " Warning: your terminal does not support color\n");
    else
       start_color();

    cbreak();
    nonl();
    noecho();

    init_pair(1, COLOR_RED, COLOR_BLACK);
    init_pair(2, COLOR_GREEN, COLOR_BLACK);
    init_pair(3, COLOR_BLUE, COLOR_BLACK);
    init_pair(4, COLOR_CYAN, COLOR_BLACK);
    init_pair(5, COLOR_YELLOW, COLOR_BLACK);
    init_pair(6, COLOR_BLACK, COLOR_BLUE);

    getmaxyx(stdscr,row,col);

    /* minimum rows & columns to display a proper GUI */
    if ((col < NCURSES_MIN_COLS) || (row < NCURSES_MIN_ROWS))
    {
        write_log(1, " Error: I need at least %d rows and %d columns \
                for a proper display!!\n", NCURSES_MIN_ROWS,
                NCURSES_MIN_COLS);
        return -1;
    }

    /* main window */
    my_wins[MAIN_SCREEN] = newwin(row- NCURSES_BWINDOW_SIZE - 1, col, 0, 0);
    /* secondary window */
    my_wins[SEC_SCREEN] = newwin(NCURSES_BWINDOW_SIZE, col, row-NCURSES_BWINDOW_SIZE, 0);

    /* Create windows for the panels */
    my_wins[LIST_FILECAPS] = newwin(20, 60, (row-20)/2, (col-60)/2);
    my_wins[LIST_ATTACKS] = newwin(20, 60, (row-20)/2, (col-60)/2);
    my_wins[IFACE_SCREEN] = newwin(10, 50, (row-10)/2, (col-50)/2);
    my_wins[ATTACK_SCREEN] = newwin(15, 50, (row-15)/2, (col-50)/2);
//    my_wins[INFO_SCREEN] = newwin(1, col, NCURSES_BWINDOW_SIZE - 1, 0);
    info_window = newwin(1, col, row - NCURSES_BWINDOW_SIZE - 1, 0);
    my_wins[HELP_SCREEN] = newwin(23, 42, (row-23)/2, (col-41)/2);
    my_wins[SPLASH_SCREEN] = newwin(19, 80, (row-19)/2, (col-80)/2);

    /* main window */
    my_panels[MAIN_SCREEN] = new_panel(my_wins[MAIN_SCREEN]);
    /* secondary window */
    my_panels[SEC_SCREEN] = new_panel(my_wins[SEC_SCREEN]);

    /* Attach a panel to each window */     /* Order is bottom up */
    my_panels[LIST_FILECAPS] = new_panel(my_wins[LIST_FILECAPS]); /* Push 0, order: stdscr-0 */
    my_panels[LIST_ATTACKS] = new_panel(my_wins[LIST_ATTACKS]);   /* Push 0, order: stdscr-0 */
    my_panels[IFACE_SCREEN] = new_panel(my_wins[IFACE_SCREEN]);   /* Push 0, order: stdscr-0 */
    my_panels[ATTACK_SCREEN] = new_panel(my_wins[ATTACK_SCREEN]); /* Push 0, order: stdscr-0 */
//    my_panels[INFO_SCREEN] = new_panel(my_wins[INFO_SCREEN]);   /* Push 0, order: stdscr-0 */
//    info_panel = new_panel(my_wins[INFO_SCREEN]);   /* Push 0, order: stdscr-0 */
    my_panels[HELP_SCREEN] = new_panel(my_wins[HELP_SCREEN]);   /* Push 2, order: stdscr-2 */
    my_panels[SPLASH_SCREEN] = new_panel(my_wins[SPLASH_SCREEN]);
    
    curs_set(0);

	if (tty_tmp->splash != 0)
	{
		ncurses_gui_splash_screen(my_wins[SPLASH_SCREEN],NULL);
		show_panel(my_panels[SPLASH_SCREEN]);
		ncurses_gui_refresh();

		/* Hmmm propaganda... */
		sleep(1);

		hide_panel(my_panels[SPLASH_SCREEN]);

		ncurses_gui_refresh();
    }

    ncurses_gui_refresh_mwindow(NCURSES_DEFAULT_MODE, my_wins[MAIN_SCREEN], 0, node);
    ncurses_gui_refresh_bwindow(NCURSES_DEFAULT_MODE, my_wins[SEC_SCREEN], node);
    
    return 0;
}


/*
 * About panel
 */
void 
ncurses_gui_splash_screen(WINDOW *splash_screen, PANEL *splash_panel)
{
    int32_t row, col, y, x;

    /* Check that the window is centered */
    getmaxyx(stdscr, row, col);
    getyx(splash_screen, y, x);
    if ((row - 19 != y) || (col - 80 != x))
        mvwin(splash_screen, (row - 19)/2, (col - 80)/2);

    wclear(splash_screen);

    wattron(splash_screen, COLOR_PAIR(3));
    box(splash_screen, 0, 0);
    mvwprintw(splash_screen, 1, 2, "    ####");
    mvwprintw(splash_screen, 2, 2, "   ##");
    mvwprintw(splash_screen, 3, 2, " #####");
    mvwprintw(splash_screen, 4, 2, "#   ##");
    mvwprintw(splash_screen, 5, 2, "#      ##");
    mvwprintw(splash_screen, 6, 2, "#       #");
    mvwprintw(splash_screen, 7, 2, "#        ###");                                                         
    mvwprintw(splash_screen, 8, 2, "##         ####");
    mvwprintw(splash_screen, 9, 2, " ##          #");
    mvwprintw(splash_screen,10, 2, " ###            ##");
    mvwprintw(splash_screen,11, 2, "  ###             ##");
    mvwprintw(splash_screen,12, 2, "     ##              ##");
    mvwprintw(splash_screen,13, 2, "      ###            ####");
    mvwprintw(splash_screen,14, 2, "       ###          ####");
    mvwprintw(splash_screen,15, 2, "         ##        ##");
    mvwprintw(splash_screen,16, 2, "         ####### ##");
    mvwprintw(splash_screen,17, 2, "             ##");
    wattroff(splash_screen, COLOR_PAIR(3));

    mvwprintw(splash_screen, 4, 8, "");
    mvwprintw(splash_screen, 5, 7, "");
    mvwprintw(splash_screen, 6, 7, "");
    mvwprintw(splash_screen, 7, 8, "");                                                         
    mvwprintw(splash_screen, 8, 8, "");
    mvwprintw(splash_screen, 9,10, "");
    mvwprintw(splash_screen,10,11, "");
    mvwprintw(splash_screen,11,12, "");
    mvwprintw(splash_screen,12,13, "");
    mvwprintw(splash_screen,13,15, "");
    mvwprintw(splash_screen,14,17, "");
    mvwprintw(splash_screen,15,19, "");
    mvwprintw(splash_screen,16,24, "");

    wattron(splash_screen, A_BOLD);
    mvwprintw(splash_screen, 2, 17, "Chaos Internetwork Operating System Software");
    mvwprintw(splash_screen, 3, 17, "%s (tm) Software (%s), Version %s, RELEASE SOFTWARE", 
            PACKAGE, INFO_PLATFORM, VERSION);
    mvwprintw(splash_screen, 4, 21, "Copyright (c) 2004-2004 by tomac & Slay, Inc.");
    mvwprintw(splash_screen, 5, 22, "Compiled %s by someone", INFO_DATE);
   if ( uptime < 60 )
      mvwprintw(splash_screen, 6, 23, "%s uptime is %02lu seconds", PACKAGE, uptime);
   else
   {
      if ( uptime < 3600 )
         mvwprintw(splash_screen, 6, 23, "%s uptime is %02lu minutes, %02lu seconds",
                      PACKAGE, uptime / 60, uptime % 60);
      else
      {
         if ( uptime < (3600*24) )
         {
            mvwprintw(splash_screen, 6, 23, "%s uptime is %02lu hours, %02lu minutes, \
                    %02lu seconds", PACKAGE, uptime / 3600, (uptime % 3600) / 60, uptime % 60);
         }
         else
            mvwprintw(splash_screen, 6, 23, "%s uptime is %02lu days, %02lu hours, \
                    %02lu minutes, %02lu seconds", PACKAGE, uptime / (3600*24), 
                    (uptime % (3600*24)) / 3600, (uptime % 3600) / 60, uptime % 60);
      }
   }
   mvwprintw(splash_screen, 8, 29, "Running Multithreading Image on"); 
   mvwprintw(splash_screen, 9, 30, "%s %s supporting:", INFO_KERN, INFO_KERN_VER);
   mvwprintw(splash_screen, 11, 40, "%02d console terminal(s)", MAX_CON);
   mvwprintw(splash_screen, 12, 40, "%02d tty terminal(s)", MAX_TTY);
   mvwprintw(splash_screen, 13, 40, "%02d vty terminal(s)", MAX_VTY);
   wattron(splash_screen, A_BOLD | A_BLINK);
   wattroff(splash_screen, A_BOLD | A_BLINK);

   if (splash_panel)
   {
      wtimeout(splash_screen, NCURSES_KEY_TIMEOUT);
      while ((wgetch(splash_screen)==ERR) && !terms->gui_th.stop);
      hide_panel(splash_panel);
   }
}


/* 
 * Help panel
 */
void 
ncurses_gui_help_screen(u_int8_t mode, WINDOW *help_screen, PANEL *help_panel)
{
    int32_t row, col, y, x;

    ncurses_gui_set_status_line(" This is the help screen. ");

    /* Check that the window is centered */
    getmaxyx(stdscr, row, col);
    getyx(help_screen, y, x);
    if ((row - 24 != y) || (col - 41 != x))
        mvwin(help_screen, (row - 24)/2, (col - 41)/2);

    wattron(help_screen, COLOR_PAIR(2));
    box(help_screen, 0, 0);

    /* common help for all modes */
    mvwprintw(help_screen, 0, 13, " Available commands ");
    mvwprintw(help_screen, 1, 2, "h");
    mvwprintw(help_screen, 2, 2, "x");
    mvwprintw(help_screen, 3, 2, "i");
    mvwprintw(help_screen, 4, 2, "ENTER");
    mvwprintw(help_screen, 5, 2, "v");
    mvwprintw(help_screen, 6, 2, "d");
    mvwprintw(help_screen, 7, 2, "e");
    mvwprintw(help_screen, 8, 2, "f");
    mvwprintw(help_screen, 9, 2, "s");
    mvwprintw(help_screen, 10, 2, "S");
    mvwprintw(help_screen, 11, 2, "L");
    mvwprintw(help_screen, 12, 2, "M");
    mvwprintw(help_screen, 13, 2, "l");
    mvwprintw(help_screen, 14, 2, "K");
    mvwprintw(help_screen, 15, 2, "c");
    mvwprintw(help_screen, 16, 2, "C");
    mvwprintw(help_screen, 17, 2, "g");
    mvwprintw(help_screen, 18, 2, "Ctrl-L");
    mvwprintw(help_screen, 19, 2, "w");
    mvwprintw(help_screen, 20, 2, "a");
    mvwprintw(help_screen, 21, 2, "q");
    wattroff(help_screen, COLOR_PAIR(2));

    mvwprintw(help_screen, 1, 9, "Help screen");
    mvwprintw(help_screen, 2, 9, "eXecute attack");
    mvwprintw(help_screen, 3, 9, "edit Interfaces");
    mvwprintw(help_screen, 4, 9, "information about selected item");
    mvwprintw(help_screen, 5, 9, "View hex packet dump");
    mvwprintw(help_screen, 6, 9, "load protocol Default values");
    mvwprintw(help_screen, 7, 9, "Edit packet fields");
    mvwprintw(help_screen, 8, 9, "list capture Files");
    mvwprintw(help_screen, 9, 9, "Save packets from protocol");
    mvwprintw(help_screen, 10, 9, "Save packets from all protocols");
    mvwprintw(help_screen, 11, 9, "Learn packet from network");
    mvwprintw(help_screen, 12, 9, "set Mac spoofing on/off");
    mvwprintw(help_screen, 13, 9, "List running attacks");
    mvwprintw(help_screen, 14, 9, "Kill all running attacks");
    mvwprintw(help_screen, 15, 9, "Clear current protocol stats");
    mvwprintw(help_screen, 16, 9, "Clear all protocols stats");
    mvwprintw(help_screen, 17, 9, "Go to other protocol screen");
    mvwprintw(help_screen, 18, 9, "redraw screen");
    mvwprintw(help_screen, 19, 9, "Write configuration file");
    mvwprintw(help_screen, 20, 9, "About this proggie");
    mvwprintw(help_screen, 21, 9, "Quit (bring da noize)");

    wtimeout(help_screen,NCURSES_KEY_TIMEOUT);

    while ( (wgetch(help_screen)==ERR) && !terms->gui_th.stop);
    
    hide_panel(help_panel);

    ncurses_gui_set_status_line("");
}


/* 
 * Attack information panel
 */
void 
ncurses_gui_attack_screen(struct term_node *node, u_int8_t mode, 
        WINDOW *attack_screen, PANEL *attack_panel)
{
    int32_t i, key_pressed=0, j, row, col, y, x;
    u_int8_t field;
    int8_t ret;
    struct attack_param *attack_param = NULL;
    struct attack *theattack = NULL;

    ncurses_gui_set_status_line(" Those strange attacks... ");

    /* Check that the window is centered */
    getmaxyx(stdscr, row, col);
    getyx(attack_screen, y, x);
    if ((row - 15 != y) || (col - 50 != x))
        mvwin(attack_screen, (row - 15)/2, (col - 50)/2);

    if (protocols[mode].attacks)
        theattack = protocols[mode].attacks;
    else {
        write_log(0, "Warning: no attacks for mode %d\n", mode);
        return;
    }

    i = 0;
    wclear(attack_screen);
    wattron(attack_screen, COLOR_PAIR(1));
    box(attack_screen, 0, 0);
    mvwprintw(attack_screen, 0, 18, " Attack Panel ");
    mvwprintw(attack_screen, 14, 5, " Select attack to launch ('q' to quit) ");
    mvwprintw(attack_screen, 1, 2, "No   DoS   Description");
    wattroff(attack_screen, COLOR_PAIR(1));

    while(theattack[i].s != NULL) 
    {
        mvwprintw(attack_screen, i+2, 2, "%d", i);
        mvwprintw(attack_screen, i+2, 7, "%c", (theattack[i].type == DOS) ? 'X' : ' ');
        mvwprintw(attack_screen, i+2, 13, "%s", theattack[i].s);
        i++;
    }


    wtimeout(attack_screen,NCURSES_KEY_TIMEOUT); /* Block for 100 millisecs...*/
    
    while ( (key_pressed !='Q') && (key_pressed != 'q') && !terms->gui_th.stop)
    {

        key_pressed = wgetch(attack_screen);
        switch (key_pressed) {
            case 'Q':
            case 'q':
            break;
            case '0':
            case '1':
            case '2':
            case '3':
            case '4':
            case '5':
            case '6':
            case '7':
            case '8':
            case '9':
                j = key_pressed - 48;
                if (j < i) 
                { /* does the attack exist? */

                    if (theattack[j].nparams) /* Do we need parameters for attack? */
                    {
                       if ((attack_param = calloc(1,
                                  (sizeof(struct attack_param) * theattack[j].nparams))) == NULL)
                       {
                          thread_error(" ncurses_gui_attack_screen attack_param calloc",errno);
                          key_pressed='Q';
                          break;
                       }
                       memcpy( attack_param, (void *)(theattack[j].param),
                               sizeof(struct attack_param) * theattack[j].nparams);
                       if (attack_init_params(node, attack_param, theattack[j].nparams) < 0)
                       {
                          free(attack_param);
                          key_pressed='Q';
                          break;
                       }
                       
		       /* Hide the attack panel */
		       hide_panel(attack_panel);

                       /* Now we can ask for parameters... */
                       ncurses_gui_get_printable_store(attack_param, theattack[j].nparams);

                       do {
                           if (ncurses_gui_attack_get_params(attack_param, 
                                                theattack[j].nparams) < 0) /* Q pressed */
                           {
                              attack_free_params(attack_param, theattack[j].nparams);
                              free(attack_param);
                              key_pressed='Q';
                              break;
                           }
                           ret = attack_filter_all_params(attack_param,theattack[j].nparams, &field);
                           if ( ret == -1) /* Error on data...*/
                           {
                              ncurses_gui_error_window(1,
                                                       "Bad data on field '%s'!!",
                                                       attack_param[field].desc);
                           }
                       } while(ret==-1);
                       
                       if (key_pressed == 'Q')
                       {
			    i = 0;
			    wclear(attack_screen);
			    wattron(attack_screen, COLOR_PAIR(1));
			    box(attack_screen, 0, 0);
			    mvwprintw(attack_screen, 0, 20, " Attack Panel ");
			    mvwprintw(attack_screen, 14, 5, " Select attack to launch ('q' to quit) ");
			    mvwprintw(attack_screen, 1, 2, "No   DoS   Description");
			    wattroff(attack_screen, COLOR_PAIR(1));

			    while(theattack[i].s != NULL) 
			    {
				mvwprintw(attack_screen, i+2, 2, "%d", i);
				mvwprintw(attack_screen, i+2, 7, "%c", (theattack[i].type == DOS) ? 'X' : ' ');
				mvwprintw(attack_screen, i+2, 13, "%s", theattack[i].s);
				i++;
			    }

                           key_pressed = 0;
                           continue;
                       }
                    }
                    
                    wrefresh(attack_screen);
		    show_panel(attack_panel);
		    ncurses_gui_refresh();

                    if (attack_launch(node, mode, j, attack_param, theattack[j].nparams) < 0)
                        write_log(0, "Error launching attack %d", j);
                    
                    key_pressed='Q';
                }
            break;
        }
    }

    hide_panel(attack_panel);

    ncurses_gui_set_status_line("");
}




/*
 * Refresh main window
 */
void 
ncurses_gui_refresh_mwindow(u_int8_t mode, WINDOW *mwindow, u_int8_t pointer, 
        struct term_node *node)
{
    u_int8_t i, row, col, j, offset, tlv, k;
    char *ptrtlv, *top;
    char timebuf[19], **values;
    struct term_console *term_console;
    struct tm *aux;
    struct position_fields *fields;
    struct tuple_type_desc *func;
    time_t this_time;

    tlv = 0;
    fields = NULL;
    values = NULL;
    func = NULL;

    term_console = node->specific;

    fields = protocols[mode].fields;

    getmaxyx(stdscr,row,col);

#ifdef CAN_RESIZE
    if (term_console->need_resize) {
/*        if (delwin(mwindow) == ERR)
            thread_error("Error in delwin", errno);
        if ((mwindow = newwin(row-5, col, 0, 0)) == NULL) 
            thread_error("Error in newwin", errno);*/
#ifdef HAVE_NCURSES_WRESIZE
        wresize(mwindow, row-NCURSES_BWINDOW_SIZE, col);
        werase(mwindow);
#endif
        term_console->need_resize--;
        write_log(0, "Resize mwindow con %d row y %d col y resize es %d\n", row, 
                col, term_console->need_resize);
    }
#endif


    offset = (col > NCURSES_MIN_COLS)? (col - NCURSES_MIN_COLS) / 2 : 0;

    wattron(mwindow, COLOR_PAIR(5) | A_BOLD);
    box(mwindow,ACS_VLINE,ACS_HLINE);

    this_time = time(NULL);
   
    aux = localtime(&this_time);

    if (aux != NULL)
       mvwprintw(mwindow, 0, col-11, "[%02d:%02d:%02d]", aux->tm_hour, aux->tm_min,
                                      aux->tm_sec);

    mvwprintw(mwindow, row-NCURSES_BWINDOW_SIZE-2, 1 + offset, " Total Packets: %d ", 
            packet_stats.global_counter.total_packets);
    mvwprintw(mwindow, row-NCURSES_BWINDOW_SIZE-2, 60 + offset, " MAC Spoofing [ ] ");
    if (node->mac_spoofing)
        mvwprintw(mwindow, row-NCURSES_BWINDOW_SIZE-2, 75 + offset, "X");

    mvwprintw(mwindow, 0, 3, " %s %s by Slay & tomac - %s mode ", 
		PACKAGE, VERSION, protocols[mode].name);
    mvwprintw(mwindow, row-NCURSES_BWINDOW_SIZE-2, 30 + offset, " %s Packets: %d ", 
		protocols[mode].name, protocols[mode].packets);
    wattroff(mwindow, COLOR_PAIR(5) | A_BOLD);

    wattron(mwindow, A_BOLD);
    i = 0;

    while(fields[i].name) {
	if (fields[i].mwindow)
		mvwprintw(mwindow, 1, fields[i].mwindowx + offset, fields[i].name);
	i++;
    }
    mvwprintw(mwindow, 1, 58 + offset, "Iface");
    mvwprintw(mwindow, 1, 64 + offset, "Last seen");

    wattroff(mwindow, A_BOLD);

    for (i=0; i < MAX_PACKET_STATS; i++)
    {
	if (protocols[mode].stats[i].header->ts.tv_sec > 0) 
	{
            if (protocols[mode].get_printable_packet)
            {
                if ((values = (*protocols[mode].get_printable_packet)(&protocols[mode].stats[i])) == NULL) 
                {
                   write_log(0, "Error in get_printable_packet (mode %d)\n", mode);
                   wrefresh(mwindow);
                   return;
                }
            }
            else
            {
                write_log(0, "Warning: there is no get_printable_packet for protocol %d\n", mode);
                wrefresh(mwindow);
                return;
            }
            
            if (i == pointer)
            wattron(mwindow, COLOR_PAIR(5) | A_BOLD);

            wmove(mwindow, i+2, 1);
            whline(mwindow, ' ', COLS-2);

            j = 0;
            
            while(fields[j].name) 
            {
                /* Recognize the TLV field if it exists */
                if ((fields[j].type == FIELD_TLV) && (fields[j].mwindow == 0))
                        tlv = j;

                if (fields[j].mwindow) 
                {
                    if (fields[j].type != FIELD_TLV)
                        mvwprintw(mwindow, i + 2, fields[j].mwindowx + offset, "%s %s", values[j],
                              (fields[j].desc) ? (*protocols[mode].get_info)(j, values[j]) : "");
                    else 
                       if (fields[j].mwindow == 1) 
                       {   /* TLV stuff; the TLV are in values[tlv]*/
                           ptrtlv = values[tlv];
                           top = ptrtlv + (2*MAX_TLV*MAX_VALUE_LENGTH);
                           /* Look for the name */
                           while((ptrtlv < top) && 
                                 (strncmp((char *)ptrtlv, (char *)fields[j].name, strlen((char *)fields[j].name)) != 0)) 
                           {
                               ptrtlv += strlen((char *)ptrtlv) + 1;
                           }
                           k = j;
                           while(fields[k].name) 
                           {
                               if (strncmp((char *)ptrtlv, (char *)fields[k].name, strlen((char *)fields[k].name)) == 0)
                                   func = (struct tuple_type_desc *)fields[k].desc;
                               k++;
                           }

                           ptrtlv += strlen((char *)ptrtlv) + 1;
                           mvwprintw(mwindow, i + 2, fields[j].mwindowx + offset, "%*c", fields[j].l, ' ');
                           mvwprintw(mwindow, i + 2, fields[j].mwindowx + offset, "%s %s", ptrtlv, 
                                         (func) ? (*protocols[mode].get_info)(fields[k-1].v, ptrtlv) : "");
                       }
                }
                j++;
            } /* while(fields[j].name) */

            mvwprintw(mwindow, i+2, 58 + offset, "%s",
                    interfaces[protocols[mode].stats[i].iface].ifname); 
            strftime(timebuf, 19, "%d %b %H:%M:%S", localtime((time_t *)&protocols[mode].stats[i].header->ts));

            mvwprintw(mwindow, i+2, 64 + offset, "%s", timebuf);

            if (i == pointer) 
                wattroff(mwindow, COLOR_PAIR(5) | A_BOLD);

	    k = 0;
	    if (values) 
	    {
		while(values[k]) 
		{
		    free(values[k]);
		    k++;
		}
		free(values);
	    }

	} /* if (protocols->tv_sec) */


    } /* for i < MAX_PACKET_STATS */

    wrefresh(mwindow);
}


/*
 * Refresh panels
 */
void 
ncurses_gui_refresh(void)
{
    /* refresh the panels */
    update_panels();

    /* Show it on the screen */
    doupdate();
}



/*
 * Main engine catching keystrokes
 */
void 
ncurses_gui_engine(WINDOW *my_wins[], PANEL *my_panels[], struct term_node *node)
{
    int32_t key, key_pressed, ret, max;
    u_int8_t end, i, j, secs, mode, iface;
    struct term_console *term_console;
    fd_set read_set;
    struct timeval timeout;

    term_console = node->specific;

    for (i = 0; i < MAX_PROTOCOLS; i++)
	pointer[i] = 0;

    end = 0;
    
    /* default initial mode */
    mode = NCURSES_DEFAULT_MODE;

    keypad(my_wins[MAIN_SCREEN], TRUE);

    secs = 0;
    j = 0;

    timeout.tv_sec  = 0;
    timeout.tv_usec = 0;

    while (!end && !terms->gui_th.stop)
    {
        max = 0;
        FD_SET(fileno(stdin), &read_set);

        if ((ret = select(max + 1, &read_set, NULL, NULL, &timeout)) == -1) {
            thread_error("select", errno);
            timeout.tv_sec = 0;
            timeout.tv_usec = 500000;
            continue;
        }
        
        if (!ret) /* Timeout...*/
        {
           if (j%2) /* 1 sec!! */
           {
              j=0;
              if (secs == NCURSES_REFRESH_TIME)
              {
                  ncurses_gui_refresh_bwindow(mode, my_wins[SEC_SCREEN], node);
                  ncurses_gui_refresh_mwindow(mode, my_wins[MAIN_SCREEN], 
                          pointer[mode], node); 
                  secs = 0;
              }
              else
                 secs++;
           }
           else
              j++;
        }
        else
        {
            key_pressed = wgetch(my_wins[MAIN_SCREEN]);

            switch(key_pressed)
            {
                /* Help screen */
                case '?':
                case 'H':
                case 'h':
                    ncurses_gui_help_screen(mode, my_wins[HELP_SCREEN], 
                            my_panels[HELP_SCREEN]);
                break;
                
                /* Edit BPDU fields */
                case 'E':
                case 'e':
                    ncurses_gui_edit_bwindow(mode, my_wins[MAIN_SCREEN], my_wins[SEC_SCREEN], node);
                break;
                
                /* About */
                case 'A':
                case 'a':
                    ncurses_gui_splash_screen(my_wins[SPLASH_SCREEN],
                            my_panels[SPLASH_SCREEN]);
                break; 
                
                /* List Attacks */
                case 'l':
                    ncurses_gui_list_attacks(my_wins[LIST_ATTACKS], node);
                break;

                /* List capture files */
                case 'f':
                    ncurses_gui_list_filecaps(my_wins[LIST_FILECAPS], node);
                break;
                
                case KEY_DOWN:
			if ( (pointer[mode] < MAX_PACKET_STATS - 1) && 
			 (protocols[mode].stats[pointer[mode] + 1].header->ts.tv_sec > 0)) 
			{
			pointer[mode]++;
			ncurses_gui_refresh_mwindow(mode, my_wins[MAIN_SCREEN], pointer[mode], node);
			}
                break;
                
                case KEY_UP:
                    if (pointer[mode] > 0) 
                    {
                        pointer[mode]--;
                        ncurses_gui_refresh_mwindow(mode, my_wins[MAIN_SCREEN], pointer[mode], node);
                    }
                break; 
                
                case 'K':
                    key = ncurses_gui_getconfirm(node,
                                    " Confirmation before nuclear war!!",
                                    "You will kill *ALL* attacks from *ALL* protocols... Are you sure?",
                                    " Killing attacks from ALL protocols ");
                    if (key == 'y')
                       attack_kill_th(node, ALL_ATTACK_THREADS);
                break;

                case 'c':
                    interfaces_clear_stats(mode);
                    wclear(my_wins[MAIN_SCREEN]);
                    pointer[mode] = 0;
                    ncurses_gui_set_status_line(" Clearing Mode stats... ");
                break;

                case 'C':
                    interfaces_clear_stats(PROTO_ALL);
                    wclear(my_wins[MAIN_SCREEN]);
                    memset(pointer,0,sizeof(pointer));
                    ncurses_gui_set_status_line(" Clearing stats... ");
                break;

                case 'd':
                    if (protocols[mode].init_attribs)
			(*protocols[mode].init_attribs)(node);
                    else {
                        write_log(0, "Warning: no init_attribs for mode %d\n", mode);
                    }
                break;

                /* Set MAC Spoofing on/off */
                case 'M':
                    if (node->mac_spoofing) 
                        node->mac_spoofing = 0;
                    else 
                        node->mac_spoofing = 1;
                break;
                
                /* Clear screen */
                case KEY_CTRL_L:
                    clearok(stdscr, TRUE);
                    clearok(my_wins[MAIN_SCREEN], TRUE);
                    clearok(my_wins[SEC_SCREEN], TRUE);
                break;
                
                /* ENTER */
                case 13:
                    if (ncurses_gui_show_info(mode, my_wins[MAIN_SCREEN], 
                                pointer[mode], node) < 0)
                    {
                       //end=1;
                       break;
                    }
                break;

                /* View packet */
                case 'v':
                    ncurses_gui_view_packet(mode, my_wins[MAIN_SCREEN], pointer[mode]);
                break;
                
                /* BPDU Information */
                case 'I':
                case 'i':
                      ncurses_gui_ifaces_screen(node, my_wins[IFACE_SCREEN], 
                              my_panels[IFACE_SCREEN]);
                break;
                
                /* Save data */
                case 'S':
                    if (node->pcap_file.pdumper)
                    {                
                        if (ncurses_gui_error_window(1,
                             "Error: pcap_file is in use at %s", 
                             node->pcap_file.name) < 0)
                        ncurses_gui_th_exit(node);
                    }
                    else
                    {
                       char *filename = (char *)calloc(1,MAX_FILENAME+1);
                       if (filename == NULL)
                       {
                           thread_error(" ncurses_engine calloc",errno);
                           ncurses_gui_th_exit(node);
                       }
                       if (ncurses_gui_getstring_window(node, 
                                            " Stealing data? Tsk, tsk...",
                                            filename,
                                            MAX_FILENAME,
                                            " Enter pcap filename for ALL protocols") < 0)
                       {
                           free(filename);
                           ncurses_gui_th_exit(node);
                       }
                       
                       iface = interfaces_get_last_int(PROTO_ALL);
                       
                       if (filename[0] && interfaces_pcap_file_open(node, PROTO_ALL, filename, iface) < 0)
                       {
                          free(filename);
                          ncurses_gui_th_exit(node);
                       }
                       free(filename);
                    }
                break;

                case 's':
                    if (node->protocol[mode].pcap_file.pdumper)
                    {                
                        if (ncurses_gui_error_window(1, "Error: pcap_file is in use at %s", 
                             node->protocol[mode].pcap_file.name) < 0)
                            ncurses_gui_th_exit(node);
                    }
                    else
                    {
                       char *filename = (char *)calloc(1,MAX_FILENAME+1);
                       if (filename == NULL)
                       {
                           thread_error(" ncurses_engine calloc",errno);
                           ncurses_gui_th_exit(node);
                       }
                       if (ncurses_gui_getstring_window(node, 
                                            " Stealing data? Tsk, tsk...",
                                            filename,
                                            MAX_FILENAME,
                                            " Enter pcap filename for current protocol") < 0)
                       {
                           free(filename);
                           ncurses_gui_th_exit(node);
                       }
                       
                       iface = interfaces_get_last_int(mode);
                       
                       if (filename[0] && interfaces_pcap_file_open(node, mode, filename, iface) < 0)
                       {
                          free(filename);
                          ncurses_gui_th_exit(node);
                       }
                       free(filename);
                    }
                break;
                
                /* Learn da packet */
                case 'L':
                      ncurses_gui_learn_packet(mode, pointer[mode], node);
                break;

                /* Display implemented attacks */
                case 'X':
                case 'x':
                    ncurses_gui_attack_screen(node, mode, my_wins[ATTACK_SCREEN], 
                            my_panels[ATTACK_SCREEN]);
                break;
                
                /* Terminal console */
                case '!':
                    endwin();
                    system("telnet localhost 12000");
                    refresh();
                break;

		/* Write configuration file */
		case 'W':
		case 'w':
		    for (i = 0; i < MAX_PROTOCOLS; i++)
			memcpy((void *)protocols[i].default_values, (void *)node->protocol[i].tmp_data, protocols[i].size);

		    if (strlen(tty_tmp->config_file) == 0) {
                        char *filename = (char *)calloc(1,MAX_FILENAME+1);
                        if (ncurses_gui_getstring_window(node, 
                                             " Save the world ",
                                             filename,
                                              MAX_FILENAME,
                                             " Enter configuration filename ") < 0)
                        {
                            free(filename);
                            ncurses_gui_th_exit(node);
			}
			strncpy(tty_tmp->config_file, filename, MAX_FILENAME);
		    }
                    if (parser_write_config_file(tty_tmp) < 0)
	                ncurses_gui_error_window(1, "Error opening config file %s", tty_tmp->config_file);
		break;
                /* Quit (bring da noize) */
                case 'Q':
                case 'q':
                    end = 1;
                break;
                case KEY_F(1):
                case KEY_F(2):
                case KEY_F(3): 
                case KEY_F(4): 
                case KEY_F(5): 
                case KEY_F(6):
                case KEY_F(7):
                   if (mode != (key_pressed - KEY_F0 - 1)) {
                       mode = key_pressed - KEY_F0 - 1;
                       wclear(my_wins[MAIN_SCREEN]);
                       wclear(my_wins[SEC_SCREEN]);
                   }
                break;
		case 'g':
		   if ((ret = ncurses_gui_get_mode(mode, my_wins[MAIN_SCREEN])) >= 0) {
		       mode = ret;
                       wclear(my_wins[MAIN_SCREEN]);
                       wclear(my_wins[SEC_SCREEN]);
                   }
                break;
#ifdef KEY_RESIZE
                case KEY_RESIZE:
                    ncurses_gui_term_resize(node);
                break;
#endif
                case ERR:
                    thread_error("Error in wgetch", errno);
                    ncurses_gui_th_exit(NULL);
                break;

                default:
                break;
            }

        }

        ncurses_gui_refresh_bwindow(mode, my_wins[SEC_SCREEN], node);
        ncurses_gui_refresh_mwindow(mode, my_wins[MAIN_SCREEN], pointer[mode], node); 

        timeout.tv_sec = 0;
        timeout.tv_usec = 500000;
    } /* while */
    
}


/*
 * Refresh the fields window (Bottom window)
 */
void 
ncurses_gui_refresh_bwindow(u_int8_t mode, WINDOW *bwindow, struct term_node *node)
{
    int32_t offset, row, col;
    struct term_console *term_console;
    u_int8_t **store_values = NULL;
    u_int8_t i;
    struct position_fields *fields;

    term_console = node->specific;

    getmaxyx(stdscr,row,col);

#ifdef CAN_RESIZE
    if (term_console->need_resize) {
/*        if (delwin(bwindow) == ERR)
            thread_error("Error in delwin", errno);
        if ((bwindow = newwin(5, col, row-5, 0)) == NULL)
            thread_error("Error in newwin", errno);*/
#ifdef HAVE_NCURSES_WRESIZE
        wresize(bwindow, NCURSES_BWINDOW_SIZE, col);
        mvwin(bwindow, row-NCURSES_BWINDOW_SIZE, 0);
        werase(bwindow); 
#endif
        term_console->need_resize--;
        write_log(0, "Resize bwindow con %d row y %d col y resize es %d\n", row, col, term_console->need_resize);
    }
#endif

    offset = (col > NCURSES_MIN_COLS)? (col - NCURSES_MIN_COLS) / 2 : 0;

    if (protocols[mode].get_printable_store) {
        if ((store_values = (u_int8_t **)(*protocols[mode].get_printable_store)(node)) == NULL) {
            write_log(0, "Error in get_printable_store (mode %d)\n", mode);
            return;
        }
    }
    else {
        write_log(0, "Warning: there is no get_printable_store for protocol %d\n", mode);
        return;
    }

    wattron(bwindow, COLOR_PAIR(2));

    box(bwindow,ACS_VLINE,ACS_HLINE);
    mvwprintw(bwindow, 0, 3, " %s Fields ", protocols[mode].name);

    fields = protocols[mode].fields;
    i = 0;
    while(fields[i].name) {
	if ((fields[i].type != FIELD_TLV) || (fields[i].mwindow == 0)) {
	   mvwprintw(bwindow, fields[i].y, fields[i].fx + offset, "%s", fields[i].name);
	}
	i++;
    }

    wattroff(bwindow, COLOR_PAIR(2));
    i = 0;
    while(fields[i].name) 
    {
	if ((fields[i].type != FIELD_TLV))
//	if ((fields[i].type != FIELD_TLV) || (fields[i].mwindow == 0))
	{
	   if ( ((fields[i].type == FIELD_STR) || (fields[i].type == FIELD_IFACE) )
	        && (fields[i].type != FIELD_TLV) )
              mvwprintw(bwindow, fields[i].y, fields[i].x + offset, "%*c", fields[i].l, ' ');
           mvwprintw(bwindow, fields[i].y, fields[i].x + offset, "%s", store_values[i]);
        }
	i++;
    }

    i = 0;

    while (store_values[i]) {
	free(store_values[i]);
	i++;
    }

    if (store_values)
        free(store_values);


    wrefresh(bwindow);

}


/*
 * Edit BPDU Fields
 */
void 
ncurses_gui_edit_bwindow(u_int8_t mode, WINDOW *mwindow, WINDOW *bwindow, struct term_node *node)
{
    int8_t state = 0;
    int32_t key_pressed;
    int8_t end_edit;
    int32_t y, x, col, row, offset, result;
    struct position_fields *fields;

    ncurses_gui_set_status_line(" Edit those nasty fields ");

    /* We want a BIG cursor */
    curs_set(1);

    getmaxyx(stdscr, row, col);
    offset = (col > NCURSES_MIN_COLS)? (col - NCURSES_MIN_COLS) / 2 : 0;

    fields = protocols[mode].fields;

    keypad(bwindow, TRUE);
    wmove(bwindow, fields[state].y, fields[state].x + offset);
    wrefresh(bwindow);

    wtimeout(bwindow,NCURSES_KEY_TIMEOUT); /* Block for 100 millisecs...*/
    
    end_edit = 0;
    
    while (!end_edit && !terms->gui_th.stop)
    {
	do
	{
	    key_pressed = wgetch(bwindow);
	} while ( (key_pressed == ERR) && !terms->gui_th.stop);
	
	if (terms->gui_th.stop)
	    break;

	switch(key_pressed)
	{
            case 27: /* ESC */
            case 13: /* ENTER */
		end_edit = 1;
            break;
		
            case 9:  /* TAB */
	        if (fields[state].type == FIELD_TLV) {
	            getyx(bwindow, y, x);
	            curs_set(0);
		    ncurses_gui_edit_tlv(node, mode);
	            curs_set(1);
                    ncurses_gui_refresh_mwindow(mode, mwindow, 0, node); 
		    ncurses_gui_refresh_bwindow(mode, bwindow, node);
                    /* fix to come back to the position */
		    wmove(bwindow, y, x);
		}
		else 
		    if (fields[state].desc && protocols[mode].update_field)
                    {
			getyx(bwindow, y, x);
			curs_set(0);
			if ((result = ncurses_gui_popup_window(bwindow, fields, state, fields[state].y, 
				    fields[state].x)) != ERR) 
			    (*protocols[mode].update_field)(state, node, (void *)&result);
			curs_set(1);
			ncurses_gui_refresh_bwindow(mode, bwindow, node);    
			/* fix to come back to the position */
			wmove(bwindow, y, x);
		    }
            break;
		
		/* jump to the next character or the next field */
            case KEY_RIGHT:
		getyx(bwindow, y, x);
		   /* jump to the next valid character */
		if ((x - offset) < fields[state].x + (fields[state].type == FIELD_TLV ? 3 : fields[state].l) - 1)
		{
			if ((fields[state].type == FIELD_MAC) && 
				((((x - offset) - fields[state].x - 1) % 3) == 0)) 
			wmove(bwindow, y, x + 2); /* jump : */
			else if ((fields[state].type == FIELD_BRIDGEID) && 
				   (((x - offset) - fields[state].x) == 3)) 
			  wmove(bwindow, y, x + 2); /* jump . */
			else if ((fields[state].type == FIELD_IP) && 
				   ((((x - offset) - fields[state].x - 1) % 4) == 1)) 
			  wmove(bwindow, y, x + 2); /* jump . */
			   else
			  wmove(bwindow, y, x + 1);
		}
		else /* jump to the next field */
		{
			if (fields[state].position == FIELD_LAST) /* return to the first state */
			   state = 0;
			else
			   state++;
			wmove(bwindow, fields[state].y, fields[state].x + offset);
		}

		/* Status line */
                if (fields[state].desc)
                    ncurses_gui_set_status_line(" TAB for available values ");
                else
                    ncurses_gui_set_status_line("");
            break;
		
            case KEY_LEFT:
		getyx(bwindow, y, x);

		if ((x - offset) > fields[state].x)
		{
	       	    if ((fields[state].type == FIELD_MAC) &&
			((((x - offset) - fields[state].x) % 3) == 0))
			/* jump : */
			wmove(bwindow, y, x - 2);
		    else if ((fields[state].type == FIELD_BRIDGEID) && 
			       (((x - offset) - fields[state].x) == 5)) 
			/* jump . */
			  wmove(bwindow, y, x - 2);
		    else if ((fields[state].type == FIELD_IP) && 
			       ((((x - offset) - fields[state].x) % 4) == 0)) 
			/* jump . */
			  wmove(bwindow, y, x - 2);
		    else
			  wmove(bwindow, y, x - 1);
		}
		else 
		{
		   if (fields[state].position == FIELD_FIRST) /* go to the last state */
			  state = protocol_return_last_field(fields);
		   else
			  state--;
		   wmove(bwindow, fields[state].y, 
			   fields[state].x + (fields[state].type == FIELD_TLV ? 3 : fields[state].l) - 1 + offset);
		}

		/* Status line */
		if (fields[state].desc)
			ncurses_gui_set_status_line(" TAB for available values ");
		else
			ncurses_gui_set_status_line("");
            break;
		
            default:
		if ((fields[state].type == FIELD_HEX) || (fields[state].type == FIELD_MAC)
			|| (fields[state].type == FIELD_BRIDGEID) || (fields[state].type == FIELD_BYTES)) 
                {
		    if (!isxdigit(key_pressed))
			/* only hexadecimal characters are allowed */
                        break;
		} else if ((fields[state].type == FIELD_DEC) || (fields[state].type == FIELD_IP)) {
			if (!isdigit(key_pressed))
			break;
		} else if (fields[state].type == FIELD_STR) {
			if (!isascii(key_pressed))
			break;
		} else /* FIELD_NONE */
			break;
		waddch(bwindow, toupper(key_pressed) | A_BOLD);

		getyx(bwindow, y, x);
		
		(*protocols[mode].update_data)(state, key_pressed, 
			(x - offset) - fields[state].x - 1, node);

		if ((x - offset) > (fields[state].x + fields[state].l - 1))
		{
			if (fields[state].position == FIELD_LAST)
			   state = 0;
			else
			   state++;
			wmove(bwindow, fields[state].y, fields[state].x + offset);
		}
		else 
		{
                    if ( (fields[state].type == FIELD_MAC) && 
                         ((((x - offset) - fields[state].x + 1) % 3) == 0)) 
                         /* jump */
                       wmove(bwindow, y, x + 1);
                    else 
                        if ( (fields[state].type == FIELD_BRIDGEID) && 
                             (((x - offset) - fields[state].x) == 4)) 
                            /* jump */
                           wmove(bwindow, y, x + 1);
                        else 
                            if ( (fields[state].type == FIELD_IP) && 
                                ((((x - offset) - fields[state].x + 1) % 4) == 0)) 
                                /* jump */
                               wmove(bwindow, y, x + 1);
                }
            break;
	}
    }

    keypad(bwindow, FALSE);

    /* Hide the cursor */
    curs_set(0);

    ncurses_gui_set_status_line(" Edit mode is over ");
}


/* 
 * Set interfaces panel
 */
void 
ncurses_gui_ifaces_screen(struct term_node *node, WINDOW *ifaces_window, 
        PANEL *ifaces_panel)
{
    u_int8_t i, end, change;
    int32_t key_pressed, j;
    u_int8_t row, col, y, x;

    end = 0;
    change = 1;

    ncurses_gui_set_status_line(" Interfaces to the world ");

    /* Check that the window is centered */
    getmaxyx(stdscr, row, col);
    getyx(ifaces_window, y, x);
    if ((row - 10 != y) || (col - 50 != x))
        mvwin(ifaces_window, (row - 10)/2, (col - 50)/2);

    wtimeout(ifaces_window,NCURSES_KEY_TIMEOUT); /* Block for 100 millisecs...*/
    
    while(!end && !terms->gui_th.stop) 
    {
        i = 0;
        if (change) {
            wclear(ifaces_window);
            wattron(ifaces_window, COLOR_PAIR(4));
            box(ifaces_window, 0, 0);

            mvwprintw(ifaces_window, 0, 15, " Global Interfaces ");
            mvwprintw(ifaces_window, 9, 17, " Press q to exit ");
            while( i < MAX_INTERFACES) {
                if (strlen(interfaces[i].ifname)) {
                    if (node->used_ints[i] == 1) 
                        wattron(ifaces_window, COLOR_PAIR(4));
                    else
                        wattroff(ifaces_window, COLOR_PAIR(4));
                    mvwprintw(ifaces_window, i+2, 2, "%c) %s (%s)", i+97, 
                            interfaces[i].ifname, node->used_ints[i] ? "ON" : "OFF");
                }
                i++;
            }
            change = 0;
        }

        do
        {
           key_pressed = wgetch(ifaces_window);
        } while ( (key_pressed == ERR) && !terms->gui_th.stop);
        
        if (!terms->gui_th.stop)
        {
            switch(key_pressed) {
                case 27: /* ESC */
                case 'Q':
                case 'q':
                    end = 1;
                break;
                default:
                    j = key_pressed - 97;
                    if ((0 <= j) && (j < MAX_INTERFACES) && (strlen(interfaces[j].ifname))) {
                        if (node->used_ints[j] == 1) {
                            interfaces_del(j);
                            node->used_ints[j] = 0;
                            change = 1;
                        }
                        else {
                            interfaces_add(interfaces[j].ifname);
                            node->used_ints[j] = 1;
                            change = 1;
                        }
                    }
                break;
            }
        }
    } /* while...*/

    wattroff(ifaces_window, COLOR_PAIR(4));

    hide_panel(ifaces_panel);

    ncurses_gui_set_status_line("");
}


/* 
 * Display information about the selected item
 */
int8_t
ncurses_gui_show_info(u_int8_t mode, WINDOW *main_window, u_int8_t pointer, struct term_node *node)
{
    u_int8_t i, j, line, row, col, k, jfields;
    int32_t key_pressed;
    WINDOW *info_window;
    char **values, *ptrtlv, *top;
    struct position_fields *fields;
    struct tuple_type_desc *func;
    
    func = NULL;

    ncurses_gui_set_status_line(" Information should be free ");

    getmaxyx(main_window,row,col);
    info_window = newpad(MAX_PAD_HEIGHT, MAX_PAD_WIDTH);
    keypad(info_window, TRUE);

    i = 0;
    line = 0;
    jfields = 0;
    fields = protocols[mode].fields;
    
    /* OK, I already know that this is a weird way to do it, but 
     * I do not have enough free time to fix it :P */
    wattron(info_window, COLOR_PAIR(4));
    wattron(main_window, COLOR_PAIR(4));
    mvwaddch(main_window, (row-INFO_HEIGHT)/2 - 1, (col-INFO_WIDTH)/2 - 1, ACS_ULCORNER);
    mvwaddch(main_window, (row-INFO_HEIGHT)/2 - 1, (col-INFO_WIDTH)/2 + INFO_WIDTH, ACS_URCORNER);
    mvwaddch(main_window, (row-INFO_HEIGHT)/2 + INFO_HEIGHT, (col-INFO_WIDTH)/2 + INFO_WIDTH, ACS_LRCORNER);
    mvwaddch(main_window, (row-INFO_HEIGHT)/2 + INFO_HEIGHT, (col-INFO_WIDTH)/2 - 1, ACS_LLCORNER);
    mvwaddch(main_window, (row-INFO_HEIGHT)/2 - 1, (col-INFO_WIDTH)/2 + INFO_WIDTH, ACS_URCORNER);
    mvwhline(main_window, (row-INFO_HEIGHT)/2 - 1, (col-INFO_WIDTH)/2, ACS_HLINE, INFO_WIDTH);
    mvwvline(main_window, (row-INFO_HEIGHT)/2, (col-INFO_WIDTH)/2 - 1, ACS_VLINE, INFO_HEIGHT);
    mvwhline(main_window, (row-INFO_HEIGHT)/2 + INFO_HEIGHT, (col-INFO_WIDTH)/2, ACS_HLINE, INFO_WIDTH);
    mvwvline(main_window, (row-INFO_HEIGHT)/2, (col-INFO_WIDTH)/2 + INFO_WIDTH, ACS_VLINE, INFO_HEIGHT);
    mvwprintw(main_window, (row-INFO_HEIGHT)/2 + INFO_HEIGHT, (col-INFO_WIDTH)/2 + 4 , " q,ENTER: exit  Up/Down: scrolling ");
    wrefresh(main_window);

    j = 0;
    while(fields[j].name) {
	if (fields[j].type != FIELD_TLV)
	  mvwprintw(info_window, j + 0, 2, "%15s", fields[j].name);
	j++;
    }

    wattroff(info_window, COLOR_PAIR(4));

    if (protocols[mode].get_printable_packet) {
        if ((values = (*protocols[mode].get_printable_packet)(&protocols[mode].stats[pointer])) == NULL) {
            write_log(0, "Error in get_printable_packet (mode %d)\n", mode);
            return -1;
        }
    }
    else {
        write_log(0, "Warning: there is no get_printable_packet for protocol %d\n", mode);
        return -1;
    }

    j = 0;
    while(fields[j].name) 
    {
        if (fields[j].type != FIELD_TLV) 
        {
            write_log(0, "Copiando el campo %d: %s: %s\n", j, fields[j].name, values[j]);
            mvwprintw(info_window, j, 19, "%s %s", values[j], 
                             (fields[j].desc) ? (*protocols[mode].get_info)(j, values[j]) : "");
        }
        else 
           if (fields[j].mwindow == 0) 
           { /* TLV stuff */
                ptrtlv = values[j];
                top = ptrtlv + (2*MAX_TLV*MAX_VALUE_LENGTH);

                jfields = j;
                while((ptrtlv < top) && (strlen((char *)ptrtlv) > 0)) 
                {
                    mvwprintw(info_window, jfields, 2, "%15s", ptrtlv);

                    k = j + 1;

                    /* Check if we've got fields description */
                    while(fields[k].name) 
                    {
                        if (strncmp((char *)ptrtlv, (char *)fields[k].name, strlen((char *)fields[k].name)) == 0)
                            func = (struct tuple_type_desc *)fields[k].desc;
                        k++;
                    }

                    ptrtlv += strlen((char *)ptrtlv) + 1;
                    mvwprintw(info_window, jfields, 19, "%s %s", ptrtlv, 
                                      (func) ? (*protocols[mode].get_info)(fields[k-1].v, ptrtlv) : "");
                    ptrtlv += strlen((char *)ptrtlv) + 1;
                    jfields++;
                }
           }
	   j++;
    }

    wattron(info_window, COLOR_PAIR(4));
    mvwprintw(info_window, (jfields > 0) ? jfields: j, 2, "%15s", "Interface");
    wattroff(info_window, COLOR_PAIR(4));
    mvwprintw(info_window, (jfields > 0) ? jfields: j, 19,"%s",  
              interfaces[protocols[mode].stats[pointer].iface].ifname);

    wtimeout(info_window,NCURSES_KEY_TIMEOUT); 
    
    do
    {
        prefresh(info_window, line, 0, (row-INFO_HEIGHT)/2, (col-INFO_WIDTH)/2, 
                  (row-INFO_HEIGHT)/2 + INFO_HEIGHT - 1, (col-INFO_WIDTH)/2 + INFO_WIDTH - 1);
        do
        {
           key_pressed = wgetch(info_window);
        } while( (key_pressed == ERR) && !terms->gui_th.stop);

        switch(key_pressed) {
            case KEY_UP:
                if (line > 0) 
                    line--;
            break;
            case KEY_DOWN:
                if (line < INFO_HEIGHT)
                  line++;
            break;
        }

    } while(!terms->gui_th.stop && (key_pressed != 13) &&
             (key_pressed!='q') && (key_pressed!='Q'));
    
    delwin(info_window);
    wclear(main_window);

    ncurses_gui_set_status_line("");

    return 0;
}

/* 
 * Display packet
 * Taken from print-ascii.c (tcpdump)
 * http://www.tcpdump.org
 */
int8_t
ncurses_gui_view_packet(u_int8_t mode, WINDOW *main_window, u_int8_t pointer)
{
    u_int8_t *packet;
    u_int16_t length, oset;
    int32_t j, key_pressed, line, row, col;
    WINDOW *view_window;
    register u_int i;
    register int s1, s2;
    register int nshorts;
    char hexstuff[HEXDUMP_SHORTS_PER_LINE*HEXDUMP_HEXSTUFF_PER_SHORT+1], *hsp;
    char asciistuff[ASCII_LINELENGTH+1], *asp;
    u_int32_t maxlength = HEXDUMP_SHORTS_PER_LINE;
   
    ncurses_gui_set_status_line(" Displaying packet ");

    getmaxyx(main_window,row,col);
    view_window = newpad(MAX_PAD_HEIGHT, MAX_PAD_WIDTH);
    keypad(view_window, TRUE);

    j = 0;
    line = 0;
    oset = 0;
    length = 0;
    packet = NULL;
    
    /* OK, I already know that this is a weird way to do it, but 
     * I do not have enough free time to fix it :P */
    wattron(info_window, COLOR_PAIR(4));
    wattron(main_window, COLOR_PAIR(4));
    mvwaddch(main_window, (row-VIEW_HEIGHT)/2 - 1, (col-VIEW_WIDTH)/2 - 1, ACS_ULCORNER);
    mvwaddch(main_window, (row-VIEW_HEIGHT)/2 - 1, (col-VIEW_WIDTH)/2 + VIEW_WIDTH, ACS_URCORNER);
    mvwaddch(main_window, (row-VIEW_HEIGHT)/2 + VIEW_HEIGHT, (col-VIEW_WIDTH)/2 + VIEW_WIDTH, ACS_LRCORNER);
    mvwaddch(main_window, (row-VIEW_HEIGHT)/2 + VIEW_HEIGHT, (col-VIEW_WIDTH)/2 - 1, ACS_LLCORNER);
    mvwaddch(main_window, (row-VIEW_HEIGHT)/2 - 1, (col-VIEW_WIDTH)/2 + VIEW_WIDTH, ACS_URCORNER);
    mvwhline(main_window, (row-VIEW_HEIGHT)/2 - 1, (col-VIEW_WIDTH)/2, ACS_HLINE, VIEW_WIDTH);
    mvwvline(main_window, (row-VIEW_HEIGHT)/2, (col-VIEW_WIDTH)/2 - 1, ACS_VLINE, VIEW_HEIGHT);
    mvwhline(main_window, (row-VIEW_HEIGHT)/2 + VIEW_HEIGHT, (col-VIEW_WIDTH)/2, ACS_HLINE, VIEW_WIDTH);
    mvwvline(main_window, (row-VIEW_HEIGHT)/2, (col-VIEW_WIDTH)/2 + VIEW_WIDTH, ACS_VLINE, VIEW_HEIGHT);
    mvwprintw(main_window, (row-VIEW_HEIGHT)/2 + VIEW_HEIGHT, (col-VIEW_WIDTH)/2 + 16 , " q,ENTER: exit  Up/Down: scrolling ");
    wrefresh(main_window);

    packet = protocols[mode].stats[pointer].packet;
    length = (protocols[mode].stats[pointer].header->len < SNAPLEN) ? protocols[mode].stats[pointer].header->len : SNAPLEN;

    nshorts = length / sizeof(u_int16_t);
    i = 0;
    hsp = hexstuff; asp = asciistuff;
    while (--nshorts >= 0) {
            s1 = *packet++;
            s2 = *packet++;

            (void)snprintf(hsp, sizeof(hexstuff) - (hsp - hexstuff),
                " %02x%02x", s1, s2);
            hsp += HEXDUMP_HEXSTUFF_PER_SHORT;
            *(asp++) = (isgraph(s1) ? s1 : '.');
            *(asp++) = (isgraph(s2) ? s2 : '.');
            i++;

            if (i >= maxlength) {
                    *hsp = *asp = '\0';
                    mvwprintw(view_window, j, 1, "0x%04x: %-*s  %s",
                         oset, HEXDUMP_HEXSTUFF_PER_LINE,
                         hexstuff, asciistuff);
                    i = 0; hsp = hexstuff; asp = asciistuff;
                    oset += HEXDUMP_BYTES_PER_LINE;
                    j++;
            }
    }

    if (length & 1) {
            s1 = *packet++;
               (void)snprintf(hsp, sizeof(hexstuff) - (hsp - hexstuff),
                   " %02x", s1);
                   hsp += 3;
                   *(asp++) = (isgraph(s1) ? s1 : '.');
            ++i;
    }
    if (i > 0) {
            *hsp = *asp = '\0';
            (void)mvwprintw(view_window, j, 1, "0x%04x: %-*s  %s",
                  oset, HEXDUMP_HEXSTUFF_PER_LINE,
                  hexstuff, asciistuff);
    }

    wtimeout(view_window, NCURSES_KEY_TIMEOUT); 
    
    do
    {
        prefresh(view_window, line, 0, (row-VIEW_HEIGHT)/2, (col-VIEW_WIDTH)/2, (row-VIEW_HEIGHT)/2 + VIEW_HEIGHT - 1, (col-VIEW_WIDTH)/2 + VIEW_WIDTH - 1);
        do
        {
           key_pressed = wgetch(view_window);
        } while( (key_pressed == ERR) && !terms->gui_th.stop);

        switch(key_pressed) {
            case KEY_UP:
                if (line > 0) 
                    line--;
            break;
            case KEY_DOWN:
                if (line < VIEW_HEIGHT)
                  line++;
            break;
        }

    } while(!terms->gui_th.stop && (key_pressed != 13) &&
             (key_pressed!='q') && (key_pressed!='Q'));
    
    delwin(view_window);
    wclear(main_window);

    ncurses_gui_set_status_line("");

    return 0;
}


/* 
 * List attack
 */
void 
ncurses_gui_list_attacks(WINDOW *list_window, struct term_node *node)
{
    u_int8_t end, kill, j, pointer, a, i, used, indx=0, files[MAX_PROTOCOLS*MAX_THREAD_ATTACK][2];
    int32_t key_pressed;
    struct attack *theattack = NULL;
    
    used = 0; j = 0;

    for (a=0; a<MAX_PROTOCOLS; a++)
    {
        for (i=0; i<MAX_THREAD_ATTACK; i++)
        {
            if (node->protocol[a].attacks[i].up) 
            {
               files[j][0]=a; /* Protocol */
               files[j][1]=i; /* Attack used */
               j++;
               used++;
            }
        }
    }
    
    ncurses_gui_set_status_line(" Listing current attacks... ");

    pointer = 0;
    kill = 0;
    end = 0;

    wclear(list_window);

    wattron(list_window, COLOR_PAIR(4));
    box(list_window, 0, 0);
    mvwprintw(list_window, 0, 2, " Running attacks ");
    mvwprintw(list_window, 2, 2, " Protocol  Type    Description");
    mvwprintw(list_window, 19, 2, " Press ENTER to cancel an attack or 'q' to quit");
    wattroff(list_window, COLOR_PAIR(4));

    keypad(list_window, TRUE);

    wtimeout(list_window,NCURSES_KEY_TIMEOUT); /* Block for 100 millisecs...*/

    while (!end && !terms->gui_th.stop) 
    {
        i = 0;
        while (!terms->gui_th.stop && (i < used)) 
        {
            /* Kill the attack thread selected */
            if (kill)
            {
                if (node->protocol[files[indx][0]].attacks[files[indx][1]].up)
                {
                    attack_kill_th(node, node->protocol[files[indx][0]].attacks[files[indx][1]].attack_th.id);
                    break;
                }
            }

            if (i == pointer)
            {
                wattron(list_window, COLOR_PAIR(1));
                indx=i;
            }
            else
                wattroff(list_window, COLOR_PAIR(1));

            if (node->protocol[files[i][0]].attacks[files[i][1]].up) 
            {
                theattack = protocols[files[i][0]].attacks;

                mvwprintw(list_window, i+4, 2, "%8s   %-2d      %s", 
                        protocols[files[i][0]].name, 
                        node->protocol[files[i][0]].attacks[files[i][1]].attack, 
                        theattack[node->protocol[files[i][0]].attacks[files[i][1]].attack].s);
            }
            i++;
        }

        wrefresh(list_window);

        if (kill)
           break;  

        do
        {
           key_pressed = wgetch(list_window);
        } while( (key_pressed == ERR) && !terms->gui_th.stop);
        
        if (!terms->gui_th.stop)
        {
            switch(key_pressed) 
            {
                case KEY_DOWN:
                    if ( pointer < (used-1)) 
                        pointer++;
                break;
                
                case KEY_UP:
                    if (pointer > 0) 
                        pointer--;
                break; 

                case 'Q':
                case 'q':
                    end = 1;
                break;
                case 13: /* ENTER */
                    kill = 1;
                break;
            }
        }
    }

    ncurses_gui_set_status_line("");
}




/* 
 * List capture files
 */
void 
ncurses_gui_list_filecaps(WINDOW *list_window, struct term_node *node)
{
    int32_t key_pressed;
    u_int8_t end, kill, j, pointer, i, used, indx=0, files[MAX_PROTOCOLS+2];
    
    used = 0; j = 0;
    
    for (i=0; i<MAX_PROTOCOLS; i++)
    {
        if (node->protocol[i].pcap_file.pdumper)
        {
           files[j]=i;
           j++;
           used++;
        }
    }
    
    if (node->pcap_file.pdumper)
    {
       files[j]=PROTO_ALL;
       used++;
    }

    ncurses_gui_set_status_line(" Listing current capture files... ");

    pointer = 0;
    kill = 0;
    end = 0;
    
    wclear(list_window);

    wattron(list_window, COLOR_PAIR(4));
    box(list_window, 0, 0);
    mvwprintw(list_window, 0, 2, " Capture files ");
    mvwprintw(list_window, 2, 2, "Protocol  Filename");
    mvwprintw(list_window, 19, 2, " Press ENTER to cancel a capture file or 'q' to quit");
    wattroff(list_window, COLOR_PAIR(4));

    keypad(list_window, TRUE);

    wtimeout(list_window,NCURSES_KEY_TIMEOUT);
    
    while (!end && !terms->gui_th.stop) 
    {
        i = 0;

        while (!terms->gui_th.stop && (i < used) ) 
        {
            if (kill)
            {
                if (files[indx] == PROTO_ALL)
                    interfaces_pcap_file_close(node,PROTO_ALL);
                else
                    interfaces_pcap_file_close(node,files[indx]);
                break;
            }        

            if (i == pointer)
            {
                wattron(list_window, COLOR_PAIR(1));
                indx=i;
            }
            else
                wattroff(list_window, COLOR_PAIR(1));

            if (files[i] == PROTO_ALL)
            {
                mvwprintw(list_window, i+4, 2, "  %-6s  %s", 
                        "ALL", 
                        node->pcap_file.name);
            }
            else
            {
                mvwprintw(list_window, i+4, 2, "  %-6s  %s", 
                        protocols[files[i]].name, 
                        node->protocol[files[i]].pcap_file.name);
            } 
            i++;
        }

        wrefresh(list_window);

        if (kill)
           break;  

        do
        {
           key_pressed = wgetch(list_window);
        } while( (key_pressed == ERR) && !terms->gui_th.stop);
        
        if (!terms->gui_th.stop)
        {
            switch(key_pressed) 
            {
                case KEY_DOWN:
                    if ( pointer < (used-1)) 
                        pointer++;
                break;
                
                case KEY_UP:
                    if (pointer > 0) 
                        pointer--;
                break; 

                case 'Q':
                case 'q':
                    end = 1;
                break;
                case 13: 
                    kill = 1;
                break;
            }
        }
    }

    ncurses_gui_set_status_line("");
}


int8_t
ncurses_gui_learn_packet(u_int8_t mode, u_int8_t pointer, struct term_node *node)
{
    ncurses_gui_set_status_line(" Packet learnt. Ouch! ");

    if (protocols[mode].load_values)
        return (*protocols[mode].load_values)((struct pcap_data *)&protocols[mode].stats[pointer], 
                            node->protocol[mode].tmp_data);
    else {
        write_log(0, "Warning: no load_values in protocol %d\n", mode);
        return -1;
    }
}


int8_t
ncurses_gui_error_window(u_int8_t mode, char *message, ...)
{
    int32_t row, col, i, message_s;
    int32_t max_y, max_x;
    va_list argp;
    WINDOW *my_window;
    PANEL *my_panel;
    char *ptr, *m_split;
    char *vmessage;

    ncurses_gui_set_status_line(" You've got a message ");

    getmaxyx(stdscr,row,col);

    /* Max message size is ERRORMSG_SIZE bytes */
    if ((vmessage = (char *) malloc(ERRORMSG_SIZE)) == NULL) {
        thread_error("Error in malloc", errno);
        return -1;
    }

    memset(vmessage, 0, ERRORMSG_SIZE);
    va_start(argp, message);
    if (vsprintf(vmessage, message, argp) < 0) {
        thread_error("Error in vsprintf", errno);
        return -1;
    }
    va_end(argp);

    i = 1;
    ptr = vmessage;
    if ((message_s = strlen(vmessage)) > ERRORMSG_SIZE)
        return -1;
    /* by default half columns than terminal columns */
    max_y = col / 2;
    /* as many rows as needed to fit the message (estimated) */
    max_x = (message_s / (col / 2)) + 4;
    if ((m_split = (char *) malloc(max_y - 3)) == NULL) {
        thread_error("Error in malloc", errno);
        return -1;
    }
    my_window = newwin(max_x, max_y, (row-max_x)/2, (col-max_y)/2);
    my_panel = new_panel(my_window);

    wattron(my_window, COLOR_PAIR(3));
    box(my_window, 0, 0);
    switch(mode) {
        case 0:
            mvwprintw(my_window, 0, 2, " Notification window ");
            write_log(0, " Notification: %s\n", vmessage);
            break;
        case 1:
            mvwprintw(my_window, 0, 2, " Error window ");
            write_log(0, " Error: %s\n", vmessage);
            break;
    }

    mvwprintw(my_window, max_x - 1, 2, " Press any key to continue ");
    wattroff(my_window, COLOR_PAIR(3));
    while (message_s > 0) {
        /* split the message */
        if (message_s >= max_y - 4) {
            strncpy(m_split, ptr, max_y - 4);
            m_split[max_y-4] = '\0';
            mvwprintw(my_window, i, 2, m_split);
            message_s -= max_y - 4;
            ptr += max_y - 4;
         /* offset */
         } else {
            strncpy(m_split, ptr, message_s);
            m_split[message_s] = '\0';
            mvwprintw(my_window, i, 2, m_split);
            message_s = 0;
        }
        i++;
    }

    free(m_split);
    free(vmessage);
    
    update_panels();
    if (doupdate() == ERR)
        return -1;

    wtimeout(my_window, NCURSES_KEY_TIMEOUT);
    while ((wgetch(my_window)==ERR) && !terms->gui_th.stop);
   
    if (del_panel(my_panel) == ERR)
        return -1;
    if (delwin(my_window) == ERR)
        return -1;

    ncurses_gui_set_status_line("");

    return 0;
}



int8_t
ncurses_gui_getstring_window(struct term_node *term, char *status, char *data, u_int8_t max, char *message)
{
    int32_t row, col;
    int32_t max_y, max_x;
    WINDOW *my_window;
    PANEL *my_panel;

    ncurses_gui_set_status_line(status);

    getmaxyx(stdscr,row,col);

    max_y = max+3;
    max_x = 5;

    my_window = newwin(max_x, max_y, (row-max_x)/2, (col-max_y)/2);
    my_panel = new_panel(my_window);

    wattron(my_window, COLOR_PAIR(3));
    box(my_window, 0, 0);

    mvwprintw(my_window, 0, 2, message);

    mvwprintw(my_window, max_x - 1, 2, " Press Enter to continue ");
    wattroff(my_window, COLOR_PAIR(3));

    wmove(my_window, max_x - 3, 1);
       
    echo();
    curs_set(2);
    wgetnstr(my_window, data, max);
    noecho();
    curs_set(0);    
    
    update_panels();

    doupdate();
    del_panel(my_panel);
    delwin(my_window);

    ncurses_gui_set_status_line("");

    return 0;
}

int32_t
ncurses_gui_getconfirm(struct term_node *term, char *status, char *message, char *title)
{
    int32_t row, col, key_pressed=0, max_y, max_x;
    u_int8_t end=0;
    char *bottom = " 'Y' to confirm  -  'N' to abort ";
    WINDOW *my_window;
    PANEL *my_panel;

    ncurses_gui_set_status_line(status);

    getmaxyx(stdscr,row,col);

    max_x = strlen(message)+2;

    if (strlen(bottom)>max_x)
       max_x = strlen(bottom);
       
    max_y = 5;

    my_window = newwin(max_y, max_x, (row-max_y)/2, (col-max_x)/2);
    my_panel = new_panel(my_window);

    noecho();
    curs_set(0);

    wattron(my_window, COLOR_PAIR(3));
    box(my_window, 0, 0);

    mvwprintw(my_window, 0, 2, title);

    mvwprintw(my_window, max_y - 1, 2, bottom);
    
    wattroff(my_window, COLOR_PAIR(3));
    
    mvwprintw(my_window, max_y - 3, 1, message);

    wtimeout(my_window,NCURSES_KEY_TIMEOUT); /* Block for 100 millisecs...*/

    while (!end && !terms->gui_th.stop)
    {
	do
	{
	   key_pressed = wgetch(my_window);
	} while ( (key_pressed == ERR) && !terms->gui_th.stop);
	
	if (terms->gui_th.stop)
	   break;

	switch(key_pressed)
	{
            case 'y':
            case 'Y':
                key_pressed = 'y';
                end=1;
            break;
            
            case 'n':
            case 'N':
                key_pressed = 'n';
		end = 1;
            break;
        }
    }
    
    update_panels();
    doupdate();
    del_panel(my_panel);
    delwin(my_window);

    ncurses_gui_set_status_line("");

    return key_pressed;
}


int8_t
ncurses_gui_term_resize(struct term_node *node)
{
#ifdef CAN_RESIZE
    struct winsize size;
    struct term_console *term_console;

    term_console = node->specific;

    if (ioctl(fileno(stdout), TIOCGWINSZ, &size) == 0) {
        if ((size.ws_row < NCURSES_MIN_ROWS) || (size.ws_col < NCURSES_MIN_COLS))
            ncurses_gui_error_window(1, "Hmmm.. %d rows and %d cols are \
                    not supported for a proper display!!!!. You need at least \
                    %d rows and %d cols.", size.ws_row, size.ws_col, 
                    NCURSES_MIN_ROWS, NCURSES_MIN_COLS);
        else {
            resize_term(size.ws_row, size.ws_col);
            wrefresh(curscr);   /* Linux needs this */
            term_console->need_resize = 2;

        }
    } else
        thread_error("Error in ioctl", errno);

#endif

    return 0;
}


int32_t 
ncurses_gui_popup_window(WINDOW *bwindow, struct position_fields *fields, u_int8_t state, u_int8_t y, u_int8_t x)
{
    WINDOW *win;
    int32_t i, j, pointer, end, result, key_pressed, row, col;
    const struct tuple_type_desc *tuple;
    
    pointer = 0;
    end = 0;
    result = ERR;
    tuple = (const struct tuple_type_desc *)fields[state].desc;

    getmaxyx(stdscr, row, col);

    if ((win = derwin(stdscr, 5, 22, row-NCURSES_BWINDOW_SIZE, 20)) == NULL)
        return ERR;

    keypad(win, TRUE);
//    scrollok(win, TRUE);
//    idlok(win, TRUE);

    while (!terms->gui_th.stop && !end) {
        j = 0;
        i = pointer;
        /* Show max 3 options */
        while(tuple[i].desc != NULL && j <= 2) {
            if (i == pointer)
                wattron(win, COLOR_PAIR(5) | A_BOLD);

            wmove(win, j+1, 1);
            whline(win, ' ', 20);

            mvwprintw(win, j+1, 1, "%s", tuple[i].desc);
            if (i == pointer)
                wattroff(win, A_BOLD);

            i++; j++;
        }

        box(win, 0, 0);
        wtimeout(win,NCURSES_KEY_TIMEOUT); /* Block for 100 millisecs...*/

        key_pressed = wgetch(win);
        switch(key_pressed) {
            case KEY_DOWN:
                pointer = (tuple[pointer+1].desc) ? (pointer+1) : pointer;
                break;
            case KEY_UP:
                pointer = (pointer > 0) ? (pointer-1) : 0;
                break;
            case 13: /* ENTER */
                result = tuple[pointer].type;
                end = 1;
                break;
            case 27: /* ESC */
                end = 1;
                break;
        }
        werase(win);
    }

    wrefresh(bwindow);

    if (delwin(win) == ERR)
        return ERR;

    return result;
}    

int8_t
ncurses_gui_set_status_line(char *msg)
{
    u_int8_t row, col, size, i;

    getmaxyx(stdscr, row, col);
    size = strlen(msg);

    wmove(info_window, 0, 1);
    /* Clear the last status line */
    for (i=0; i < col; i++)
        waddch(info_window, ' ');

    if (size) {
        wattron(info_window, COLOR_PAIR(3));

        if (size > row)
            mvwaddnstr(info_window, 0, 0, msg, row - 1);
        else 
            mvwaddnstr(info_window, 0, 0, msg, size);
    }

    /* Nice */
//    wmove(info_window, 0, 1);
    wrefresh(info_window);

    return 0;
}


int8_t
ncurses_gui_edit_tlv(struct term_node *node, u_int8_t mode)
{
    u_int8_t **values = NULL;
    WINDOW *win;
    int32_t pointer, end, key_pressed, row, col, modified;
    u_int8_t j, jfields, *top, *ptrtlv;
    struct position_fields *fields;

    pointer = 0;
    end = 0;
    modified = 1;

    fields = protocols[mode].fields;

    getmaxyx(stdscr, row, col);

    if ((win = derwin(stdscr, 23, 41, (row-23)/2, (col-41)/2)) == NULL)
        return ERR;

    keypad(win, TRUE);
    curs_set(0);
   
    while (!terms->gui_th.stop && !end) {
	if (modified) {
	    if ((values = (u_int8_t **)(*protocols[mode].get_printable_store)(node)) == NULL) {
		write_log(0, "Error in get_printable_store (mode %d)\n", mode);
		return -1;
	    }

	    modified = 0;
	    wclear(win);
	    wrefresh(win);
	}

	    j = 0;
	    jfields = 0;
	    while(fields[j].name) 
	    {
	       if ((fields[j].type == FIELD_TLV) && (fields[j].mwindow == 0)) 
	       { /* TLV stuff */
		    ptrtlv = values[j];
		    top = ptrtlv + (2*MAX_TLV*MAX_VALUE_LENGTH);

		    while((ptrtlv < top) && (strlen((char *)ptrtlv) > 0)) 
		    {
			if (jfields == pointer) 
			    wattron(win, COLOR_PAIR(4));
			else
			    wattroff(win, COLOR_PAIR(4));

			mvwprintw(win, jfields + 1, 2, "%15s", ptrtlv);
			ptrtlv += strlen((char *)ptrtlv) + 1;

			mvwprintw(win, jfields + 1, 19, "%s", ptrtlv);
			ptrtlv += strlen((char *)ptrtlv) + 1;
			jfields++;
		    }
	       }
	       j++;
	    }


        wattron(win, COLOR_PAIR(4));
        box(win, 0, 0);
        mvwprintw(win, 22, 7, "q: EXIT, a: ADD, d:DEL");
        wtimeout(win, NCURSES_KEY_TIMEOUT); /* Block for 100 millisecs...*/

        key_pressed = wgetch(win);
        switch(key_pressed) {
            case KEY_DOWN:
                pointer = (pointer < jfields - 1) ? (pointer+1) : pointer;
            break;
            case KEY_UP:
                pointer = (pointer > 0) ? (pointer-1) : 0;
            break;
            /* Add TLV */
            case 'a':
            case 'A':
                    ncurses_gui_add_selected_tlv_type(win, node, mode); 
		    modified = 1;
            break;
            /* Delete TLV */
            case 'd':
            case 'D':
                    ncurses_gui_del_selected_tlv(node, mode, pointer);
                    modified = 1;
            break;
            case 'q':
            case 'Q':
                end = 1;
            break;
            case 27: /* ESC */
                end = 1;
            break;
        }
    }

    werase(win);

    j = 0;
    if (values) 
    {
	while(values[j]) 
	{
	    free(values[j]);
	    j++;
	}
	free(values);
    }

    if (delwin(win) == ERR)
        return ERR;

    return 0;
}


int8_t
ncurses_gui_add_selected_tlv_type(WINDOW *win, struct term_node *node, u_int8_t mode)
{
    int32_t i, j, pointer, end, key_pressed;
    struct attack_param *tlv, *attack_param;
    struct tuple_type_desc *type_desc;
    u_int16_t type;
    u_int8_t field;
    int8_t ret;

    pointer = 0;
    end = 0;
    type = 0;
    i = 0;
    key_pressed = 0;
    attack_param = NULL;

    if (protocols[mode].ttd)
	type_desc = (struct tuple_type_desc *)protocols[mode].ttd;
    else {
	write_log(0, "Warning: there is no ttd (mode %d)\n", mode);
	return -1;
    }

    if (protocols[mode].tlv)
	tlv = (struct attack_param *)protocols[mode].tlv;
    else {
	write_log(0, "Warning: there is no tlv (mode %d)\n", mode);
	return -1;
    }

    while (!terms->gui_th.stop && !end) {
        werase(win);
        j = 0;
        i = 0;
        while((i < protocols[mode].tlv_params) && (tlv[i].type != FIELD_NONE)) {
            if (i == pointer)
                wattron(win, COLOR_PAIR(5) | A_BOLD);

            wmove(win, j+1, 1);
            whline(win, ' ', 20);

            mvwprintw(win, j+1, 1, "%s", tlv[i].desc);
            if (i == pointer)
                wattroff(win, A_BOLD);

            i++; j++;
        }

        box(win, 0, 0);
        mvwprintw(win, 22, 7, "Press ENTER to add the selected TYPE");
        wtimeout(win,NCURSES_KEY_TIMEOUT); /* Block for 100 millisecs...*/

        key_pressed = wgetch(win);
        switch(key_pressed) {
            case KEY_DOWN:
                pointer = (tlv[pointer+1].type != FIELD_NONE) ? (pointer+1) : pointer;
                break;
            case KEY_UP:
                pointer = (pointer > 0) ? (pointer-1) : 0;
                break;
            case 13: /* ENTER */
		/* Find the type */
		i = 0;
		while (type_desc[i].desc) {
		    if (strcmp(type_desc[i].desc, tlv[pointer].desc) == 0) {
                        memcpy((void *)&type, (void *)&type_desc[i].type, 2);
			break;
		    }
		    i++;
		}
                end = 1;
                break;
            case 27: /* ESC */
                end = 1;
                break;
        }
    }

    write_log(0, "tipo es %hd e i es %hd y pointer es %hd\n", type, i, pointer);
    /* Set up the param... */
    if (type) {
	if ((attack_param = calloc(1, (sizeof(struct attack_param)))) == NULL)
	{
	   thread_error(" ncurses_gui_attack_screen attack_param calloc",errno);
	   key_pressed='Q';
	   return -1;
	}

	memcpy( attack_param, (void *)(&tlv[pointer]), sizeof(struct attack_param));

	if (attack_init_params(node, attack_param, 1) < 0)
	{
	   free(attack_param);
	   key_pressed='Q';
	   return -1;
	}
	
	/* Now we can ask for parameters... */
	ncurses_gui_get_printable_store(attack_param, 1);
     
	do {
	    if (ncurses_gui_attack_get_params(attack_param, 1) < 0) /* Q pressed */
	    {
	      attack_free_params(attack_param, 1);
	      free(attack_param);
	      key_pressed='Q';
	      return -1;
	    }
	    ret = attack_filter_all_params(attack_param, 1, &field);
	    if ( ret == -1) /* Error on data...*/
	    {
	      ncurses_gui_error_window(1, "Bad data on field '%s'!!",
				       attack_param[field].desc);
	    }
	} while(ret==-1);
	
	if (key_pressed == 'Q')
	{
	    i = 0;
	    wclear(win);
	    wattron(win, COLOR_PAIR(1));
	    box(win, 0, 0);
	    key_pressed = 0;
	}
    }

    if (protocols[mode].edit_tlv) {
	if ((*protocols[mode].edit_tlv)(node, TLV_ADD, 0, type, attack_param->value) < 0) {
	    write_log(0, "Error in edit_tlv (mode %d)\n", mode);
	    return -1;
	}
    } else {
	write_log(0, "Warning: there is no edit_tlv (mode %d)\n", mode);
	return -1;
    }


    return 0;
}


int8_t
ncurses_gui_del_selected_tlv(struct term_node *node, u_int8_t mode, u_int8_t pointer)
{
    if (protocols[mode].edit_tlv) {
	if ((*protocols[mode].edit_tlv)(node, TLV_DELETE, pointer, 0, NULL) < 0) {
	    write_log(0, "Error in edit_tlv (mode %d)\n", mode);
	    return -1;
	}
    } else {
	write_log(0, "Warning: there is no edit_tlv (mode %d)\n", mode);
	return -1;
    }

    return 0;
}



void
ncurses_gui_get_printable_store(struct attack_param *attack_param, u_int8_t nparams)
{
    u_int8_t i, *aux_char;
    u_int16_t *aux_short;
    u_int32_t *aux_long;
    
    for (i=0; i< nparams; i++)
    {
        switch(attack_param[i].type)
        {
            case FIELD_BRIDGEID:
                aux_char = (u_int8_t *)attack_param[i].value;
                snprintf(attack_param[i].print, 18, "%02X%02X.%02X%02X%02X%02X%02X%02X",
                         aux_char[0], aux_char[1], aux_char[2], aux_char[3],
                         aux_char[4], aux_char[5], aux_char[6], aux_char[7]);
            break;

            case FIELD_MAC:
                aux_char = (u_int8_t *)attack_param[i].value;
                snprintf(attack_param[i].print, 18, "%02X:%02X:%02X:%02X:%02X:%02X",
                         aux_char[0], aux_char[1], aux_char[2],
                         aux_char[3], aux_char[4], aux_char[5]); 
            break;

            case FIELD_HEX:
            case FIELD_DEC:
                switch(attack_param[i].size)
                {
                    case 1:
                       aux_char = (u_int8_t *)attack_param[i].value;
                       if (attack_param[i].type == FIELD_HEX)
                          snprintf(attack_param[i].print, attack_param[i].size_print+1, "%02X", *(aux_char));
                       else
                          snprintf(attack_param[i].print, attack_param[i].size_print+1, "%02d", *(aux_char));
                    break;
                    case 2:
                       aux_short = (u_int16_t *)attack_param[i].value;
                       if (attack_param[i].type == FIELD_HEX)                       
                          snprintf(attack_param[i].print, attack_param[i].size_print+1, "%04X", *(aux_short));
                       else
                          snprintf(attack_param[i].print, attack_param[i].size_print+1, "%04d", *(aux_short));
                    break;
                    default:
                       aux_long = (u_int32_t *)attack_param[i].value;
                       if (attack_param[i].type == FIELD_HEX)
                          snprintf(attack_param[i].print, attack_param[i].size_print+1, "%0*X", 
                                   attack_param[i].size_print, *(aux_long));
                       else
                          snprintf(attack_param[i].print, attack_param[i].size_print+1, "%0*d",
                                   attack_param[i].size_print, *(aux_long));
                    break;
                }
            break;

            case FIELD_IP:
                aux_long = (u_int32_t *)attack_param[i].value;
                parser_get_formated_inet_address(*aux_long, attack_param[i].print, 16);
            break;

            case FIELD_STR:
                memcpy(attack_param[i].print, attack_param[i].value, attack_param[i].size_print);
            break;
        }
    }
}


/*
 * Get parameters for attack
 */
int8_t
ncurses_gui_attack_get_params(struct attack_param *param, u_int8_t nparams)
{
    int32_t row, col;
    int32_t max_y, max_x, y, x;
    u_int8_t i, end_edit=0, offset_x=0, offset_y, origin_x, origin_y, max_print=0;
    int8_t ret=0;
    int32_t key_pressed;
    char *bottom = "ESC/Q to abort - ENTER to continue";
    WINDOW *my_window;
    PANEL *my_panel;

    for (i=0; i< nparams; i++)
    {
        if (strlen(param[i].desc) > offset_x)
           offset_x = strlen(param[i].desc);
        if (param[i].size_print > max_print)
           max_print = param[i].size_print;
    }
    
    offset_y = 2;
    
    getmaxyx(stdscr,row,col);

    if ((offset_x+max_print+3) > (strlen(bottom)+4))
       max_x = offset_x + max_print+3;
    else
       max_x = strlen(bottom)+4;
       
    max_y = nparams+4;

    my_window = newwin(max_y, max_x, (row-max_y)/2, (col-max_x)/2);
    my_panel = new_panel(my_window);

    wattron(my_window, COLOR_PAIR(3));
    box(my_window, 0, 0);

    mvwprintw(my_window, 0, 2, "Attack parameters");

    mvwprintw(my_window, max_y - 1, 2, bottom);
    wattroff(my_window, COLOR_PAIR(3));

    wmove(my_window, max_x - 3, 1);
       
    noecho();
    curs_set(0);
    
    for (i=0; i< nparams; i++)
    {
        wattron(my_window, COLOR_PAIR(2));
        mvwprintw(my_window, i+2, 1, "%*s", offset_x, param[i].desc); 
        wattroff(my_window, COLOR_PAIR(2));
        mvwprintw(my_window, i+2, 2+offset_x, "%s", param[i].print); 
    }
    
    origin_x = 2+offset_x;
    origin_y = 2;
    
    wmove(my_window, origin_y, origin_x);
    
    noecho();    
    curs_set(1);
    keypad(my_window, TRUE);
    
    wtimeout(my_window,NCURSES_KEY_TIMEOUT);
    
    end_edit = 0;
    
    while (!end_edit && !terms->gui_th.stop)
    {
	do
	{
	   key_pressed = wgetch(my_window);
	} while ( (key_pressed == ERR) && !terms->gui_th.stop);
	
	if (terms->gui_th.stop)
	   break;

	switch(key_pressed)
	{
            case 13: /* ENTER */
		end_edit = 1;
		ret = 0;
            break;

            case 27: /* ESC */
                 end_edit = 1;
                 ret = -1;
            break;
            
            case 9:  /* TAB */
            case KEY_DOWN:            
                getyx(my_window,y,x);
                if ( (y - offset_y) == (nparams-1))
                    wmove(my_window, origin_y, origin_x);
                else
                    wmove(my_window, y+1, origin_x);
            break;

            case KEY_UP:
                getyx(my_window,y,x);
                if ( (y - offset_y) == 0)
                    wmove(my_window, origin_y+(nparams-1), origin_x);
                else
                    wmove(my_window, y-1, origin_x);
            break;

            case KEY_RIGHT:
                getyx(my_window, y, x);
                   /* jump to the next valid character */
                if ( x < (origin_x + param[y-offset_y].size_print - 1))
                {
                   if ((param[y-offset_y].type == FIELD_MAC) && 
                       (((x - origin_x - 1) % 3) == 0)) 
                       wmove(my_window, y, x + 2); /* jump : */
                   else 
                      if ((param[y-offset_y].type == FIELD_BRIDGEID) && 
                          ((x - origin_x) == 3)) 
                          wmove(my_window, y, x + 2); /* jump . */
                      else 
                         if ((param[y-offset_y].type == FIELD_IP) && 
                             (((x - origin_x-1 ) % 4) == 1)) 
                             wmove(my_window, y, x + 2); /* jump . */
                         else
                             wmove(my_window, y, x + 1);
                }                
                else /* jump to the next field */
                {
                   if ((y-offset_y) == (nparams-1))
                      wmove(my_window, origin_y, origin_x);
                   else
                      wmove(my_window, y+1, origin_x);
                }
            break;

            case KEY_LEFT:
		getyx(my_window, y, x);

                if (x  > origin_x)
                {
                   if ((param[y-offset_y].type == FIELD_MAC) &&
                       (((x - origin_x) % 3) == 0) )
                       /* jump */
                       wmove(my_window, y, x - 2);
                   else 
                      if ((param[y-offset_y].type == FIELD_BRIDGEID) && 
                          ((x - origin_x) == 5) ) 
                          /* jump */
                          wmove(my_window, y, x - 2);
                      else 
                         if ((param[y-offset_y].type == FIELD_IP) && 
                             (((x - origin_x) % 4) == 0) ) 
                         /* jump */
                            wmove(my_window, y, x - 2);
                        else
                            wmove(my_window, y, x - 1);
                }
                else 
                {
                    if ( (y - offset_y) == 0)
                        wmove(my_window, origin_y+(nparams-1), 
                                origin_x+param[origin_y+(nparams-1)-offset_y].size_print-1);
                    else
                        wmove(my_window, y-1, 
                                origin_x+param[y-1-offset_y].size_print-1);                
                }
            break;
            
            default:
                getyx(my_window, y, x);

                if ((key_pressed == 'Q' || key_pressed == 'q') && 
                    (param[y-offset_y].type != FIELD_STR) &&
                    (param[y-offset_y].type != FIELD_IFACE) )
                {
                   end_edit = 1;
                   ret = -1;
                }            

                if ((param[y-offset_y].type == FIELD_HEX) || 
                    (param[y-offset_y].type == FIELD_MAC) || 
                    (param[y-offset_y].type == FIELD_BRIDGEID)) 
                {
                    if (!isxdigit(key_pressed))
                        /* only hexadecimal characters are allowed */
                        break;
                } 
                else 
                   if ((param[y-offset_y].type == FIELD_DEC) || 
                       (param[y-offset_y].type == FIELD_IP)) 
                   {
                      if (!isdigit(key_pressed))
                         break;
                   } 
                   else 
                      if ( (param[y-offset_y].type == FIELD_STR) ||
                           (param[y-offset_y].type == FIELD_IFACE) ) 
                      {
                         if (!isascii(key_pressed))
                            break;
                      } 
                      else /* FIELD_NONE */
                         break;

                waddch(my_window, toupper(key_pressed) | A_BOLD);

                getyx(my_window, y, x);
  
                param[y-offset_y].print[x-origin_x-1] = key_pressed;
    
                if ( x >= (origin_x + param[y-offset_y].size_print ))
                {
                   if ((y-offset_y) == (nparams-1))
                      wmove(my_window, origin_y, origin_x);
                   else
                      wmove(my_window, y+1, origin_x);                    
                }
                else 
                {
                    if ( (param[y-offset_y].type == FIELD_MAC) && 
                        (((x - origin_x + 1) % 3) == 0)) 
                         /* jump */
                       wmove(my_window, y, x + 1);
                    else 
                        if ( (param[y-offset_y].type == FIELD_BRIDGEID) && 
                            ((x - origin_x) == 4)) 
                           /* jump */
                           wmove(my_window, y, x + 1);
                       else 
                           if ( (param[y-offset_y].type == FIELD_IP) && 
                               (((x - origin_x + 1) % 4) == 0)) 
                               /* jump */
                              wmove(my_window, y, x + 1);
                }
            break;

	}
    }

    keypad(my_window, FALSE);
    noecho();
    curs_set(0);    
    del_panel(my_panel);
    delwin(my_window);

    if (terms->gui_th.stop)
        return -1;

    return ret;
}


/* 
 * Display available modes and let the user choose one!
 */
int8_t
ncurses_gui_get_mode(u_int8_t mode, WINDOW *main_window)
{
    WINDOW *win;
    int32_t i, j, k, pointer, end, result, key_pressed, row, col;
            
    pointer = mode;
    end = 0;
    result = ERR;

    ncurses_gui_set_status_line(" Choose your life (mode) ");

    getmaxyx(main_window, row, col);

    if ((win = derwin(main_window, 10, 45, (row-10)/2, (col-45)/2)) == NULL)
        return ERR;

    keypad(win, TRUE);

//    scrollok(win, TRUE);
//    idlok(win, TRUE);

    while (!terms->gui_th.stop && !end) {
        j = 0;
        i = pointer;
        /* Show max 10 options */
	/* -1 for ARP, we'll fix that! */
	for (k = i; (k < MAX_PROTOCOLS-1) && (k - i) < 9; k++) {
            if (i == pointer)
                wattron(win, COLOR_PAIR(5) | A_BOLD);
	    else
		wattroff(win, COLOR_PAIR(5) | A_BOLD);

            wmove(win, j+1, 1);
            whline(win, ' ', 20);

            mvwprintw(win, j+1, 2, "%-6s %s", protocols[k].name, protocols[k].description);
            if (i == pointer)
                wattroff(win, A_BOLD);

            i++; j++;
        }

        wattron(win, COLOR_PAIR(5));
        box(win, 0, 0);
        mvwprintw(win, 0, 2, "Choose mode");
        mvwprintw(win, 11, 2, "Up/Down   q: quit");
        wtimeout(win,NCURSES_KEY_TIMEOUT); /* Block for 100 millisecs...*/

        key_pressed = wgetch(win);
        switch(key_pressed) {
            case KEY_DOWN:
	/* -1 for ARP, we'll fix that! */
                pointer = (pointer+1 < MAX_PROTOCOLS-1) ? (pointer+1) : pointer;
                break;
            case KEY_UP:
                pointer = (pointer > 0) ? (pointer-1) : 0;
                break;
            case 13: /* ENTER */
                result = pointer;
                end = 1;
                break;
            case 27: /* ESC */
            case 'q': /* ESC */
                end = 1;
                break;
        }
        werase(win);
    }

    wrefresh(main_window);

    if (delwin(win) == ERR)
        return ERR;

    ncurses_gui_set_status_line("");

    return result;
} 
