#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <limits.h>

void write_datfile(FILE *dat, char *datname, char *resolution);
unsigned int md(unsigned char x, unsigned char y, unsigned char z);
unsigned char plot_point(unsigned char *matrix, int density,
		unsigned char x, unsigned char y, unsigned char z,
		char shift);
char *pk_fgets(char *buf, int length, FILE *stream);
void phentropy_usage();

int verbose=0;

int main(int argc, char **argv)
{

	int opt;
	extern char *optarg;
	extern int   opterr;	
   
	int i,j,k,l;
	FILE *input = NULL, *dat = NULL, *raw = NULL;
	int max=0;
	int density=127;
	int bytes=4;
	int ascii=0;
	char shift=0;
	char datname[1024], rawname[1024];
	unsigned long divisor = 1;
	unsigned char buf3[1024],buf2[1024],buf1[1024],buf[1024];
	unsigned long  *l3,*l2,*l1,*l0;
	unsigned short *s3,*s2,*s1,*s0;
	unsigned char  *c3,*c2,*c1,*c0;
	unsigned char   x, y, z;

	unsigned char *pretty = malloc(256*256*256);
	memset(pretty, 0, 256*256*256);
	memset(&datname, 0, sizeof(datname));
	memset(&rawname, 0, sizeof(datname));
	
	while ((opt = getopt(argc, argv, "hm:d:b:avo:s:t:")) != EOF) {
	   switch (opt) {
	   	case 'h':
	   	   phentropy_usage();
	   	   break;
	   	case 'm':
	   	   max=atoi(optarg);
	   	   break;
	   	case 'd':
	   	   density=atoi(optarg);
	   	   break;
	   	case 'b':
	   	   bytes=atoi(optarg);
	   	   if(bytes!=1 && bytes!=2 && bytes!=4)phentropy_usage();
	   	   break;
	   	case 'a':
	   	   ascii++;
	   	   break;
	   	case 'v':
	   	   verbose++;
	   	   break;
	   	case 'o':
	   	   snprintf(datname, sizeof(datname), "%s.dat", optarg);
	   	   snprintf(rawname, sizeof(rawname), "%s.raw", optarg);
	   	   break;
	   	case 's':
	   	   shift=(char)atoi(optarg);
	   	   break;
	   	case 't':
	   	   divisor=atol(optarg);
	   	   break;
	   	}
	}

	if(argv[optind] != NULL)
	{
	   snprintf(buf, sizeof(buf), "%s", argv[optind]);
	}  else {
	   fprintf(stderr, "Source of entropy required.\n");
	   phentropy_usage();
	}

	if(buf[0]=='-' && buf[1]==0)input=stdin;
	else          input=fopen(buf, "r");

	if(!input){
		fprintf(stderr, "Input file required.\n");
		phentropy_usage();
	}
	if(!datname[0])snprintf(datname, sizeof(datname), "%s.dat", buf);
	if(!rawname[0])snprintf(rawname, sizeof(rawname), "%s.raw", buf);

	dat = fopen(datname, "w");
	raw = fopen(rawname, "w");
	if(!dat || !raw)
	{
		fprintf(stderr, "Couldn't write dat or raw file\n");
		phentropy_usage();
	}

	write_datfile(dat, rawname, "256 256 256");
	fclose(dat);

	if(!divisor){
		if(bytes==2) divisor=256; /* 1 extra byte */
		/* can't support 3 bytes yet -- need rollover on 24 bits */
		if(bytes==4) divisor=256*256*256; /* 3 extra bytes */
	}
	i=1;
	if(ascii){
	   l3=malloc(sizeof(unsigned long));
	   l2=malloc(sizeof(unsigned long));
	   l1=malloc(sizeof(unsigned long));
	   l0=malloc(sizeof(unsigned long));
	   s3=malloc(sizeof(unsigned short));
	   s2=malloc(sizeof(unsigned short));
	   s1=malloc(sizeof(unsigned short));
	   s0=malloc(sizeof(unsigned short));
	   c3=malloc(sizeof(unsigned char));
	   c2=malloc(sizeof(unsigned char));
	   c1=malloc(sizeof(unsigned char));
	   c0=malloc(sizeof(unsigned char));

	   while(
		pk_fgets(buf3, sizeof(buf3), input) &&
		pk_fgets(buf2, sizeof(buf2), input) &&
		pk_fgets(buf1, sizeof(buf1), input) &&
		pk_fgets(buf, sizeof(buf), input)   &&
		i!=max
	   ){
		switch(bytes){
		   case 4:
			*l3=strtoul(buf3,NULL,0);
			*l2=strtoul(buf2,NULL,0);
			*l1=strtoul(buf1,NULL,0);
			*l0=strtoul(buf,NULL,0);
			x=(*l3-*l2)/divisor;
			y=(*l2-*l1)/divisor;
			z=(*l0-*l1)/divisor;
			if(verbose)fprintf(stderr, "%u %u %u %u: ", *l3,*l2,*l1,*l0);
			plot_point(pretty, density, x, y, z, shift);
			break;
		   case 2:
			*s3=strtoul(buf3,NULL,0);
			*s2=strtoul(buf2,NULL,0);
			*s1=strtoul(buf1,NULL,0);
			*s0=strtoul(buf,NULL,0);
			x=(*s3-*s2)/divisor;
			y=(*s2-*s1)/divisor;
			z=(*s0-*s1)/divisor;
			if(verbose)fprintf(stderr, "%u %u %u %u: ", *s3,*s2,*s1,*s0);
			plot_point(pretty, density, x, y, z, shift);
			break;
		   case 1:
   			*c3=strtoul(buf3,NULL,0);
			*c2=strtoul(buf2,NULL,0);
			*c1=strtoul(buf1,NULL,0);
			*c0=strtoul(buf,NULL,0);
			x=*c3-*c2;
			y=*c2-*c1;
			z=*c1-*c0;
			if(verbose)fprintf(stderr, "%u %u %u %u: ", *c3,*c2,*c1,*c0);
			plot_point(pretty, density, x, y, z, shift);
			break;
		}
		i++;
	   }
	} else {
	if(!ascii){
	   while(
	   	/* NOT endian aware :-) */
		fread(&buf3, bytes, 1, input) &&
		fread(&buf2, bytes, 1, input) &&
		fread(&buf1, bytes, 1, input) &&
		fread(&buf, bytes, 1, input) &&
		i!=max
	   ){
		switch(bytes){
		   case 4:
			l3=(unsigned long *)buf3;
			l2=(unsigned long *)buf2;
			l1=(unsigned long *)buf1;
			l0=(unsigned long *)buf;
			x=(*l3-*l2)/divisor;
			y=(*l2-*l1)/divisor;
			z=(*l0-*l1)/divisor;
			if(verbose)fprintf(stderr, "%u %u %u %u: ", *l3,*l2,*l1,*l0);
			plot_point(pretty, density, x, y, z, shift);
			break;
		   case 2:
			s3=(unsigned short *)buf3;
			s2=(unsigned short *)buf2;
			s1=(unsigned short *)buf1;
			s0=(unsigned short *)buf;
			x=(*s3-*s2)/divisor;
			y=(*s2-*s1)/divisor;
			z=(*s0-*s1)/divisor;
			if(verbose)fprintf(stderr, "%u %u %u %u: ", *s3,*s2,*s1,*s0);
			plot_point(pretty, density, x, y, z, shift);
			break;
		   case 1:
			c3=buf3;
			c2=buf2;
			c1=buf1;
			c0=buf;
			x=(*c3-*c2)/divisor;
			y=(*c2-*c1)/divisor;
			z=(*c0-*c1)/divisor;
			if(verbose)fprintf(stderr, "%u %u %u %u: ", *c3,*c2,*c1,*c0);
			plot_point(pretty, density, x, y, z, shift);
			break;
		}
		i++;
	   }
	}}


	fwrite(pretty, sizeof(unsigned char), 256*256*256, raw);

	fclose(raw);
	exit(0);
}

unsigned char plot_point(unsigned char *matrix, int density,
		unsigned char x, unsigned char y, unsigned char z,
		char shift)
{
	x+=shift;
	y+=shift;
	z+=shift;
	if(  matrix[md(x,y,z)]+ density > matrix[md(x,y,z)] )
	     matrix[md(x,y,z)]+=density;
	if(verbose)fprintf(stdout, "%3.3u x %3.3u x %3.3u = %3.3u\n", x, y, z, matrix[md(x,y,z)]);
	return(matrix[md(x,y,z)]);
}

unsigned int md(unsigned char x, unsigned char y, unsigned char z)
{
	return((x * 65536) + (y * 256) + z);
}

void write_datfile(FILE *dat, char *datname, char *resolution)
{
	fprintf(dat, "ObjectFileName: %s\n", datname);
	fprintf(dat, "TaggedFileName: ---\n");
	fprintf(dat, "Resolution: %s\n", resolution);
	fprintf(dat, "SliceThickness: 1 1 1\n");
	fprintf(dat, "Format:         UCHAR\n");
	fprintf(dat, "NbrTags:        0\n");
	fprintf(dat, "ObjectType:     TEXTURE_VOLUME_OBJECT\n");
	fprintf(dat, "ObjectModel:    RGBA\n");
	fprintf(dat, "GridType:       EQUIDISTANT\n");
}

char *pk_fgets(char *buf, int length, FILE *stream)
{
	int i, j;
	bzero(buf, length);
	if(feof(stream)) {
		return(NULL);
		}

	for(i=0; i<length; i++)
	{
		buf[i]=getc(stream);
		if(buf[i]=='\\'){
			while(getc(stream)!='\n') if(feof(stream)){
			 return(NULL);
			 }
			buf[i]=getc(stream);
		}
		if(buf[i]=='\r'){
			getc(stream);
			buf[i]=getc(stream);
		}
		if(buf[i]=='\n') return(buf);
	}
	return(buf);
}


void phentropy_usage()
{
   fprintf(stderr, "phentropy 1.0:  Zalewskian Phase Space Entropy Viz for OpenQVIS\n");
   fprintf(stderr, "Component of:  Paketto Keiretsu 1.0;    Dan Kaminsky  (dan@doxpara.com)\n\n");
   fprintf(stderr, "        Usage:  phentropy [filename|-]\n\n");
   fprintf(stderr, "      Options:  -a       : Samples are in ASCII format (instead of raw)\n");
   fprintf(stderr, "                -b[1/2/4]: Samples range across 1/2/4 bytes (4)\n");
   fprintf(stderr, "                -d[1-255]: Plot points with this density  (127)\n");
   fprintf(stderr, "                -m[n]    : Read at most this many values  (inf)\n");
   fprintf(stderr, "                -o[fname]: Write metadata and raw matrix here  \n");
   fprintf(stderr, "                -s[1-255]: Shift plot by this amount (0, but 127 is useful)\n");
   fprintf(stderr, "                -t[long] : Specify divisor to make deltas range from 0-255\n");   
   fprintf(stderr, "                -v       : Engage Verbosity\n");
   exit(1);
}
