/*
 * (c) Copyright 1992 by Panagiotis Tsirigotis
 * (c) Sections Copyright 1998-2001 by Rob Braun
 * All rights reserved.  The file named COPYRIGHT specifies the terms 
 * and conditions for redistribution.
 */

static char RCSid[] = "$Id: log.c,v 1.1.1.1 1999/10/12 17:28:59 bbraun Exp $" ;

#include "config.h"
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <syslog.h>
#include <time.h>
#include <stdio.h>

#include "sio.h"
#include "str.h"

#include "connection.h"
#include "defs.h"
#include "access.h"
#include "sconst.h"
#include "service.h"
#include "server.h"
#include "xconfig.h"

#define LOGBUF_SIZE                  1024

struct xaddr
{
   unsigned char a;
   unsigned char b;
   unsigned char c;
   unsigned char d;
};


#ifdef INET6
static char ipv6_ret[48];
#endif
char *xntoa(struct sockaddr *inaddr)
{
#ifdef INET6
   bzero(ipv6_ret, sizeof(ipv6_ret));

   inet_ntop(AF_INET6, &( ((struct sockaddr_in6 *)inaddr)->sin6_addr), ipv6_ret, sizeof(ipv6_ret)-1);

   return ipv6_ret;
#else
   return inet_ntoa( ((struct sockaddr_in *)inaddr)->sin_addr );
#endif

}


PRIVATE int log_common(mask_t *, char *, int, connection_s *) ;

/*
 * This function writes log records of the form:
 *
 *      START: service [pid] [from_address]
 */
void svc_log_success( struct service *sp, connection_s *cp, pid_t pid )
{
   char                    buf[ LOGBUF_SIZE ] ;
   int                     bufsize ;
   struct service_config   *scp = SVC_CONF( sp ) ;
   int                     len ;
   int                     cc ;

   if ( ! SVC_LOGS_ON_SUCCESS( sp ) )
      return ;
   
   bufsize = sizeof( buf ) ;
   len = 0 ;
   
   cc = strx_nprint( buf, bufsize, "%s: %s", START_ENTRY, SC_ID( scp ) ) ;
   len += cc ;
   bufsize -= cc ;

   if ( SC_LOGS_PID( scp ) )
   {
      cc = strx_nprint( &buf[ len ], bufsize, " pid=%d", pid ) ;
      len += cc ;
      bufsize -= cc ;
   }

   cc = log_common( &SC_LOG_ON_SUCCESS( scp ), &buf[len], bufsize, cp ) ;
   len += cc ;
   bufsize -= cc ;

   xlog_write( sp->svc_log, buf, len, XLOG_NOFLAGS ) ;
}


/*
 * This function writes log records of the form:
 *
 *      FAIL: service failure-type [from_address]
 *
 */
void svc_log_failure( struct service *sp, 
                      connection_s *cp, 
                      access_e access_failure )
{
   char                    buf[ LOGBUF_SIZE ] ;
   int                     bufsize ;
   struct service_config   *scp = SVC_CONF( sp ) ;
   int                     len = 0 ;
   int                     cc ;
   
   if ( ! SVC_LOGS_ON_FAILURE( sp ) )
      return ;
   
   bufsize = sizeof( buf ) ;
   cc = strx_nprint( buf, bufsize, "%s: %s", FAIL_ENTRY, SC_ID( scp ) ) ;
   len += cc ;
   bufsize -= cc ;

   cc = strx_nprint( &buf[ len ], bufsize,
                        " %s", access_explain( access_failure ) ) ;
   len += cc ;
   bufsize -= cc ;

   cc = log_common( &SC_LOG_ON_FAILURE( scp ), &buf[ len ], bufsize, cp ) ;
   len += cc ;
   bufsize -= cc ;

   xlog_write( sp->svc_log, buf, len, XLOG_NOFLAGS ) ;
}



PRIVATE int log_common( mask_t *logmask, 
                        char *buf, 
                        int bufsize, 
                        connection_s *cp )
{
   int len = 0 ;
   int cc ;

   if ( M_IS_SET( *logmask, LO_HOST ) )
   {
      cc = strx_nprint( &buf[ len ], bufsize, " from=%s", conn_addrstr( cp ) ) ;
      len += cc ;
      bufsize -= cc ;
   }
   return( len ) ;
}



void svc_log_exit( struct service *sp, struct server *serp )
{
   char                    buf[ LOGBUF_SIZE ] ;
   int                     bufsize ;
   int                     cc ;
   int                     len ;
   int                     exit_status = SERVER_EXITSTATUS( serp ) ;
   struct service_config   *scp = SVC_CONF( sp ) ;
   char                    *func = "log_exit" ;

   if ( ! SVC_LOGS_ON_EXIT( sp ) )
      return ;

   bufsize = sizeof( buf ) ;
   len = 0 ;

   cc = strx_nprint( buf, bufsize, "%s: %s", EXIT_ENTRY, SC_ID( scp ) ) ;
   bufsize -= cc ;
   len += cc ;

   /*
    * If the EXIT flag was used, log the exit status or the signal that
    * killed the process. We assume that these are the only reasons
    * for process termination.
    */
   if ( SC_LOGS_EXITS( scp ) )
   {
      int num  = 0;
      char *s ;

      if ( PROC_EXITED( exit_status ) )
      {
         s = "status" ;
         num = PROC_EXITSTATUS( exit_status ) ;
      }
      else if ( PROC_SIGNALED( exit_status ) )
      {
         s = "signal" ;
         num = PROC_TERMSIG( exit_status ) ;
      }
      else
      {
         msg( LOG_ERR, func, "Bad exit status" ) ;
         s = NULL ;
      }

      if ( s )
      {
         cc = strx_nprint( &buf[ len ], bufsize, " %s=%d", s, num ) ;
         len += cc ;
         bufsize -= cc ;
      }
   }

   if ( SC_LOGS_PID( scp ) )
   {
      cc = strx_nprint( &buf[ len ], bufsize, " pid=%d", SERVER_PID( serp ) ) ;
      len += cc ;
      bufsize -= cc ;
   }

   if ( SC_LOGS_DURATION( scp ) )
   {
      time_t current_time ;

      (void) time( &current_time ) ;
      cc = strx_nprint( &buf[ len ], bufsize, " duration=%d(sec)", 
                           current_time - SERVER_STARTTIME( serp ) ) ;
      len += cc ;
      bufsize -= cc ;
   }
   xlog_write( sp->svc_log, buf, len, XLOG_NOFLAGS ) ;
}



/*
 * Used by other parts of xinetd that want to log something without
 * going through the proper channels (i.e. log_{success,failure} and log_exit)
 */
/* VARARGS3 */
void svc_logprint( struct service *sp, char *line_id, char *fmt, ...)
{
   char     buf[ LOGBUF_SIZE ] ;
   int      bufsize = sizeof( buf ) ;
   int      len ;
   int      cc ;
   va_list  ap ;

   if ( ! SVC_IS_LOGGING( sp ) )
      return ;

   len = strx_nprint( buf, bufsize, "%s: %s ", line_id, SVC_ID( sp ) ) ;
   va_start( ap, fmt ) ;
   cc = strx_nprintv( &buf[ len ], bufsize-len, fmt, ap ) ;
   va_end( ap ) ;
   xlog_write( sp->svc_log, buf, len+cc, XLOG_NO_SIZECHECK ) ;
}

