/*
 * (c) Copyright 1992 by Panagiotis Tsirigotis
 * (c) Sections Copyright 1998-2001 by Rob Braun
 * All rights reserved.  The file named COPYRIGHT specifies the terms 
 * and conditions for redistribution.
 */

static char RCSid[] = "$Id: addr.c,v 1.2 1999/10/15 05:18:39 bbraun Exp $" ;

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <syslog.h>
#include <netdb.h>
#include <memory.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/param.h>
#include "config.h"
#if defined(HAVE_ARPA_INET_H)
#include <arpa/inet.h>
#endif
#include <netinet/in.h>

#include "pset.h"

#include "defs.h"
#include "addr.h"
#include "str.h"
#include "sio.h"
#include "log.h"
#include "parse.h"

#define OPEN_CURLY_BRACKET              '{'
#define CLOSED_CURLY_BRACKET            '}'
#define COMMA                           ','
#define DOT                             '.'

/*
 * address types denote how the actual numeric address was obtained.
 * Currently they are only useful for debugging.
 * Note that NUMERIC_ADDR includes both simple (e.g. 128.138.91.1) and
 * factorized symbolic addresses (e.g. 128.138.91.{1,2,3}).
 */
typedef enum {   NUMERIC_ADDR, NET_ADDR, HOST_ADDR } address_e ;

typedef enum { CANT_PARSE, PARSED, ERROR } result_e ;

struct comp_addr
{
   address_e       addr_type ;
   char            name[MAXHOSTNAMELEN+1] ;
#ifdef INET6
   struct in6_addr addr ;
   struct in6_addr mask ;
#else
   uint32_t        addr ;  /* host byte order */
   uint32_t        mask ;
#endif
} ;

#define CAP( p )              ( (struct comp_addr *) (p) )


#define NEW_CAP()               NEW( struct comp_addr )
#define FREE_CAP( cap )         FREE( cap )


/*
 * Try to match the given address 'addr' with the specified address list.
 * Returns TRUE if the address matched, FALSE otherwise.
 * If the address matched, '*matchp' will hold the mask of the matching
 * address (a greater numerical value for the mask implies a more
 * precise match).
 */

#ifdef INET6

/* The addrlist_match function sets the mask for IPv6 addresses.
 * mask is a pointer to the in6_addr structure, bits is the
 * number of bits to set in the mask, and len is the length of mask.
 */
void xsetmask(char *mask, unsigned int bits, int len)
{
   int i;
   int bytes = bits/8;
   int remain = bits - (bytes * 8);
   char *func = "xsetmask";
   
   bzero(mask, len);

   /* This may be wrong for bigendian... */
   for(i=0; i < bytes; i++ ) {
      mask[i] = 0xFF;
   }

   if( remain > 0 )
      mask[i] = ( 0xFF << (8-remain) );

   return;
}

/* This is a helper function to make address matching with mask
 * work ok w/ipv6 
 * Returns TRUE if addr1&mask1 == addr2
 */
PRIVATE bool_int xmatch(char *addr1, char *mask1, char *addr2, int len)
{
   int i;

   for(i=0; i < len; i++ ) {
      if( (addr1[i] & mask1[i]) != addr2[i] ) {
         return( FALSE );
      }
   }
   return TRUE;
} 

/* maps a v4 address into a v6 address */
void xaddrmap(char *v4, char *v6)
{
   if( (v4 == NULL) || (v6 == NULL) )
      return;

   strcpy(v6, "::ffff:");
   strncat(v6, v4, 38);

   return;
}

/*
*   This function returns 0 if no match and the offset+1
*   to list which element in the list matched
*/
int addrlist_match( pset_h addr_list, 
                    struct in6_addr *addr, 
                    struct in6_addr *matchp )
{
   struct in6_addr remote_addr;
   unsigned u, addr_count;

   addr_count = pset_count( addr_list );
   if (addr_count == 0)
      return 0;
      
   memcpy(&remote_addr, addr, sizeof(remote_addr));

   for ( u = 0 ; u < addr_count ; u++ ) {
      struct comp_addr *cap = CAP( pset_pointer( addr_list, u ) ) ;

      if( (cap->addr_type == HOST_ADDR) && (cap->name != NULL) ) {
         struct hostent *hep;
         char *hname;

         hep = gethostbyaddr((char *)addr, 16, AF_INET6);
         if( hep == NULL ) {
            if( memcmp(&remote_addr, &(cap->addr), sizeof(remote_addr)) == 0 )
               return ( u+1 ) ;
            continue;
         }

         /* Parse the address as a domain portion */
         if( cap->name[0] == '.' )
         {
            hname = str_casefind( hep->h_name, cap->name );
            if( hname != NULL ) {
               if( strlen(cap->name) == strlen(hname) )
                  return( u+1 );
            }
         } else {
            if( str_casefind( hep->h_name, cap->name ) == hep->h_name )
               return( u+1 );
         }
      } else {
         if( xmatch( (char *)&remote_addr, (char *)&(cap->mask), (char *)&(cap->addr), sizeof(remote_addr)) == TRUE ) {
            memcpy(matchp, &(cap->mask), sizeof(*matchp));
            *matchp = cap->mask ;
            return( u+1 );
         }
      }
   }
   return ( 0 );
}

#else /* INET6 */

bool_int addrlist_match( pset_h addr_list, 
                         struct in_addr *addr, 
                         uint32_t *matchp )
{
   uint32_t remote_addr = ntohl( addr->s_addr ) ;
   unsigned u ;

   for ( u = 0 ; u < pset_count( addr_list ) ; u++ )
   {
      register struct comp_addr *cap = CAP( pset_pointer( addr_list, u ) ) ;

      /* If it's a HOST_ADDR type, do a name lookup and see if they 
       * match.
       */
      if( (cap->addr_type == HOST_ADDR) && (cap->name != NULL) )
      {
         struct hostent *hep = NULL;
         char *hname = NULL;
      
         hep = gethostbyaddr((char *)addr, 4, AF_INET);
         if( hep == NULL ) {
            if( memcmp(&remote_addr, &(cap->addr), sizeof(cap->addr)) == 0 )
               return TRUE;
            continue;
         }

         /* Parse the address as a domain portion */
         if( cap->name[0] == '.' )
         {
            hname = str_casefind( hep->h_name, cap->name );
            if( hname != NULL ) {
               if( strlen(cap->name) == strlen(hname) ) 
                  return TRUE ;
            }
         } else {
            struct hostent *hcp = NULL;
            /* try to match addresses associated with the allowed name */
            hcp = gethostbyname(cap->name);
            if( hcp != NULL ) {
               int j;
               for( j=0; hcp->h_addr_list[j] != NULL; j++){
                  if( memcmp( (struct in_addr *)hcp->h_addr_list[j], hep->h_addr, 4) == 0 )
                     return TRUE ;
               }
            }
            if( str_casefind( hep->h_name, cap->name ) == hep->h_name ) 
               return TRUE ;
         }
      } else {
         if ( ( remote_addr & cap->mask ) == cap->addr )
         {
            *matchp = cap->mask ;
            return TRUE ;
         }
      }
   }
   return FALSE ;
}
#endif /* INET6 */

void addrlist_dump( pset_h addr_list, int fd )
{
   unsigned u, num ;
   struct in_addr inaddr ;

   num = pset_count( addr_list );
   for ( u = 0 ; u < num ; u++ )
   {
      struct comp_addr *cap = CAP( pset_pointer( addr_list, u ) ) ;
      char *type ;

#ifndef INET6
      inaddr.s_addr = htonl(cap->addr) ;
#else
      memcpy(&inaddr, &(cap->addr), sizeof(cap->addr));
#endif

      if ( cap->addr_type == NUMERIC_ADDR )
         type = "NUMERIC" ;
      else if ( cap->addr_type == NET_ADDR )
         type = "NET" ;
      else if ( cap->addr_type == HOST_ADDR )
         type = "HOST" ;
      else
         type = "BAD" ;
      
/*
#ifndef INET6
      Sprint( fd, "foo");
      Sprint( fd, " %s(%s)", xntoa( inaddr.s_addr ), type ) ;
#else
      Sprint( fd, " %s(%s)", xntoa( cap->addr ), type ) ;
#endif
*/
#ifdef INET6
      Sprint( fd, " %s(%s)",  cap->name, type ) ;
#else
      if ( cap->addr_type == NET_ADDR )
         Sprint(fd, " %s/%X(%s)", inet_ntoa( inaddr ), cap->mask, type);
      else if ( cap->addr_type == HOST_ADDR )
         Sprint( fd, " %s(%s)",  cap->name, type ) ;
      else
         Sprint( fd, " %s(%s)", inet_ntoa( inaddr), type ) ;
#endif
   }
}


void addrlist_free( pset_h addr_list )
{
   pset_apply( addr_list, free, NULL ) ;
}


/*
 * Add an address to the address list
 */
PRIVATE status_e add( pset_h addr_list, struct comp_addr *cap )
{
   struct comp_addr *new_cap = NULL;
   char *func = "add" ;

   new_cap = NEW_CAP() ;
   if ( new_cap == NULL )
   {
      out_of_memory( func ) ;
      return( FAILED ) ;
   }

   *new_cap = *cap ;
   if ( pset_add( addr_list, new_cap ) == NULL )
   {
      out_of_memory( func ) ;
      FREE_CAP( new_cap ) ;
      return( FAILED ) ;
   }
   return( OK ) ;
}


/*
 * Find the address and remove it from the list
 * Since there is no check when we add entries that an
 * address is not entered twice, in this function we remove all
 * addresses that match.
 *
 * XXX: we need to work on the way two cap's are considered equal
 */

PRIVATE status_e xremove( pset_h addr_list, struct comp_addr *cap )
{
   unsigned u = 0 ;
   struct comp_addr *old_cap ;

   for ( u = 0 ; u < pset_count( addr_list ) ; u++ )
   {
      old_cap = CAP( pset_pointer( addr_list, u ) ) ;

#ifdef INET6
      if ( IN6_ARE_ADDR_EQUAL( &(old_cap->addr), &(cap->addr)) && 
         IN6_ARE_ADDR_EQUAL( &(old_cap->mask), &(cap->mask)) )
#else
      if ( old_cap->addr == cap->addr && old_cap->mask == cap->mask )
#endif
      {
         pset_pointer( addr_list, u ) = NULL ;
         FREE_CAP( old_cap ) ;
      }
   }
   pset_compact( addr_list ) ;
   return( OK ) ;
}

/*
 * Function allows the use of 0.0.0.0/24 style address ranges for access cntl.
 *   --bbraun 10/26/98
 *
 * Updated to handle ::/46 style address ranges for access cntl.
 */
PRIVATE result_e explicit_mask( char *str_addr, statfunc op, pset_h addr_list)
{
   struct comp_addr         ca ;
   char newaddr[16];
   int val;
   unsigned a, b, c, d; 
   int mask;
   char tmpaddr[MAXHOSTNAMELEN+1];
   struct in_addr t_addr;

   if (strchr(str_addr, '{'))  /* Don't even try factorized */
      return CANT_PARSE ;
   strncpy(tmpaddr, str_addr, sizeof(tmpaddr)-1);
   tmpaddr[sizeof(tmpaddr)-1] = '\0';
#ifdef INET6
{
   char saddr[INET6_ADDRSTRLEN];
   char v6addr[46];
   bzero(saddr, INET6_ADDRSTRLEN);

   /* Ugly hack, but sscanf doesn't like the '/' */
   for( a=0; a < strlen(tmpaddr); a++ ) {
      if( tmpaddr[a] == '/' )
         tmpaddr[a] = ' ';
   }
   val = sscanf(tmpaddr, "%46s %d", saddr, &mask);

   /* If v6 address fails, assume it is a v4 address, mapp it and try again */
   if( inet_pton(AF_INET6, saddr, &ca.addr) != 1 ) {
      bzero(v6addr, 46);
      xaddrmap(saddr, v6addr);
      if( inet_pton(AF_INET6, v6addr, &ca.addr) != 1 )
         return(CANT_PARSE);

      /* convert the v4 mask into a v6 mapped mask */
      mask += 96;
   }

   if( val < 2 ) 
      return CANT_PARSE ;

   if( mask >= 128 ) {
      ca.addr_type = HOST_ADDR;
   } else {
      ca.addr_type = NET_ADDR;
   }

   xsetmask((char *)&ca.mask, mask, sizeof(ca.mask));
   
   return( ( (*op)( addr_list, &ca ) == OK ) ? PARSED : ERROR ) ;
}
#else
   val = sscanf(tmpaddr, "%d.%d.%d.%d/%d", &a, &b, &c, &d, &mask);
   if( val < 5 )
      return CANT_PARSE ;

#ifdef DEBUG
   syslog(LOG_ERR, "explicit_mask: %d.%d.%d.%d/%d", a, b, c, d, mask);
#endif
   if ((a > 255)||(b > 255)||(c > 255)||(d > 255))
      return CANT_PARSE ;

   strx_sprint(newaddr, sizeof(newaddr)-1, "%d.%d.%d.%d", a, b, c, d);
   if (inet_aton(newaddr, &t_addr) == 0)
	return CANT_PARSE;

   ca.addr = ntohl(t_addr.s_addr);
   if(mask == 32)
   {
      ca.addr_type = HOST_ADDR;
      ca.mask      = 0xFFFFFFFF;
   }
   else
   {
      uint32_t i, n;

      ca.addr_type = NET_ADDR ;
   
      /* this is binary 100000000000000000000000000000000 */
      /* i = 2147483648; */
      /* A much more reliable method */
      i = ( 1 << ((sizeof(i)*8)-1));
      ca.mask = 0;

      /* Go through and set each bit to 1 */
      for( n=mask; n > 0 ; n--)
      {
         ca.mask |= i;
         i /= 2;
      }
   }

   ca.addr &= ca.mask;

#ifdef DEBUG
   syslog(LOG_ERR, "explicit_mask: %x %x", ca.addr, ca.mask);
#endif
   
   return( ( (*op)( addr_list, &ca ) == OK ) ? PARSED : ERROR ) ;
#endif /* INET6 */
}

/*
 * Try to parse 'str_addr' as a symbolic net name
 */
PRIVATE result_e net_addr( char *str_addr, statfunc op, pset_h addr_list )
{

   /*
    *
    *  The following table demonstrates how the mask is determined
    *  given a net number N and following the relation:
    *     net #1 <= N <= #2
    *
    *     net #1      net #2      mask
    *        0           0        FFFFFFFF    (this should be rejected)
    *        1           FF       FF000000
    *        100         FFFF     FFFF0000
    *        10000       FFFFFF   FFFFFF00
    *        1000000     FFFFFFFF FFFFFFFF
    */
   static struct { int lim, mask, shift ; } net_to_mask[] =
   {
      { 0,            0xFF000000,   24  },
      { 0xFF,         0xFFFF0000,   16  },
      { 0xFFFF,       0xFFFFFF00,   8   },
      { 0xFFFFFF,     0xFFFFFFFF,   0   },
      { 0xFFFFFFFF,   0,            0   }
   } ;
   struct comp_addr         ca ;
   struct netent            *nep ;
   uint32_t                 net_num ;
   int                      i ;
   char                     *func = "net_addr" ;
   char                     tmpaddr[MAXHOSTNAMELEN+1];

#ifdef INET6
   /* Not yet implemented */
   return(CANT_PARSE);
#else

   strncpy(tmpaddr, str_addr, sizeof(tmpaddr)-1);
   tmpaddr[sizeof(tmpaddr)-1] = '\0';
   nep = getnetbyname( tmpaddr ) ;
   if ( nep == NULL || nep->n_addrtype != AF_INET || nep->n_net == 0 )
      return( CANT_PARSE ) ;

   for ( i = 0, net_num = (uint32_t) nep->n_net ;; i++ )
   {
      if ( net_to_mask[ i ].mask == 0 )
      {
         msg( LOG_CRIT, func,
            "INTERNAL ERROR: Cannot process net number %ld", net_num ) ;
         return( ERROR ) ;
      }
      if ( net_to_mask[i].lim < net_num && net_num <= net_to_mask[i+1].lim )
      {
         ca.addr_type = NET_ADDR ;
         ca.addr = net_num << net_to_mask[ i ].shift ;
         ca.mask = net_to_mask[ i ].mask ;
         return( ( (*op)( addr_list, &ca ) == OK ) ? PARSED : ERROR ) ;
      }
   }
#endif /* INET6 */
}


/*
 * Try to parse 'str_addr' as a numeric address
 */
PRIVATE result_e numeric_addr( char *str_addr, 
                               status_e (*op)(), 
                               pset_h addr_list )
{
   uint32_t addr;
   uint32_t mask ;
   struct in_addr t_addr;
   struct comp_addr ca ;
   char tmpaddr[MAXHOSTNAMELEN+1];

   if (strchr(str_addr, '/')) /* Don't even try explicit masks */
      return CANT_PARSE;
   if(strchr(str_addr, ':'))  /* Don't even try IPv6 if machine is dual boot */
      return CANT_PARSE;
   if(strchr(str_addr, '{'))  /* Don't even try factorized */
      return CANT_PARSE;

   strncpy(tmpaddr, str_addr, sizeof(tmpaddr)-1);
   tmpaddr[sizeof(tmpaddr)-1] = '\0';
#ifdef INET6
   /* Not yet implemented */
   return CANT_PARSE;
#else

   if ( inet_aton( tmpaddr, &t_addr ) == 0)
      return CANT_PARSE;

   addr = ntohl(t_addr.s_addr);

   if ( addr == 0 )
      mask = 0 ;         /* i.e. matches everything */
   else
   {
      for ( mask = 0xFF ;; )
      {
         if ( addr & mask )
            break ;
         mask <<= 8 ;
         mask |= 0xFF ;
      }
      mask = ~( mask >> 8 ) ;
   }
   ca.mask = mask ;
   ca.addr = addr ;
   ca.addr_type = NUMERIC_ADDR ;
   return( ( (*op)( addr_list, &ca ) == OK ) ? PARSED : ERROR ) ;
#endif /* INET6 */
}


/*
 * Try to parse 'str_addr' as a symbolic host name
 * Apply 'op' to the 'addrlist' for *all* IP addresses of the host
 */
PRIVATE result_e host_addr( char *str_addr, status_e (*op)(), pset_h addr_list )
{
   struct hostent *hep ;
   struct comp_addr ca ;
   char **ap ;
   char tmpaddr[MAXHOSTNAMELEN+1];


#ifdef INET6
   struct addrinfo hints, *res, *ressave;
   struct sockaddr_in6 remsa;
   char v6addr[46];
   int n;

   if (strchr(str_addr, '/')) /* Don't even try explicit masks */
      return CANT_PARSE;
   if(strchr(str_addr, '{'))  /* Don't even try factorized */
      return CANT_PARSE;
   bzero(&hints, sizeof(hints));
   hints.ai_flags = AI_CANONNAME;
   hints.ai_family = AF_INET6;

   strncpy(tmpaddr, str_addr, sizeof(tmpaddr)-1);
   tmpaddr[sizeof(tmpaddr)-1] = '\0';

   if( (n = getaddrinfo(tmpaddr, NULL, &hints, &res)) < 0 ) {
      hints.ai_family = AF_INET;

      if( (n = getaddrinfo(tmpaddr, NULL, &hints, &res)) < 0 ) 
         return CANT_PARSE;

      bzero(v6addr, 46);
      hints.ai_family = AF_INET6;
      xaddrmap( inet_ntoa(((struct sockaddr_in *)(res->ai_addr))->sin_addr), v6addr);
      freeaddrinfo(res);
      if( (n = getaddrinfo(v6addr, NULL, &hints, &res)) < 0 )
         return( CANT_PARSE );
   }

   ca.addr_type = HOST_ADDR ;

   ressave = res;
   while( res != NULL ) {
      
      memcpy(&(ca.addr), &((struct sockaddr_in6 *)(res->ai_addr))->sin6_addr, 
         sizeof(ca.addr));
      xsetmask((char*)&(ca.mask), 128, sizeof(ca.mask));

      if ( (*op)( addr_list, &ca ) == FAILED ) {
         freeaddrinfo(res);
         return( ERROR ) ;
      }
   
      res = res->ai_next;
   }

   freeaddrinfo(res);
   return( PARSED );
#else
   if (strchr(str_addr, '/')) /* Don't even try explicit masks */
      return CANT_PARSE;
   if(strchr(str_addr, '{'))  /* Don't even try factorized */
      return CANT_PARSE;
   if(strchr(str_addr, ':'))  /* Don't even try IPv6 if machine is dual boot */
      return CANT_PARSE;

   strncpy(tmpaddr, str_addr, sizeof(tmpaddr)-1);
   tmpaddr[sizeof(tmpaddr)-1] = '\0';

   if( tmpaddr[0] == '.' ) {
      ca.addr_type = HOST_ADDR;
      strncpy(ca.name, tmpaddr, sizeof(ca.name)-1) ;
      ca.name[sizeof(ca.name)-1] = '\0';
      if ( (*op)( addr_list, &ca ) == FAILED )
         return( ERROR ) ;
      return( PARSED );
   }

   hep = gethostbyname( tmpaddr ) ;
   if ( hep == NULL || hep->h_addrtype != AF_INET ) {
      return( CANT_PARSE ) ;
   }

   strncpy(ca.name, tmpaddr, sizeof(ca.name)-1) ;
   ca.name[sizeof(ca.name)-1] = '\0';
   if( isdigit(tmpaddr[strlen(tmpaddr)-1]) ) 
      ca.addr_type = NUMERIC_ADDR;
   else 
      ca.addr_type = HOST_ADDR ;

   for ( ap = hep->h_addr_list ; *ap ; ap++ )
   {
      struct in_addr inaddr ;

      /*
       * Copy the address to avoid alignment problems
       */
      (void) memcpy( (char *) &inaddr, *ap, hep->h_length ) ;

      ca.addr = ntohl( inaddr.s_addr ) ; 
      ca.mask = 0xFFFFFFFF ;
      if ( (*op)( addr_list, &ca ) == FAILED )
         return( ERROR ) ;
   }
   return( PARSED ) ;
#endif
}


/*
 * Try to parse 'str_addr' as a factorized address
 * (for example, 128.138.{200,201})
 *
 * XXX: It is unclear whether this function should exist. It is really doing
 *      the job of a preprocessor.
 */
PRIVATE result_e factorized_addr( char *str_addr, 
                                  status_e (*op)(), 
                                  pset_h addr_list )
{
   char               *p ;
   char               *fact_start ;
   int                pass ;
   char               last   = DOT ;
   unsigned           num   = 0 ;
   int                shift = 24 ;   /* because we assume a 32-bit IP address */
   uint32_t           addr = 0 ;
   struct comp_addr   ca ;
   char               *func = "factorized_addr" ;

#ifdef INET6
   /* Not implemented */
   return(CANT_PARSE);
#else

   for ( p = str_addr ; *p != OPEN_CURLY_BRACKET ; last = *p++ )
   {
      if ( isdigit( *p ) )
      {
         num = num * 10 + *p - '0' ;
         continue ;
      }
      switch ( *p )
      {
         case DOT:
            if ( last == DOT )
            {
               parsemsg( LOG_ERR, func,
                  "Bad address: %s. Consecutive dots", str_addr ) ;
               return( ERROR ) ;
            }
            addr = addr * 256 + num ;
            num = 0 ;
            shift -= 8 ;
            break ;
         
         default:
            return( CANT_PARSE ) ;
      }
   }

   ca.addr_type = NUMERIC_ADDR ;
   fact_start = p+1 ;
   if ( addr != 0 )
      addr <<= ( shift+8 ) ;

   /*
    * First pass is for syntax checking
    */
   for ( pass = 0 ; pass < 2 ; pass++ )
   {
      num = 0 ;
      for ( p = fact_start, last = COMMA ;; last = *p++ )
      {
         if ( isdigit( *p ) )
         {
            num = num * 10 + *p - '0' ;
            continue ;
         }
         switch ( *p )
         {
            case COMMA:
            case CLOSED_CURLY_BRACKET:
               if ( last == COMMA )
               {
                  parsemsg( LOG_ERR, func,
                     "Bad address: %s. Consecutive commas", str_addr ) ;
                  return( ERROR ) ;
               }

               if ( pass == 1 )
               {
                  ca.addr = addr + ( num << shift ) ;
                  ca.mask = ~( ( 1 << shift ) - 1 ) ;
                  if ( (*op)( addr_list, &ca ) == FAILED )
                     return( ERROR ) ;
                  num = 0 ;
               }
               break ;
            
            default:
               parsemsg( LOG_ERR, func, "Bad address: %s", str_addr ) ;
               return( ERROR ) ;
         }
         if ( *p == CLOSED_CURLY_BRACKET )
         {
            if ( p[1] != NUL )
            {
               parsemsg( LOG_ERR, func, "Bad address: %s", str_addr ) ;
               return( ERROR ) ;
            }

            if ( pass == 0 )
               break ;
            else
               return( PARSED ) ;
         }
      }
   }
   /* NOTREACHED */
#endif /* INET6 */

   return( ERROR );
}


/*
 * Try to parse 'str_addr' using all known methods.
 * Try until one of the methods succeeds.
 * A successful method will apply 'op' with the parsed address to the 
 * 'addr_list'. The 'op' can be either 'add' or 'remove'
 * This means that the parsed address will be either added or removed
 * from the addr_list.
 */
PRIVATE status_e addrlist_op( pset_h addr_list, 
                              status_e (*op)(), 
                              char *str_addr )
{
   int i ;
   static result_e (*addr_parser[])() =
      {
         numeric_addr,
         host_addr,
         explicit_mask,
         factorized_addr,
         net_addr,
         NULL
      } ;
   char *func = "addrlist_op" ;

   for ( i = 0 ; addr_parser[ i ] != NULL ; i++ )
      switch ( (*addr_parser[ i ])( str_addr, op, addr_list ) )
      {
         case PARSED:
            return( OK ) ;
         case ERROR:
            return( FAILED ) ;
         case CANT_PARSE:
            break;
      }

   parsemsg( LOG_ERR, func, "failed to parse %s", str_addr ) ;
   return( OK ) ;
}


status_e addrlist_add( pset_h addr_list, char *str_addr ) 
{
   return( addrlist_op( addr_list, add, str_addr ) ) ;
}


status_e addrlist_remove( pset_h addr_list, char *str_addr ) 
{
   return( addrlist_op( addr_list, xremove, str_addr ) ) ;
}


status_e addrlist_copy( pset_h from, pset_h *to )
{
   status_e copy_pset() ;

   return( copy_pset( from, to, sizeof( struct comp_addr ) ) ) ;
}

