#include <sys/types.h>
#include <dirent.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <syslog.h>
#include <stdio.h>
#include "pset.h"
#include "conf.h"

#if !defined(NAME_MAX)
	#ifdef FILENAME_MAX
	#define NAME_MAX FILENAME_MAX
	#else
	#define NAME_MAX 256
	#endif
#endif

static int compfunc( char **a, char **b )
{
	if( a == NULL || a[0] == NULL )
		return -1;
	if( b == NULL || b[0] == NULL )
		return 1;
	return strcmp(a[0], b[0]);
}

void handle_includedir(char *service_name, struct configuration *confp)
{
	char *filename;
	pset_h dir_list;
	DIR *dirfp;
	struct dirent *direntry;
	char *storename;
	struct stat sb;
	int u, incfd;
	char *func = "handle_includedir";

	if( service_name == NULL )
		return;

	dir_list = pset_create(0, 0);
	if( dir_list == NULL )
		return;

	filename = (char *)malloc(strlen(service_name) + NAME_MAX + 2);
	if (! filename) {
		parsemsg( LOG_ERR, func, "includedir: Out of memory\n");
		return;
	}
	memset(filename, 0, strlen(service_name) + NAME_MAX+2);

	strncpy(filename,service_name, strlen(service_name));
	filename[strlen(filename)] = '/';
	dirfp = opendir(service_name);
	if (! dirfp) {
		parsemsg( LOG_ERR, func, "Unable to read included directory: %s", service_name);
		free(filename);
		return;
	}
	/* Get the list of files in the directory */
	while ((direntry = readdir(dirfp))) {
		storename = (char *)malloc(strlen(direntry->d_name)+1);
		if( storename == NULL ) {
			parsemsg( LOG_ERR, func, "Out of memory\n" );
			return;
		}
		strcpy(storename, direntry->d_name);
		storename[strlen(direntry->d_name)] = '\0';
		pset_add(dir_list, storename);
	}

	/* Sort the list using "compfunc" */
	pset_sort(dir_list, compfunc);

	/* Now, traverse the list in alphabetic order 
	 * (as determined by strcmp).
	 */
	for( u = 0; u < pset_count(dir_list); u++ ) {
		char *storename = pset_pointer(dir_list, u);

		memset(filename, 0, strlen(service_name) + NAME_MAX+2);

		strncpy(filename,service_name, strlen(service_name));
		filename[strlen(filename)] = '/';
		strncpy(filename + strlen(service_name) + 1,
			storename, NAME_MAX);

		/* Don't try to parse any file beginning with a '.'
		* This catches the case of '.' and '..', as well as preventing
		* the parsing of other files beginning with a .
		*/
		if ( storename[0] == '.' )
			continue;
	
		if( lstat(filename, &sb) < 0 ) {
			parsemsg( LOG_ERR, func, "Unable to stat includedir file %s\n", filename);
			continue;
		}
	
		/* Only open it if it's a regular file. */
		if( !S_ISREG(sb.st_mode) )
			continue;
	
		incfd = open(filename, O_RDONLY);
		if( incfd < 0 ) {
			parsemsg( LOG_ERR, func, "Unable to open included configuration file: %s", filename);
			continue;
		}
		parsemsg( LOG_DEBUG,func,"Reading included configuration file: %s",filename);
		parse_conf_file(incfd, confp);
		close(incfd);
		free(storename);
	}
	if ( errno != 0) {
		parsemsg( LOG_ERR, func, "Error reading included directory: %s", service_name);
	}
	pset_destroy(dir_list);
	closedir(dirfp);
	free(filename);
}
