/*
 * (c) Copyright 1992 by Panagiotis Tsirigotis
 * All rights reserved.  The file named COPYRIGHT specifies the terms 
 * and conditions for redistribution.
 */

static char RCSid[] = "$Id: builtins.c,v 1.1.1.1 1999/10/12 17:28:59 bbraun Exp $" ;

#include <sys/types.h>
#include <sys/socket.h>
#include <errno.h>
#include <time.h>
#include <syslog.h>
#include <stdlib.h>
#include <unistd.h>

#include "str.h"

#include "config.h"
#include "state.h"
#include "defs.h"
#include "builtin.h"
#include "sconf.h"
#include "server.h"
#include "service.h"
#include "access.h"

extern int errno ;

time_t time() ;

void msg() ;
status_e write_buf() ;

#define BUFFER_SIZE					1024

PRIVATE void stream_echo() ;
PRIVATE void dgram_echo() ;
PRIVATE void stream_discard() ;
PRIVATE void dgram_discard() ;
PRIVATE void stream_time() ;
PRIVATE void dgram_time() ;
PRIVATE void stream_daytime() ;
PRIVATE void dgram_daytime() ;
PRIVATE void stream_chargen() ;
PRIVATE void dgram_chargen() ;
PRIVATE void stream_servers() ;
PRIVATE void stream_services() ;
PRIVATE void xadmin_handler() ;
PRIVATE int bad_port_check() ;


static struct builtin_service builtin_services[] =
	{
		{ "echo",		SOCK_STREAM,	{ stream_echo, 		FORK  	} },
		{ "echo",		SOCK_DGRAM,		{ dgram_echo,			NO_FORK	} },
		{ "discard",	SOCK_STREAM,	{ stream_discard,		FORK		} },
		{ "discard",	SOCK_DGRAM,		{ dgram_discard,		NO_FORK	} },
		{ "time",		SOCK_STREAM,	{ stream_time,			NO_FORK	} },
		{ "time",		SOCK_DGRAM,		{ dgram_time,			NO_FORK	} },
		{ "daytime",	SOCK_STREAM,	{ stream_daytime,		NO_FORK	} },
		{ "daytime",	SOCK_DGRAM,		{ dgram_daytime,		NO_FORK	} },
		{ "chargen",	SOCK_STREAM,	{ stream_chargen,		FORK		} },
		{ "chargen",	SOCK_DGRAM,		{ dgram_chargen,		NO_FORK	} },
		{ "servers",	SOCK_STREAM,	{ stream_servers,		FORK		} },
		{ "services",	SOCK_STREAM,	{ stream_services,	FORK		} },
		{ "xadmin",    SOCK_STREAM,   { xadmin_handler,    FORK     } },
		{ NULL }
	} ;


builtin_s *builtin_find( service_name, type )
	char	*service_name ;
	int	type ;
{
	register builtin_s	*bsp ;
	char						*func = "builtin_find" ;
	
	if ( (bsp = builtin_lookup( builtin_services, service_name, type )) )
		return( bsp ) ;
	
	msg( LOG_ERR, func, "No such internal service: %s", service_name ) ;
	return( NULL ) ;
}


builtin_s *builtin_lookup( services, service_name, type )
	struct builtin_service	services[] ;
	register char				*service_name ;
	register int				type ;
{
	register struct builtin_service *bsp ;
	
	for ( bsp = services ; bsp->bs_name != NULL ; bsp++ )
		if ( EQ( bsp->bs_name, service_name ) && bsp->bs_socket_type == type )
			return( &bsp->bs_handle ) ;
	return( NULL ) ;
}


/*
 * The rest of this file contains the functions that implement the 
 * builtin services
 */


PRIVATE void stream_echo( serp )
	struct server *serp ;
{
	char				buf[ BUFFER_SIZE ] ;
	register int	cc ;
	int				descriptor = SERVER_FD( serp ) ;

	for ( ;; )
	{
		cc = read( descriptor, buf, sizeof( buf ) ) ;
		if ( cc == 0 )
			break ;
		if ( cc == -1 ) {
			if ( errno == EINTR )
				continue ;
			else
				break ;
		}

		if ( write_buf( descriptor, buf, cc ) == FAILED )
			break ;
	}
}

/* For internal UDP services, make sure we don't respond to
 * ports our ports on other servers.  This can cause looping.
 */
PRIVATE int bad_port_check(struct sockaddr_in *sa)
{
	unsigned short bad_ports[] = { 7, 9, 13, 19, 37, 0 };
	unsigned short port;
	int i;

	if( sa->sin_family == AF_INET )
		port = ntohs(sa->sin_port);
#ifdef INET6
	else if( sa->sin_family == AF_INET6 )
		port = ntohs( ((struct sockaddr_in6 *)(sa))->sin6_port);
#endif
	
	for( i=0; bad_ports[i] != 0; i++ ) {
		if( port == bad_ports[i] ) {
			return (-1);
		}
	}

	return (0);
}

PRIVATE void dgram_echo( serp )
	struct server *serp ;
{
	char			buf[ DATAGRAM_SIZE ] ;
	struct sockaddr_in	sin ;
	int			cc ;
	int			sin_len = sizeof( sin ) ;
	int			descriptor = SERVER_FD( serp ) ;
	char *func = "dgram_echo";

	cc = recvfrom( descriptor, buf, sizeof( buf ), 0, SA( &sin ), &sin_len ) ;
	if ( cc != -1 ) {
		if( bad_port_check(&sin) != 0 ) {
			if( sin.sin_family == AF_INET )
				msg(LOG_WARNING, func, 
				"Possible Denial of Service attack from %s %d", 
				inet_ntoa(sin.sin_addr), ntohs(sin.sin_port));
			return;
		}
		(void) sendto( descriptor, buf, cc, 0, SA( &sin ), sizeof( sin ) ) ;
	}
}


PRIVATE void stream_discard( serp )
	struct server *serp ;
{
	char				buf[ BUFFER_SIZE ] ;
	register int	cc ;

	for ( ;; )
	{
		cc = read( SERVER_FD( serp ), buf, sizeof( buf ) ) ;
		if ( (cc == 0) || ((cc == -1) && (errno != EINTR)) )
			break ;
	}
}


PRIVATE void dgram_discard( serp )
	struct server *serp ;
{
	char buf[ 1 ] ;

	(void) recv( SERVER_FD( serp ), buf, sizeof( buf ), 0 ) ;
}



/*
 * Generate the current time using the SMTP format:
 *		02 FEB 1991 12:31:42 MST
 *
 * The result is placed in buf.
 * buflen is a value-result parameter. It indicates the size of
 * buf and on exit it has the length of the string placed in buf.
 */
PRIVATE void daytime_protocol( buf, buflen )
	char	*buf ;
	int	*buflen ;
{
	static char *month_name[] =
		{
			"JAN", "FEB", "MAR", "APR", "MAY", "JUN",
			"JUL", "AUG", "SEP", "OCT", "NOV", "DEC"
		} ;
	time_t		now ;
	struct tm	*tmp ;
	int			size = *buflen ;
#ifdef HAVE_STRFTIME
	int		   cc ;
#endif
	
	(void) time( &now ) ;
	tmp = localtime( &now ) ;
#ifndef HAVE_STRFTIME
	strx_print( buflen, buf, size,
		"%02d %s %d %02d:%02d:%02d %s\r\n",
			tmp->tm_mday, month_name[ tmp->tm_mon ], 1900 + tmp->tm_year,
				tmp->tm_hour, tmp->tm_min, tmp->tm_sec, tmp->tm_zone ) ;
#else
	cc = strx_nprint( buf, size,
		"%02d %s %d %02d:%02d:%02d",
			tmp->tm_mday, month_name[ tmp->tm_mon ], 1900 + tmp->tm_year,
				tmp->tm_hour, tmp->tm_min, tmp->tm_sec ) ;
	*buflen = cc ;
	size -= cc ;
	cc = strftime( &buf[ *buflen ], size, " %Z\r\n", tmp ) ;
	*buflen += cc ;
#endif
}


PRIVATE void stream_daytime( serp )
	struct server *serp ;
{
	char	time_buf[ BUFFER_SIZE ] ;
	int	buflen = sizeof( time_buf ) ;

	daytime_protocol( time_buf, &buflen ) ;
	(void) write_buf( SERVER_FD( serp ), time_buf, buflen ) ;
}


PRIVATE void dgram_daytime( serp )
	struct server *serp ;
{
	char			time_buf[ BUFFER_SIZE ] ;
	struct sockaddr_in	sin ;
	int			sin_len		= sizeof( sin ) ;
	int			buflen		= sizeof( time_buf ) ;
	int			descriptor	= SERVER_FD( serp ) ;
	char *func = "dgram_daytime";

	if ( recvfrom( descriptor, time_buf, sizeof( time_buf ), 0,
				SA( &sin ), &sin_len ) == -1 )
		return ;

	if( bad_port_check(&sin) != 0 ) {
		if( sin.sin_family == AF_INET )
			msg(LOG_WARNING, func, 
				"Possible Denial of Service attack from %s %d", 
				inet_ntoa(sin.sin_addr), ntohs(sin.sin_port));
		return;
	}

	daytime_protocol( time_buf, &buflen ) ;
	
	(void) sendto( descriptor, time_buf, buflen, 0, SA(&sin), sizeof( sin ) ) ;
}


#define TIME_OFFSET			2208988800UL

/*
 * We always report the time using network-byte-order
 */
PRIVATE void time_protocol( timep )
	register unsigned long *timep ;
{
	time_t now ;

	(void) time( &now ) ;
	*timep = now + TIME_OFFSET ;
	*timep = htonl( *timep ) ;
}


PRIVATE void stream_time( serp )
	struct server *serp ;
{
	unsigned long now ;

	time_protocol( &now ) ;
	(void) write_buf( SERVER_FD( serp ), (char *) &now, sizeof( now ) ) ;
}


PRIVATE void dgram_time( serp )
	struct server *serp ;
{
	char			buf[ 1 ] ;
	unsigned long		now ;
	struct sockaddr_in	sin ;
	int			sin_len	= sizeof( sin ) ;
	int			fd			= SERVER_FD( serp ) ;
	char *func = "dgram_daytime";

	if ( recvfrom( fd, buf, sizeof( buf ), 0, SA( &sin ), &sin_len ) == -1 )
		return ;

	if( bad_port_check(&sin) != 0 ) {
		if( sin.sin_family == AF_INET )
			msg(LOG_WARNING, func, 
				"Possible Denial of Service attack from %s %d", 
				inet_ntoa(sin.sin_addr), ntohs(sin.sin_port));
		return;
	}

	time_protocol( &now ) ;
	(void) sendto( fd, (char *) &now, sizeof( now ), 0, SA( &sin ), sin_len ) ;
}


#define ASCII_PRINTABLE_CHARS				94
#define LINE_LENGTH							72

#define RING_BUF_SIZE						ASCII_PRINTABLE_CHARS + LINE_LENGTH

static char ring_buf[ RING_BUF_SIZE ] ;
static char *ring ;


#define ASCII_START				( ' ' + 1 )
#define ASCII_END					126

#define min( a, b )				((a)<(b) ? (a) : (b))

PRIVATE char *generate_line( buf, len )
	char	*buf ;
	int	len ;
{
	int line_len = min( LINE_LENGTH, len-2 ) ;

	if ( line_len < 0 )
		return( NULL ) ;

	if ( ring == NULL )
	{
		register char ch ;
		register char *p ;

		for ( p = ring_buf, ch = ASCII_START ;
							p < &ring_buf[ RING_BUF_SIZE ] ; p++ )
		{
			*p = ch++ ;
			if ( ch == ASCII_END )
				ch = ASCII_START ;
		}
		ring = ring_buf ;
	}
	(void) memcpy( buf, ring, line_len ) ;
	buf[ line_len   ] = '\r' ;
	buf[ line_len+1 ] = '\n' ;

	ring++ ;
	if ( &ring_buf[ RING_BUF_SIZE ] - ring < LINE_LENGTH )
		ring = ring_buf ;
	return( buf ) ;
}


PRIVATE void stream_chargen( serp )
	struct server *serp ;
{
	char	line_buf[ LINE_LENGTH+2 ] ;
	int	descriptor = SERVER_FD( serp ) ;

	(void) shutdown( descriptor, 0 ) ;
	for ( ;; )
	{
		if ( generate_line( line_buf, sizeof( line_buf ) ) == NULL )
			break ;
		if ( write_buf( descriptor, line_buf, sizeof( line_buf ) ) == FAILED )
			break ;
	}
}


PRIVATE void dgram_chargen( serp )
	struct server *serp ;
{
	char			buf[ BUFFER_SIZE ] ;
	register char		*p ;
	register int		len ;
	struct sockaddr_in	sin ;
	int			sin_len	= sizeof( sin ) ;
	int			fd		= SERVER_FD( serp ) ;
	register int		left		= sizeof( buf ) ;
	char *func = "dgram_chargen";

	if ( recvfrom( fd, buf, sizeof( buf ), 0, SA( &sin ), &sin_len ) == -1 )
		return ;

#if BUFFER_SIZE < LINE_LENGTH+2
	bad_variable = 1 ;		/* this will cause a compilation error */
#endif

	if( bad_port_check(&sin) != 0 ) {
		if( sin.sin_family == AF_INET )
			msg(LOG_WARNING, func, 
				"Possible Denial of Service attack from %s %d", 
				inet_ntoa(sin.sin_addr), ntohs(sin.sin_port));
		return;
	}

	for ( p = buf ; left > 2 ; left -= len, p += len )
	{
		len = min( LINE_LENGTH+2, left ) ;
		if ( generate_line( p, len ) == NULL )
			break ;
	}
	(void) sendto( fd, buf, p-buf, 0, SA( &sin ), sin_len ) ;
}


PRIVATE void stream_servers( this_serp )
	struct server *this_serp ;
{
	register unsigned		u ;
	int						descriptor = SERVER_FD( this_serp ) ;

	for ( u = 0 ; u < pset_count( SERVERS( ps ) ) ; u++ )
	{
		struct server *serp = SERP( pset_pointer( SERVERS( ps ), u ) ) ;

		/*
		 * We cannot report any useful information about this server because
		 * the data in the server struct are filled by the parent.
		 */
		if ( serp == this_serp )
			continue ;

		server_dump( serp, descriptor ) ;
	}
}


PRIVATE void stream_services( serp )
	struct server *serp ;
{
	register unsigned u ;
	int fd = SERVER_FD( serp ) ;

	for ( u = 0 ; u < pset_count( SERVICES( ps ) ) ; u++ )
	{
		int cc ;
		char buf[ BUFFER_SIZE ] ;
		struct service_config *scp ;
		
		scp = SVC_CONF( SP( pset_pointer( SERVICES( ps ), u ) ) ) ;

		strx_print( &cc, buf, sizeof( buf ), "%s %s %d\n",
			SC_NAME( scp ), SC_PROTONAME( scp ), SC_PORT( scp ) ) ;
			
		if ( write_buf( fd, buf, cc ) == FAILED )
			break ;
	}
}

#define MAX_CMD 100

/* Do the redirection of a service */
/* This function gets called from child.c after we have been forked */
PRIVATE void xadmin_handler( struct server *serp )
{
	struct service *sp = SERVER_SERVICE( serp );
	int descriptor = SERVER_FD( serp );
	char cmd[MAX_CMD]; 
	int red = 0, u = 0;
	
	while(1)
	{
		Sprint(descriptor, "> ");
		Sflush();
		bzero(cmd, MAX_CMD);
		red = read(descriptor, cmd, MAX_CMD);
		if( red < 0 )
		{
			syslog(LOG_ERR, "xadmin:reading command: %s", strerror(errno));
			exit(1);
		}
		if( red > MAX_CMD )
		{
			/* shouldn't ever happen */
			syslog(LOG_ERR, "xadmin:reading command: read more bytes than MAX_CMD");
			continue;
		}

		if( red == 0 )
			exit(0);

	  if( (strncmp(cmd, "bye",(red<3)?red:3) == 0) ||
	    (strncmp(cmd, "exit", (red<4)?red:4) == 0) ||
	    (strncmp(cmd, "quit", (red<4)?red:4) == 0) )
	  {
		  Sprint(descriptor, "bye bye\n");
		  Sflush();
		  close(descriptor);
		  exit(0);
	  }

     if( strncmp(cmd, "help", (red<4)?red:4) == 0 ) 
     {
        Sprint(descriptor, "xinetd admin help:\n");
        Sprint(descriptor, "show run  :   shows information about running services\n");
        Sprint(descriptor, "show avail:   shows what services are currently available\n");
        Sprint(descriptor, "bye, exit :   exits the admin shell\n");
     }

	  if( strncmp(cmd, "show", (red<4)?red:4) == 0 )
	  {
		  char *tmp = cmd+4;
		  red -= 4;
		  if( tmp[0] == ' ' )
		  {
		  	tmp++;
			red--;
		  }

		  if( red <= 0 )
		  	continue;

		  if( strncmp(tmp, "run", (red<3)?red:3) == 0 )
		  {
			  Sprint(descriptor, "Running services:\n");
           Sprint(descriptor, "service  run retry attempts descriptor\n");
			  for( u = 0 ; u < pset_count( SERVERS( ps ) ) ; u++ )
			  {
				  struct service *sp = 
					(SERP( pset_pointer( SERVERS(ps), u ) ))->svr_sp;
               
              server_dump( SERP( pset_pointer( SERVERS(ps), u ) ), descriptor );
  
#ifdef FOO
				  Sprint(descriptor, "%-10s %-3d %-5d %-7d %-5d\n", 
					(char *)SVC_ID( sp ), sp->svc_running_servers+1, 
					sp->svc_retry_servers, sp->svc_attempts, sp->svc_fd );
#endif
			  }
		  }
		  else if( strncmp(tmp, "avail", (red<5)?red:5) == 0 )
		  {
		
			  Sprint(descriptor, "Available services:\n");
			  Sprint(descriptor, "service    port   bound address    uid redir addr redir port\n");

			  for( u = 0 ; u < pset_count( SERVICES( ps ) ) ; u++ )
			  {
				  struct service *sp=NULL;
#ifdef INET6
                  struct sockaddr_in6 baddr;
                  struct sockaddr_in6 raddr;
#else
                  struct sockaddr_in baddr;
                  struct sockaddr_in raddr;
#endif

                  bzero(&baddr, sizeof(baddr));
                  bzero(&raddr, sizeof(raddr));

				  sp = SP( pset_pointer( SERVICES( ps ), u ) );

				  if( SVC_CONF(sp)->sc_bind_addr != NULL )
#ifdef INET6
                    memcpy(&baddr.sin6_addr, &SVC_CONF(sp)->sc_bind_addr, sizeof(baddr));
#else
                    memcpy(&baddr.sin_addr, &SVC_CONF(sp)->sc_bind_addr, sizeof(baddr));
#endif
  
				 if( SVC_CONF(sp)->sc_redir_addr != NULL )
				 {
#ifdef INET6
                    memcpy(&raddr.sin6_addr, &SVC_CONF(sp)->sc_bind_addr, sizeof(raddr));
#else
                    memcpy(&raddr.sin_addr, &SVC_CONF(sp)->sc_bind_addr, sizeof(raddr));
#endif
				   Sprint(descriptor, "%-10s ", SC_NAME( SVC_CONF( sp ) ) );
				   Sprint(descriptor, "%-6d ", SC_PORT( SVC_CONF( sp ) ) );
				   Sprint(descriptor, "%-16s ", xntoa( baddr ) );
				   Sprint(descriptor, "%-6d ", SVC_CONF(sp)->sc_uid );
				   Sprint(descriptor, "%-16s ", xntoa(raddr) );
				   Sprint(descriptor, "%-6d\n", SVC_CONF(sp)->sc_redir_port );
				 }
				 else
				 {
				   Sprint(descriptor, "%-10s ", SC_NAME( SVC_CONF( sp ) ) );
				   Sprint(descriptor, "%-6d ", SC_PORT( SVC_CONF( sp ) ) );
				   Sprint(descriptor, "%-16s ", xntoa( baddr ) );
				   Sprint(descriptor, "%-6d\n", SVC_CONF(sp)->sc_uid );
				 }

			  }
		  }
		  else
		  {
			  Sprint(descriptor, "Relevant commands:\n");
			  Sprint(descriptor, "run   : Show currently running servers\n");
			  Sprint(descriptor, "avail : Show currently available servers\n");
		  }
	  }
  }
	Sprint(descriptor, "exiting\n");
	Sflush();
}
