/* Copyright (C) 2003 Red Hat, Inc. */
/* Licensed under the terms of the GPL */
/* Written by Elliot Lee <sopwith@redhat.com> */
/* New personality options & code added by Jindrich Novy <jnovy@redhat.com> */
/* ADD_NO_RANDOMIZE flag added by Arjan van de Ven <arjanv@redhat.com> */
/* based on ideas from the ppc32 util by Guy Streeter (2002-01), based on the
   sparc32 util by Jakub Jelinek (1998, 1999) */
/* */

#include <syscall.h>
#include <linux/personality.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/utsname.h>

#define set_pers(pers) ((long)syscall(SYS_personality, pers))

struct {
    char c;
    const char *name;
    unsigned int option;
} flags[] = {
    {'R', "ADDR_NO_RANDOMIZE",  0x0040000},
    {'F', "FDPIC_FUNCPTRS",     0x0080000},
    {'Z', "MMAP_PAGE_ZERO",     0x0100000},
    {'L', "ADDR_COMPAT_LAYOUT", 0x0200000},
    {'X', "READ_IMPLIES_EXEC",  0x0400000},
    {'B', "ADDR_LIMIT_32BIT",   0x0800000},
    {'I', "SHORT_INODE",        0x1000000},
    {'S', "WHOLE_SECONDS",      0x2000000},
    {'T', "STICKY_TIMEOUTS",    0x4000000},
    {'3', "ADDR_LIMIT_3GB",     0x8000000}
};

int
set_arch(const char *pers, unsigned long options)
{
  struct utsname un;
  int i;
  unsigned long pers_value, res;

  struct {
    int perval;
    char *target_arch, *result_arch;
  } transitions[] = {
    {PER_LINUX32, "linux32", NULL},
#if defined(__powerpc__) || defined(__powerpc64__)
    {PER_LINUX32, "ppc32", "ppc"},
    {PER_LINUX32, "ppc", "ppc"},
    {PER_LINUX, "ppc64", "ppc64"},
    {PER_LINUX, "ppc64pseries", "ppc64"},
    {PER_LINUX, "ppc64iseries", "ppc64"},
#endif
#if defined(__x86_64__) || defined(__i386__) || defined(__ia64__)
    {PER_LINUX32, "i386", "i386"},
    {PER_LINUX32, "i486", "i386"},
    {PER_LINUX32, "i586", "i386"},
    {PER_LINUX32, "i686", "i386"},
    {PER_LINUX32, "athlon", "i386"},
#endif
#if defined(__x86_64__) || defined(__i386__)
    {PER_LINUX, "x86_64", "x86_64"},
#endif
#if defined(__ia64__) || defined(__i386__)
    {PER_LINUX, "ia64", "ia64"},
#endif
#if defined(__s390x__) || defined(__s390__)
    {PER_LINUX32, "s390", "s390"},
    {PER_LINUX, "s390x", "s390x"},
#endif
#if defined(__sparc64__) || defined(__sparc__)
    {PER_LINUX32, "sparc", "sparc"},
    {PER_LINUX, "sparc64", "sparc64"},
#endif
    {-1, NULL, NULL}
  };

  for(i = 0; transitions[i].perval >= 0; i++)
    {
      if(!strcmp(pers, transitions[i].target_arch))
	break;
    }

  if(transitions[i].perval < 0)
    {
      fprintf(stderr, "Don't know how to set arch to %s\n", pers);
      exit(1);
    }

  pers_value = transitions[i].perval | options;
  res = set_pers(pers_value);
  if(res == -EINVAL)
    return 1;

  uname(&un);
  if(transitions[i].result_arch &&
	strcmp(un.machine, transitions[i].result_arch))
    {
      if(strcmp(transitions[i].result_arch, "i386")
	 || (strcmp(un.machine, "i486")
	     && strcmp(un.machine, "i586")
	     && strcmp(un.machine, "i686")
	     && strcmp(un.machine, "athlon")))
	{
	  fprintf(stderr, "Don't know how to set arch to %s\n", pers);
	  exit(1);
	}
    }

  return 0;
}

int main(int argc, char *argv[])
{
  char *p = strrchr(argv[0], '/');
  unsigned long options = 0;
  int got_arch = 0, verbose = 0;

  p = p ? p + 1 : argv[0];

  if(argc <= 1) {
    fprintf(stderr, "Usage: %s %s program arguments\n",
	    p, !strcmp(p, "setarch")?"arch":"");
    return 1;
  }
  
  if(!strcmp(p, "setarch"))
    {
      argv++;
      argc--;
      p = argv[0];
      got_arch = 1;
    }
  argv++;
  argc--;
  
  for (; argc && argv[0][0] == '-'; argv++, argc--) {
    int n, unknown = 1;
    char *arg = argv[0];
    
    for (n = 1; arg[n]; n++) {
      int f;
      
      if (arg[n] == 'v') {
        verbose = 1;
        continue;
      }
    
      for (f = 0; f < sizeof(flags)/sizeof(flags[0]); f++) {
        if (arg[n] == flags[f].c) {
	  if (verbose)
	    fprintf(stderr,"Switching on %s.\n", flags[f].name);
	  options |= flags[f].option;
	  unknown = 0;
	  break;
	}
      }
      if (unknown)
        fprintf(stderr, "warning: unknown option `%c'\n", arg[n]);
    }
  }

  if(set_arch(p, options))
    {
      perror(argv[0]);
      return 1;
    }

  if(!argc)
    {
      execl("/bin/sh", "-sh", NULL);
      return 3;
    }

  execvp(argv[0], argv);
  return 1;
}
