/* Internal definitions for libasm.
   Copyright (C) 2002 Red Hat, Inc.

   This program is Open Source software; you can redistribute it and/or
   modify it under the terms of the Open Software License version 1.0 as
   published by the Open Source Initiative.

   You should have received a copy of the Open Software License along
   with this program; if not, you may obtain a copy of the Open Software
   License version 1.0 from http://www.opensource.org/licenses/osl.php or
   by writing the Open Source Initiative c/o Lawrence Rosen, Esq.,
   3001 King Ranch Road, Ukiah, CA 95482.   */

#ifndef _LIBASMP_H
#define _LIBASMP_H 1

#include <stdio.h>

#include <libasm.h>
#include <libebl.h>

/* gettext helper macros.  */
#define _(Str) dgettext ("libasm", Str)


/* Known error codes.  */
enum
  {
    ASM_E_NOERROR,
    ASM_E_NOMEM,		/* No more memory.  */
    ASM_E_CANNOT_CREATE,	/* Output file cannot be created.  */
    ASM_E_INVALID,		/* Invalid parameters.  */
    ASM_E_CANNOT_CHMOD,		/* Cannot change mode of output file.  */
    ASM_E_CANNOT_RENAME,	/* Cannot rename output file.  */
    ASM_E_DUPLSYM,		/* Duplicate symbol definition.  */
    ASM_E_LIBELF,		/* Refer to error in libelf.  */
    ASM_E_TYPE,			/* Invalid section type for operation.  */
    ASM_E_NUM			/* Keep this entry as the last.  */
  };


/* Special sections.  */
#define ASM_ABS_SCN ((Elf_Scn *) 1)
#define ASM_COM_SCN ((Elf_Scn *) 2)


/* And the hash table for symbols.  */
#include <symbolhash.h>


/* Descriptor for a section.  */
struct AsmScn
{
  /* The underlying assembler context.  */
  AsmCtx_t *ctx;

  /* Subsection ID.  */
  unsigned int subsection_id;

  /* Section type.  */
  GElf_Word type;

  union
  {
    /* Data only stored in the record for subsection zero.  */
    struct
    {
      /* The ELF section.  */
      Elf_Scn *scn;

      /* Entry in the section header string table.  */
      struct Ebl_Strent *strent;

      /* Next member of group.  */
      struct AsmScn *next_in_group;
    } main;

    /* Pointer to the record for subsection zero.  */
    AsmScn_t *up;
  } data;

  /* Current offset in the (sub)section.  */
  GElf_Off offset;
  /* Maximum alignment of the section so far.  */
  GElf_Word max_align;

  /* Section content.  */
  struct AsmData
  {
    /* Currently used number of bytes in the block.  */
    size_t len;

    /* Number of bytes allocated.  */
    size_t maxlen;

    /* Pointer to the next block.  */
    struct AsmData *next;

    /* The actual data.  */
    char data[flexarr_size];
  } *content;

  /* Fill pattern.  */
  struct FillPattern
  {
    size_t len;
    char bytes[flexarr_size];
  } *pattern;

  /* Next subsection.  */
  AsmScn_t *subnext;

  /* List of all allocated sections.  */
  AsmScn_t *allnext;

  /* Name of the section.  */
  char name[flexarr_size];
};


/* Descriptor used for the assembling session.  */
struct AsmCtx
{
  /* File descriptor of the temporary file.  */
  int fd;

  /* True if text output is wanted.  */
  bool textp;

  /* Output file handle.  */
  union
  {
    /* ELF descriptor of the temporary file.  */
    Elf *elf;
    /* I/O stream for text output.  */
    FILE *file;
  } out;


  /* List with defined sections.  */
  AsmScn_t *section_list;
  /* Section header string table.  */
  struct Ebl_Strtab *section_strtab;

  /* Table with defined symbols.  */
  asm_symbol_tab symbol_tab;
  /* Number of symbols in the table.  */
  unsigned int nsymbol_tab;
  /* Symbol string table.  */
  struct Ebl_Strtab *symbol_strtab;

  /* List of section groups.  */
  struct AsmScnGrp *groups;
  /* Number of section groups.  */
  size_t ngroups;

  /* Current required alignment for common symbols.  */
  GElf_Word common_align;

  /* Lock to handle multithreaded programs.  */
  rwlock_define (,lock);

  /* Counter for temporary symbols.  */
  unsigned int tempsym_count;

  /* Name of the output file.  */
  char *fname;
  /* The name of the temporary file.  */
  char tmp_fname[flexarr_size];
};


/* Descriptor for a symbol.  */
struct AsmSym
{
  /* Reference to the section which contains the symbol.  */
  AsmScn_t *scn;

  /* Type of the symbol.  */
  int8_t type;
  /* Binding of the symbol.  */
  int8_t binding;

  /* Size of the symbol.  */
  GElf_Xword size;

  /* Offset in the section.  */
  GElf_Off offset;

  /* Symbol table index of the symbol in the symbol table.  */
  size_t symidx;

  /* Reference to name of the symbol.  */
  struct Ebl_Strent *strent;
};


/* Descriptor for section group.  */
struct AsmScnGrp
{
  /* Entry in the section header string table.  */
  struct Ebl_Strent *strent;

  /* The ELF section.  */
  Elf_Scn *scn;

  /* The signature.  */
  struct AsmSym *signature;

  /* First member.  */
  struct AsmScn *members;
  /* Number of members.  */
  size_t nmembers;

  /* Flags.  */
  Elf32_Word flags;

  /* Next group.  */
  struct AsmScnGrp *next;

  /* Name of the section group.  */
  char name[flexarr_size];
};


/* The default fill pattern: one zero byte.  */
extern const struct FillPattern *__libasm_default_pattern
     attribute_hidden;


/* Ensure there are at least LEN bytes available in the output buffer
   for ASMSCN.  */
extern int __libasm_ensure_section_space (AsmScn_t *asmscn, size_t len)
     internal_function;

/* Free all resources associated with the assembler context.  */
extern void __libasm_finictx (AsmCtx_t *ctx) internal_function;

/* Set error code.  */
extern void __libasm_seterrno (int err) internal_function;

/* Return handle for the named section.  If it was not used before
   create it.  */
extern AsmScn_t *__asm_newscn_internal (AsmCtx_t *ctx, const char *scnname,
					GElf_Word type, GElf_Xword flags)
     attribute_hidden;


/* Internal aliases of the asm_addintXX functions.  */
extern int __asm_addint8_internal (AsmScn_t *asmscn, int8_t num)
     attribute_hidden;
extern int __asm_addint16_internal (AsmScn_t *asmscn, int16_t num)
     attribute_hidden;
extern int __asm_addint32_internal (AsmScn_t *asmscn, int32_t num)
     attribute_hidden;
extern int __asm_addint64_internal (AsmScn_t *asmscn, int64_t num)
     attribute_hidden;



/* Test whether given symbol is an internal symbol and if yes, whether
   we should nevertheless emit it in the symbol table.  */
// XXX The second part should probably be controlled by an option which
// isn't implemented yet
// XXX Also, the format will change with the backend.
#define asm_emit_symbol_p(name) (strncmp (name, ".L", 2) != 0)

#endif	/* libasmP.h */
