/*
 * QUOTA    An implementation of the diskquota system for the LINUX
 *          operating system. QUOTA is implemented using the BSD systemcall
 *          interface as the means of communication with the user level.
 *          Should work for all filesystems because of integration into the
 *          VFS layer of the operating system.
 *          This is based on the Melbourne quota system wich uses both user and
 *          group quota files.
 *
 *          This part does the lookup of the info.
 *
 * Version: $Id: rquota_server.c,v 2.16 2000/09/05 19:08:29 mvw Exp mvw $
 *
 * Author:  Marco van Wieringen <mvw@planets.elm.net>
 *
 *          This program is free software; you can redistribute it and/or
 *          modify it under the terms of the GNU General Public License
 *          as published by the Free Software Foundation; either version
 *          2 of the License, or (at your option) any later version.
 */
#include <rpc/rpc.h>
#include <rquota.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <sys/quota.h>
#include <sys/mount.h>
#include <dirent.h>
#include <paths.h>
#include <mntent.h>
#include <quotaops.h>
#include <stdio.h>
#include <syslog.h>
#include <tcpd.h>
#include <netdb.h>

#include "bylabel.h"

#define STDIN_FILENO	0

#define TYPE_EXTENDED	0x01
#define ACTIVE		0x02

#define FACILITY	LOG_LOCAL7

#ifndef MAXPATHNAMELEN
#define MAXPATHNAMELEN BUFSIZ
#endif

#define NETTYPE AF_INET

#ifdef HOSTS_ACCESS
#define good_client(a,b) hosts_ctl("rpc.rquotad", b, inet_ntoa(a->sin_addr), "")
#endif

int allow_severity = LOG_INFO;
int deny_severity = LOG_WARNING;

/*
 * Global unix authentication credentials.
 */
extern struct authunix_parms *unix_cred;

int in_group (gid_t *gids, u_int len, gid_t gid)
{
   gid_t *gidsp = gids + len;

   while (gidsp > gids)
      if (*(--gids) == gid)
         return 1;

   return 0;
}

setquota_rslt *setquotainfo(int flags, caddr_t *argp, struct svc_req *rqstp)
{
   static setquota_rslt result;
#if defined(RPC_SETQUOTA)
   union {
      setquota_args *args;
      ext_setquota_args *ext_args;
   } arguments;
   struct stat st;
   dev_t device;
   FILE *fp;
   struct dqblk dq_dqb;
   struct mntent *mnt;
   char *pathname, *qfpathname;
   int fd, id, qcmd,type;
   const char *mnt_fsname;

#ifdef HOSTS_ACCESS
   struct hostent *hp;
   struct sockaddr_in *addr;
   
   addr = (svc_getcaller(rqstp->rq_xprt));
   hp = gethostbyaddr((char *)&(addr->sin_addr), sizeof(addr->sin_addr), AF_INET);
   
   if (!good_client(svc_getcaller(rqstp->rq_xprt),hp->h_name)) {
      result.status = Q_EPERM;
      return(&result);
   }

#endif
   
   /*
    * First check authentication.
    */
   if (flags & TYPE_EXTENDED) {
      arguments.ext_args = (ext_setquota_args *)argp;

      id = arguments.ext_args->sqa_id;
      if (unix_cred->aup_uid != 0) {
            result.status = Q_EPERM;
            return(&result);
      }

      qcmd = arguments.ext_args->sqa_qcmd;
      type = arguments.ext_args->sqa_type;
      pathname = arguments.ext_args->sqa_pathp;
      memcpy((caddr_t *)&dq_dqb,(caddr_t *)&arguments.ext_args->sqa_dqblk,sizeof(struct dqblk));
   } else {
      id = arguments.args->sqa_id;
      if (unix_cred->aup_uid != 0) {
         result.status = Q_EPERM;
         return(&result);
      }

      arguments.args = (setquota_args *)argp;
      type = USRQUOTA;
      pathname = arguments.args->sqa_pathp;
   }

   result.status = Q_NOQUOTA;   

   if (stat(pathname, &st) == -1)
      return(&result);
   
   device = st.st_dev;
   result.setquota_rslt_u.sqr_rquota.rq_bsize = BLOCK_SIZE;

   fp = setmntent(_PATH_MOUNTED, "r");
   while ((mnt = getmntent(fp)) != (struct mntent *)NULL) {
      if (stat(mnt->mnt_dir, &st) == -1)
         continue;

      if (st.st_dev != device)
         continue;

      if (hasquota(mnt, type, &qfpathname)) {
         mnt_fsname = get_device_name(mnt->mnt_fsname);
         if (quotactl(qcmd, mnt_fsname, id, (caddr_t)&dq_dqb) != 0)
	    write_quota_to_file(qfpathname, id, &dq_dqb);

       result.status = Q_OK;
       break;
     }
   }
   endmntent(fp);

#else
   result.status = Q_EPERM;
#endif
   return(&result);
}

getquota_rslt *getquotainfo(int flags, caddr_t *argp, struct svc_req *rqstp)
{
   static getquota_rslt result;
   union {
      getquota_args *args;
      ext_getquota_args *ext_args;
   } arguments;
   struct stat st;
   dev_t device;
   FILE *fp;
   struct dqblk dq_dqb;
   struct mntent *mnt;
   char *pathname, *qfpathname;
   int fd, err, id, qcmd, type;
   const char *mnt_fsname;
   
#ifdef HOSTS_ACCESS
   struct hostent *hp;
   struct sockaddr_in *addr;
   
   addr = (svc_getcaller(rqstp->rq_xprt));
   hp = gethostbyaddr((char *)&(addr->sin_addr), sizeof(addr->sin_addr), AF_INET);
   
   if (!good_client(svc_getcaller(rqstp->rq_xprt),hp->h_name)){
      return(FALSE);
   }
#endif

   /*
    * First check authentication.
    */
   if (flags & TYPE_EXTENDED) {
      arguments.ext_args = (ext_getquota_args *)argp;
      id = arguments.ext_args->gqa_id;
      
      if (type == USRQUOTA && unix_cred->aup_uid && unix_cred->aup_uid != id) {
         result.status = Q_EPERM;
         return(&result);
      }
      
      if (type == GRPQUOTA && unix_cred->aup_uid && unix_cred->aup_gid != id &&
         !in_group((gid_t *)unix_cred->aup_gids, unix_cred->aup_len, id)) {
            result.status = Q_EPERM;
            return(&result);
      }

      type = arguments.ext_args->gqa_type;
      pathname = arguments.ext_args->gqa_pathp;
   } else {
      arguments.args = (getquota_args *)argp;
      id = arguments.args->gqa_uid;
      
      if (unix_cred->aup_uid && unix_cred->aup_uid != id) {
         result.status = Q_EPERM;
         return(&result);
      }

      type = USRQUOTA;
      pathname = arguments.args->gqa_pathp;
   }
   
   result.status = Q_NOQUOTA;

   if (stat(pathname, &st) == -1)
      return(&result);
   
   device = st.st_dev;
   result.getquota_rslt_u.gqr_rquota.rq_bsize = BLOCK_SIZE;

   fp = setmntent(_PATH_MOUNTED, "r");
   while ((mnt = getmntent(fp)) != (struct mntent *)NULL) {
      if (stat(mnt->mnt_dir, &st) == -1)
         continue;

      if (st.st_dev != device)
         continue;

      if (hasquota(mnt, type, &qfpathname)) {
         mnt_fsname = get_device_name(mnt->mnt_fsname);
         if ((err = quotactl(QCMD(Q_GETQUOTA, type), mnt_fsname, id,
			    (caddr_t)&dq_dqb)) == -1 && !(flags & ACTIVE))
	    if (read_quota_from_file(qfpathname, id, &dq_dqb) != 0)
	        break;
         
         if (err && (flags & ACTIVE))
            break;

         result.status = Q_OK;
         result.getquota_rslt_u.gqr_rquota.rq_active = (err == 0) ? TRUE : FALSE;

         /*
	  * Make a copy of the info into the last part of the remote quota
	  * struct which is exactly the same.
	  */
         memcpy((caddr_t *)&result.getquota_rslt_u.gqr_rquota.rq_bhardlimit,
	        (caddr_t *)&dq_dqb, sizeof(struct dqblk));
         
         break;
      }
   }
   endmntent(fp);
   
   return(&result);
}

/*
 * Map RPC-entrypoints to local function names.
 */
getquota_rslt *rquotaproc_getquota_1_svc(getquota_args *argp, struct svc_req *rqstp)
{
   return(getquotainfo(0, (caddr_t *)argp, rqstp));
}

getquota_rslt *rquotaproc_getactivequota_1_svc(getquota_args *argp, struct svc_req *rqstp)
{
   return(getquotainfo(ACTIVE, (caddr_t *)argp, rqstp));
}

getquota_rslt *rquotaproc_getquota_2_svc(ext_getquota_args *argp, struct svc_req *rqstp)
{
   return(getquotainfo(TYPE_EXTENDED, (caddr_t *)argp, rqstp));
}

getquota_rslt *rquotaproc_getactivequota_2_svc(ext_getquota_args *argp, struct svc_req *rqstp)
{
   return(getquotainfo(TYPE_EXTENDED | ACTIVE, (caddr_t *)argp, rqstp));
}

setquota_rslt * 
rquotaproc_setquota_1_svc(setquota_args *argp, struct svc_req *rqstp)
{
   return(setquotainfo(0, (caddr_t *)argp, rqstp));
}

setquota_rslt * 
rquotaproc_setactivequota_1_svc(setquota_args *argp, struct svc_req *rqstp)
{
   return(setquotainfo(ACTIVE, (caddr_t *)argp, rqstp));
}

setquota_rslt * 
rquotaproc_setquota_2_svc(ext_setquota_args *argp, struct svc_req *rqstp)
{
   return(setquotainfo(TYPE_EXTENDED, (caddr_t *)argp, rqstp));
}

setquota_rslt * 
rquotaproc_setactivequota_2_svc(ext_setquota_args *argp, struct svc_req *rqstp)
{
   return(setquotainfo(TYPE_EXTENDED | ACTIVE, (caddr_t *)argp, rqstp));
}
