/*
 * Main daemon code: invokes the actual POP handling routines. This is
 * the only file containing code executed as root when running from an
 * inetd clone.
 */

#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>
#include <syslog.h>
#include <errno.h>
#include <grp.h>
#include <pwd.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/types.h>

#include "params.h"
#include "protocol.h"
#include "pop_auth.h"
#include "pop_trans.h"

#if AUTH_SHADOW
#include <shadow.h>
#ifdef __GLIBC__
#include <crypt.h>
#endif
#endif

static struct passwd pop_pw;
static char *mailbox;

int log_error(char *s)
{
	syslog(SYSLOG_PRIORITY, "%s: %m", s);
	return 1;
}

static int set_user(struct passwd *pw)
{
	gid_t groups[2];

	if (!pw->pw_uid) return 1;

	groups[0] = groups[1] = pw->pw_gid;
	if (setgroups(1, groups)) return log_error("setgroups");
	if (setgid(pw->pw_gid)) return log_error("setgid");
	if (setuid(pw->pw_uid)) return log_error("setuid");

	return 0;
}

/*
 * Attempts to read until EOF, and returns the number of bytes read.
 * We don't expect any signals, so even EINTR is considered an error.
 */
static int read_loop(int fd, char *buffer, int count)
{
	int offset, block;

	offset = 0;
	while (count > 0) {
		block = read(fd, &buffer[offset], count);

		if (block < 0) return block;
		if (!block) return offset;

		offset += block;
		count -= block;
	}

	return offset;
}

#if AUTH_SHADOW
/*
 * The /etc/shadow authentication routine. This one is really tricky,
 * in order to make sure we don't have an /etc/shadow fd or sensitive
 * data in our address space after we drop the root privileges. It is
 * arguable whether this was worth the extra code and the performance
 * penalty or not, but such discussions are outside of the scope of a
 * comment like this. ;^)
 */
static struct passwd *do_shadow_auth(char *user, char *pass)
{
	int channel[2];
	struct passwd *pw;
	struct spwd *spw;
	char result;

	if ((pw = getpwnam(user))) mailbox = user;
	endpwent();
	result = 0;

	if (pipe(channel)) {
		log_error("pipe");
		return NULL;
	}

	switch (fork()) {
	case -1:
		log_error("fork");
		return NULL;

	case 0:
		close(channel[0]);
		if (!(spw = getspnam(user)) || !pw || !*spw->sp_pwdp ||
		    *spw->sp_pwdp == '*' || *spw->sp_pwdp == '!')
			crypt(pass, AUTH_DUMMY_SALT);
		else
		if (!strcmp(crypt(pass, spw->sp_pwdp), spw->sp_pwdp))
			result = 1;
		write(channel[1], &result, 1);
		exit(0);
	}

	if (close(channel[1]))
		pw = NULL;
	else {
		if (read(channel[0], &result, 1) != 1) pw = NULL;
		if (result != 1) pw = NULL;
		if (close(channel[0])) pw = NULL;
	}

	wait(NULL);

	return result == 1 ? pw : NULL;
}
#else
/*
 * The /etc/passwd authentication routine.
 */
static struct passwd *do_passwd_auth(char *user, char *pass)
{
	struct passwd *pw, *result;

	if ((pw = getpwnam(user))) mailbox = user;
	endpwent();
	result = NULL;

	if (!pw || !*pw->pw_passwd ||
	    *pw->pw_passwd == '*' || *pw->pw_passwd == '!')
		crypt(pass, AUTH_DUMMY_SALT);
	else
	if (!strcmp(crypt(pass, pw->pw_passwd), pw->pw_passwd))
		result = pw;

	if (pw)
		memset(pw->pw_passwd, 0, strlen(pw->pw_passwd));

	return result;
}
#endif

/*
 * The root-privileged part of the AUTHORIZATION state handling: reads
 * the authentication data obtained over POP from its end of the pipe,
 * attempts authentication, and, if successful, drops privilege to the
 * authenticated user. Returns one of the AUTH_* result codes.
 */
static int do_root_auth(int channel)
{
	static char auth[AUTH_BUFFER_SIZE + 2];
	char *user, *pass;
	struct passwd *pw;

	mailbox = NULL;

/* The POP client could have sent extra commands without waiting for
 * successful authentication. We're passing them into the TRANSACTION
 * state if we ever get there. */
	if (read_loop(channel, (char *)&pop_buffer, sizeof(pop_buffer)) !=
	    sizeof(pop_buffer)) return AUTH_NONE;

/* Now, the authentication data. */
	memset(auth, 0, sizeof(auth));	/* Ensure the NUL termination */
	if (read_loop(channel, auth, AUTH_BUFFER_SIZE) < 0) return AUTH_NONE;

	user = auth;
	pass = &user[strlen(user) + 1];

#if AUTH_SHADOW
	if (!(pw = do_shadow_auth(user, pass))) return AUTH_FAILED;
#else
	if (!(pw = do_passwd_auth(user, pass))) return AUTH_FAILED;
#endif
	if (!*user || !*pass) return AUTH_FAILED;

	memset(pass, 0, strlen(pass));

	if (set_user(pw)) return AUTH_FAILED;

	return AUTH_OK;
}

int do_pop_startup()
{
	struct passwd *pw;

	umask(077);
	signal(SIGPIPE, SIG_IGN);

	openlog(SYSLOG_IDENT, SYSLOG_OPTIONS, SYSLOG_FACILITY);

	errno = 0;
	if (!(pw = getpwnam(POP_USER))) {
		syslog(SYSLOG_PRIORITY, "getpwnam(\"" POP_USER "\"): %s",
			errno ? strerror(errno) : "No such user");
		return 1;
	}
	memcpy(&pop_pw, pw, sizeof(pop_pw));

	return 0;
}

int do_pop_session()
{
	int channel[2];
	int result, status;

	signal(SIGCHLD, SIG_IGN);

	if (pipe(channel)) return log_error("pipe");

	switch (fork()) {
	case -1:
		return log_error("fork");

	case 0:
		if (close(channel[0])) return log_error("close");
		if (set_user(&pop_pw)) return 1;
		return do_pop_auth(channel[1]);
	}

	if (close(channel[1]))
		result = AUTH_NONE;
	else
		result = do_root_auth(channel[0]);

	if (wait(&status) < 0)
		status = 1;
	else
	if (WIFEXITED(status))
		status = WEXITSTATUS(status);
	else
		status = 1;

	if (result == AUTH_OK) {
		if (close(channel[0])) return log_error("close");
		log_pop_auth(result, mailbox);
		return do_pop_trans(mailbox);
	}

	if (set_user(&pop_pw)) return 1;
	log_pop_auth(result, mailbox);

#ifdef AUTH_FAILED_MESSAGE
	if (result == AUTH_FAILED) pop_reply("-ERR %s", AUTH_FAILED_MESSAGE);
#else
	if (result == AUTH_FAILED) pop_reply_error();
#endif

	return status;
}

#if !POP_STANDALONE
int main()
{
	if (do_pop_startup()) return 1;
	return do_pop_session();
}
#endif
