/*
 * lftp - file transfer program
 *
 * Copyright (c) 1999-2002 by Alexander V. Lukyanov (lav@yars.free.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* $Id: ChmodJob.cc,v 1.9 2006/07/24 08:50:18 lav Exp $ */

#include <config.h>
#include <assert.h>
#include "ChmodJob.h"
#include "url.h"

CDECL_BEGIN
#include "filemode.h"
CDECL_END

void ChmodJob::Init()
{
   verbose=V_NONE;
   m=0;
   simple_mode=-1;
}

ChmodJob::ChmodJob(FileAccess *s,ArgV *a) : TreatFileJob(s,a)
{
   Init();
}

ChmodJob::ChmodJob(FileAccess *s,int mode,ArgV *a) : TreatFileJob(s,a)
{
   Init();
   simple_mode=mode;
}

ChmodJob::~ChmodJob()
{
   free(m);
}

void ChmodJob::Recurse()
{
   set_maxdepth(-1);
   Need(FileInfo::TYPE);
}

int ChmodJob::GetMode(const FileInfo *fi) const
{
   if(simple_mode != -1)
      return simple_mode;

   if(fi->defined&fi->MODE)
      return mode_adjust(fi->mode, false, 7777, m, NULL);
   if(!RelativeMode(m))
      return mode_adjust(0, false, 7777, m, NULL);

   return -1;
}

void ChmodJob::CurrentFinished(const char *d,const FileInfo *fi)
{
   const char *fmt;
   if(session->Done() < 0)
   {
      if(quiet)
	 return;
      fmt = _("Failed to change mode of `%s' to %04o (%s).\n");
   }
   else
      fmt = _("Mode of `%s' changed to %04o (%s).\n");

   int mode=GetMode(fi);
   if(mode==-1)
   {
      eprintf(_("Failed to change mode of `%s' because no old mode is available.\n"),fi->name);
      return;
   }
   if(verbose == V_ALL || (verbose == V_CHANGES
			 && (!(fi->defined&fi->mode) || mode != (int)fi->mode)))
   {
      char perms[11];               /* "-rwxrwxrwx" ls-style modes. */

      strmode (mode, perms);
      perms[10] = '\0';             /* `mode_string' does not null terminate. */

      eprintf (fmt, fi->name, (int) mode, perms+1);
   }
}

void ChmodJob::SetMode(mode_change *newm)
{
   m=newm;
   /* request mode info only if we need it */
   if(RelativeMode(m))
      Need(FileInfo::MODE);

   /* one or the other */
   assert(simple_mode == -1);
}

void ChmodJob::SetVerbosity(verbosity v)
{
   verbose=v;

   /* need file mode to show changes */
   if(verbose == V_CHANGES)
      Need(FileInfo::MODE);
}

bool ChmodJob::RelativeMode(const mode_change *m) const
{
   // The mode_change library doesn't exactly bother with type-safety. This
   // `pointer to mode_change' is in reality a pointer to an array of
   // mode_changes. We can increment it by sizeof[mode_change] quite
   // safely.

   size_t i = 0;
   for (; m[i].flag != MODE_DONE; ++i)
   {
      if(m[i].flag || m[i].op != '=')
         return true;
   }

   return false;
}

void ChmodJob::TreatCurrent(const char *d,const FileInfo *fi)
{
   int new_mode=GetMode(fi);
   if(new_mode!=-1)
      session->Chmod(fi->name,new_mode);
}
