/*
 *  pcap.c - interface to libpcap
 *
 *  Copyright (C) 1999-2002 Robert Cheramy <robert@cheramy.ney>
 *  Copyright (C) 1999 Andres Krapf <dae@via.ecp.fr>
 *
 */

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <unistd.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <pcap.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <sys/types.h>

#include "config.h"
#include "data.h"
#include "filter.h"
#include "init.h"
#include "pcap.h"
#include "utils.h"
#include "debug.h"

#define SNAPLEN 200

/* dirty code */
extern int promisc;
extern char *device;
extern struct OptionsType Options;

/* Global variable. This forbids to open 2 pcap descriptors. */
pcap_t  *pcap_global_descriptor = NULL;
struct bpf_program pcap_filter;
int pcap_filter_initialized = 0;

int     offset;

int SigHup = 0;
int SigDump = 0;
extern struct AllLogsType * pAllLogs;

/* pcap error buffer */
extern char errbuf[PCAP_ERRBUF_SIZE];

int iphdr_chksum(u_char *iphead, int size)
{
  unsigned long int sum = 0;
  int i;

  for (i = (size * 4) - 2; i > -1; i-=2) {
    sum += iphead[i] + 256 * iphead[i+1];
  }
  sum = (sum & 0xffff) + (sum >> 16);
  /* mmm I'm not sure the "(sum >> 16)" is useful, but better be sure */
  sum += (sum >> 16) + 1;
  return (sum & 0xffff);
}


/*
 * Find a defaultpcap device
 */
char *pcap_default_device() {
  char errbuf[PCAP_ERRBUF_SIZE];
  char *tdev;

  tdev = pcap_lookupdev(errbuf);
  if (!tdev) { 
    fprintf(stderr, "[pcap] pcap_lookupdev error: %s\n", errbuf);
    exit(1);
  }
  return (tdev);
}


/*
 * Opens a Pcap descriptor
 * promisc = 0 for no promisc mode, 1 for this mode.
*/
int openpcap(char *device, int promisc) {
  char errbuf[PCAP_ERRBUF_SIZE];
  int datalink;
  
  if (NULL != pcap_global_descriptor) {
    printf("[pcap] I can't open two pcap descriptors\n");
    return 0 ;
  }

  if (Options.ReadFromFile == NULL) {
    pcap_global_descriptor = pcap_open_live(device, SNAPLEN, promisc,
                                            0, errbuf);
  } else {
    pcap_global_descriptor = pcap_open_offline(Options.ReadFromFile, errbuf);
  }

  if (NULL == pcap_global_descriptor) {
    fprintf(stderr, "[pcap] error opening pcap: %s\n", errbuf);
    return 0;
  }

  /* compile the IPv4 filter and assign it to the global descriptor */
  if (pcap_compile(pcap_global_descriptor, &pcap_filter,
                   "ip", 1, 0) == 1) {
    fprintf(stderr, "[pcap] error compiling filter \"ip and not ip6\": %s\n",
            pcap_geterr(pcap_global_descriptor));
    return 0;
  }
  pcap_filter_initialized = 1;
  if (pcap_setfilter(pcap_global_descriptor, &pcap_filter) == -1) {
    fprintf(stderr, "[pcap] error assigning filter to descriptor: %s",
            pcap_geterr(pcap_global_descriptor));
    return 0;
  }

  /* set the offset to the beginning of the iphdr */
  if ((datalink = pcap_datalink(pcap_global_descriptor)) < 0) {
    fprintf(stderr, "[pcap] error getting datalink info : %s\n",pcap_geterr(pcap_global_descriptor));
    return 0;
  }

  /*
    these offsets were taken from libpcap source.
    See file gencode.c for more information.
  */
  switch(datalink) {
  case DLT_ARCNET:
    offset = 6;
    break;
  case DLT_EN10MB:
    offset = 14;
    break;
  case DLT_SLIP:
    offset = 16;
    break;
  case DLT_SLIP_BSDOS:
    offset = 24;
    break;
  case DLT_NULL:
  case DLT_LOOP:
    offset = 4;
    break;
  case DLT_PPP:
  case DLT_C_HDLC:		/* BSD/OS Cisco HDLC */
  case DLT_PPP_SERIAL:		/* NetBSD sync/async serial PPP */
    offset = 4;
    break;
#ifdef DLT_PPP_ETHER
  case DLT_PPP_ETHER:
    offset = 8;
    break;
#endif
  case DLT_PPP_BSDOS:
    offset = 24;
    break;
    /*
[RC, 2002/10/20] Commented out FDDI since there is a strange pad complication
on NetBSD, DEC OSF/1 aka Digital Unix aka Tru64 Unix and Ultrix.
If someone yells because she needs FDDI, we'll have a closer look...

  case DLT_FDDI:
     *
     * FDDI doesn't really have a link-level type field.
     * We set "off_linktype" to the offset of the LLC header.
     *
     * To check for Ethernet types, we assume that SSAP = SNAP
     * is being used and pick out the encapsulated Ethernet type.
     * XXX - should we generate code to check for SNAP?
     *
    offset = 21;

#ifdef PCAP_FDDIPAD
    offset += pcap_fddipad;
#endif
    break;
    */
  case DLT_IEEE802:
    offset = 22;
    break;
  case DLT_IEEE802_11:
    offset = 30;
    break;
#ifdef DLT_PRISM_HEADER
  case DLT_PRISM_HEADER:
    offset = 144+30;
    break;
#endif
  case DLT_ATM_RFC1483:
    offset = 8;
    break;
  case DLT_RAW:
    offset = 0;
    break;
  case DLT_ATM_CLIP:	/* Linux ATM defines this */
    offset = 8;
    break;
  case DLT_LINUX_SLL:	/* fake header for Linux cooked socket */
    offset = 16;
    break;
#ifdef DLT_LTALK
  case DLT_LTALK:
    offset = 0;
    break;
#endif
  default:
    fprintf(stderr, "[pcap] Unknown datalink type : %d.", datalink);
    return 0;
  }
  return 1;
}

void closepcap()
{
  if (NULL == pcap_global_descriptor) {
    /* nothing to do */
  } else {
    pcap_close(pcap_global_descriptor);
    pcap_global_descriptor = NULL;
  }
  if (pcap_filter_initialized) {
    pcap_freecode(&pcap_filter);
    pcap_filter_initialized = 0;
  }
}

u_char *getnextippkt() {
  /* these are allocated by pcap (I think they are saved in pcap_global_descriptor allocated by me...) */
  struct pcap_pkthdr *header;
  u_char *packet;
  
  int pcap_result;
  struct AllLogsType *pTempLog;

  if (NULL == pcap_global_descriptor) {
    fprintf(stderr, "[pcap] Can't read non-opened pcap descriptor\n");
  }

  for (;;) {
    pcap_result = pcap_next_ex(pcap_global_descriptor, &header, (const u_char **) &packet);
    DEBUG_MSG("pcap_next_ex returned %i\n", pcap_result);
    switch (pcap_result) {
      case 1:
        /* A packet was received, nothing to do! */
        DEBUG_MSG("packet time/length/size %i %i %i\n", header->ts, header->caplen, header->len);
	break;
      case 0:
        /* pcap_next_ex reached a time out, check if there is someting to dump and if not, continue */
	/* FIXME: ugly code duplication here! */
	/* handle HUP signal */
	if (SigHup) {
          for (pTempLog = pAllLogs; pTempLog; pTempLog = pTempLog->Next) {
	    data_dump(pTempLog);
	    data_clear(pTempLog);
          }

          Clean();
          ReInit();
          SigHup = 0;
        }
	
        /* handle USR1 signal */
        if (SigDump) {
          for (pTempLog = pAllLogs; pTempLog; pTempLog = pTempLog->Next) {
            data_dump(pTempLog);
          }
          SigDump = 0;
        }
	
	continue;
	break;
      
      case -2:
        /* we reached the end of the file ; dump and exit. */
        for (pTempLog = pAllLogs;
             NULL != pTempLog;
             pTempLog = pTempLog->Next) {
          data_dump(pTempLog);
          data_clear(pTempLog);
        }
        Exit(0);
	break;
      
      default:
        /* If pcap_next returns with -1, there is an error on the link, so we close it and try to reopen it
	   Other return codes should not exist, but as I don't know what could hapen, we'll handel them like a -1 return code.
	*/
        DEBUG_MSG("pcap_next_ex error [errormessage : %s]\n", pcap_geterr(pcap_global_descriptor));
      
        closepcap();
        sleep(2);
        while (!openpcap(device, promisc)) {
	  sleep(2);
        }
        continue;
        break;

      
    }
    packet = packet + offset;

#ifdef IP_CHECK
    /* If activated in ./configure, we check if this is a valid IP packet */
    if ( ((struct ip *) packet)->ip_hl < 5 || ((struct ip *) packet)->ip_v != 4 || iphdr_chksum(packet, ((struct ip *) packet)->ip_hl) != 0 ) {
      /* Not an IP packet */
      DEBUG_MSG("getnextippkt : dropped a malformed IP packet\n");
      continue;
    }
#endif

    /* handle HUP signal */
    if (SigHup) {
      for (pTempLog = pAllLogs; pTempLog; pTempLog = pTempLog->Next) {
	data_dump(pTempLog);
	data_clear(pTempLog);
      }

      Clean();
      ReInit();
      SigHup = 0;
    }
    /* handle USR1 signal */

    if (SigDump) {
      for (pTempLog = pAllLogs; pTempLog; pTempLog = pTempLog->Next) {
	data_dump(pTempLog);
      }
      SigDump = 0;
    }
    break;
  }
  return (packet);
}

