/*
 *  init.c - ipfm initialisation functions
 *
 *  Copyright (C) 1999 Andres Krapf <dae@via.ecp.fr>
 *
 */

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#ifdef __OS_LINUX__
#include <getopt.h>
#endif /* __OS_LINUX__ */
#include <time.h>


#include "utils.h"
#include "pcap.h"
#include "filter.h"
#include "data.h"
#include "init.h"

extern char * device;
extern int SigHup;
extern struct AllLogsType * pAllLogs;
int run_in_background = 1;

struct OptionsType Options;

#ifndef __OS_LINUX__
void print_help() {
  printf("ipfm version %s\n\n", VERSION);
  printf("\t-n:\tdo not run in background\n");
  printf("\t-c <file>:\tspecify an alternate configuration file\n");
  printf("\t-h:\tdisplay this help and exit\n");
  printf("\nPlease report bugs to ipfm@via.ecp.fr\n");
}
#else
void print_help() {
  printf("ipfm version %s\n\n", VERSION);
  printf("\t-n, --nofork:\tdo not run in background\n");
  printf("\t-c, --config <file>:\tspecify an alternate configuration file\n");
  printf("\t-h, --help:\tdisplay this help and exit\n");
  printf("\nPlease report bugs to ipfm@via.ecp.fr\n");
}
#endif /* __OS_LINUX__ */

void ParseCmdLine(int argc, char * argv[]) {
  int ch;
#ifdef __OS_LINUX__
  int option_index = 0;
#endif /* __OS_LINUX__ */
  extern char *optarg;

#ifdef __OS_LINUX__
  static struct option long_options[] = {
    { "config", 1, NULL, 0 },
    { "help", 0, NULL, 0 },
    { "nofork", 0, &run_in_background, 0 },
    { NULL, 0, NULL, 0 }
  };
#endif /* __OS_LINUX__ */
	
  Options.ConfigFile = NULL;

#ifndef __OS_LINUX__
  while (EOF != (ch = getopt(argc, argv, "hnc:"))) {
#else
  while (EOF != (ch = getopt_long(argc, argv, "hnc:", long_options,
                                 &option_index))) {
#endif /* __OS_LINUX__ */
    switch(ch) {	  
#ifdef __OS_LINUX__
    case 0:
      switch (option_index) {
        case 0:
          Options.ConfigFile = strdup(optarg);
	  break;
	case 1:
          print_help();
	  exit(0);
	  break;
    }
#endif /* __OS_LINUX__ */
    case 'c':
      Options.ConfigFile = strdup(optarg);
      break;
    case 'n':
      run_in_background = 0;
      break;
    case 'h':
    case '?':
      print_help();
      exit(0);
    }
  }
}

void InitOnce()
{
/*
  struct sigaction SigChld;
  sigset_t SigChldMask;

  sigemptyset(&SigChldMask);
  SigChld.sa_handler = SIG_DFL;
  SigChld.sa_mask = SigChldMask;
  SigChld.sa_flags = SA_NOCLDWAIT;
*/
  signal(SIGHUP,  SigHupHandler);
  signal(SIGTERM, SigTermHandler);
  signal(SIGINT,  SigIntHandler);
  signal(SIGCHLD, SigChldHandler);

  if (NULL == Options.ConfigFile)
  {
    Options.ConfigFile = strdup(DEFAULT_OPTIONS_CONFIGFILE);
  }
}

void ReInit()
{
  struct AllLogsType * pNewLog;

  pNewLog = (struct AllLogsType *) malloc (sizeof(struct AllLogsType));
  pNewLog->Next = pAllLogs;
  pNewLog->Filter = NULL;
  pNewLog->Data = NULL;
  pNewLog->TimeInterval = DEFAULT_OPTIONS_TIMEINTERVAL;
  pNewLog->NextTime = time(NULL) + pNewLog->TimeInterval;
  pNewLog->LogFile = strdup(DEFAULT_OPTIONS_LOGFILE);
  pNewLog->Sort = DEFAULT_OPTIONS_SORT;
  pNewLog->SortFunc = NULL;
  pNewLog->ReverseLookup = DEFAULT_OPTIONS_REVERSELOOKUP;

  pAllLogs = pNewLog;

  configure();

  TestLogs();

  if (NULL == device) {
    device = strdup(pcap_default_device());
  }

  if(!openpcap(device, 1))
  {
    printf("[ipfm] Unable to open pcap descriptor on device %s\n", device);
    Exit(-1);
  }
}

void Init()
{
  InitOnce();
  ReInit();
}

void Clean()
{
  struct AllLogsType * pTempLog;
  struct s_filter * pTempRule;
  struct s_data * pTempData;

/* clean the rules (deallocate the chained list) */
  
  for (pTempLog = pAllLogs; NULL != pTempLog; pTempLog = pTempLog->Next)
  {
    while (NULL != pTempLog->Filter)
    {
      pTempRule = pTempLog->Filter;;
      pTempLog->Filter = pTempLog->Filter->next;
      if (pTempRule)
	free(pTempRule);
      else
	fprintf(stderr, "[init.c] BUG!\n");
    }
  
/* clean the stats (the same) */
  
    while (NULL != pTempLog->Data)
    {
      pTempData = pTempLog->Data;
      pTempLog->Data = pTempLog->Data->next;
      if (pTempData)
	free(pTempData);
      else
	fprintf(stderr, "[init.c] BUG 2!\n");
    }
  }

/* close the pcap descriptor */
   closepcap();

  if (device)
  {
    free(device);
    device = NULL;
  }
}

void Exit(int errcode)
{
  Clean();

  if (Options.ConfigFile)
    free(Options.ConfigFile);

  exit(errcode);
}

/*
 * we flush the tables. something should be done to avoid overwriting files ?
 * (for example a flush a few millisecs after a flush ?)
 */

void SigHupHandler(int a)
{
  SigHup = 1;

#if DEBUG >= 2
  printf("IPFM : Received SigHUP\n");
#endif
  
    /* For linux libc5 backward compatibility */
    /* This is dirty and should check sysV or BSD behaviour */
    signal(SIGHUP, SigHupHandler);
}

/* 
 * use SIGTERM to flush the data so that the user 
 * can do it any time he wants
 */

void SigTermHandler(int a)
{
  struct AllLogsType * pTempLog;

#if DEBUG >= 2
  printf("IPFM : Received SigTERM\n");
#endif
  for (pTempLog = pAllLogs; NULL != pTempLog; pTempLog = pTempLog->Next)
    data_flush(pTempLog);
  Exit(0);
}

void SigIntHandler(int a)
{
  struct AllLogsType * pTempLog;

#if DEBUG >= 2
  printf("IPFM : Received SigINT\n");
#endif
	  
  for (pTempLog = pAllLogs; NULL != pTempLog; pTempLog = pTempLog->Next)
    data_flush(pTempLog);
  Exit(0);
}

/*
 * we don't want any zombis left, and we don't do anything special when 
 * children die
 */

void SigChldHandler(int a)
{
  pid_t Pid = 1;
  int Status;
 
#if DEBUG >=3
  printf("Received SigChild\n");
#endif

  while (Pid > 0)
    Pid = waitpid(-1, &Status, WNOHANG);

  /* For linux libc5 backward compatibility */
  /* This is dirty and should check sysV or BSD behaviour */
  signal(SIGCHLD, SigChldHandler);
}

void TestLogs()
{
  struct AllLogsType * pTempLog;
  struct AllLogsType * pPrevLog;
  struct AllLogsType * pNextLog;

  pNextLog = pAllLogs->Next;
  pPrevLog = NULL;
  for (pTempLog = pAllLogs; NULL != pTempLog; pTempLog = pNextLog)
  {
    pNextLog = pTempLog->Next;
    if (NULL == pTempLog->Filter)
    {
      if (pTempLog->LogFile)
	free (pTempLog->LogFile);

      if (NULL == pPrevLog)
	pAllLogs = pTempLog->Next;
      else
	pPrevLog->Next = pTempLog->Next;

      free (pTempLog);
    }
    else
      pPrevLog = pTempLog;
  }
}
