#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <signal.h>
#include <termios.h>
#include <sys/time.h>
#include <fcntl.h>
#include <errno.h>

/* Copy data across a TCP connection. */
static int _tty_copy(int crnl, int infd, int outfd)
{
  int i, cnt, nfound;
  fd_set rfds, wfds;
  int input_blocked = 0,  output_blocked = 0;
  int rcv_bytes = 0,      send_bytes = 0;
  char rcv_buffer[8192],  send_buffer[8192];
  char *rcv_bufp = NULL, *send_bufp = NULL;
  char *s;

  /* Copy between socket and tty */
  while (1) {
	do {
		FD_ZERO(&rfds);
		FD_ZERO(&wfds);
		if (input_blocked) 
			FD_SET(infd, &wfds);
		else
			FD_SET(outfd, &rfds);
		if (output_blocked) 
			FD_SET(outfd, &wfds);
		else
			FD_SET(infd, &rfds);
		nfound = select(outfd + 1, &rfds, &wfds, (fd_set *)NULL, (struct timeval *)NULL);
	} while (nfound == -1 && errno == EINTR);

	/* Data is ready on the inputdevice. */
	if (FD_ISSET(infd, &rfds)) {
		send_bytes = read(infd, send_buffer, sizeof(send_buffer));
		send_bufp = send_buffer;
		if (crnl > 0) {
			for(i = 0, s = send_bufp; i < send_bytes; i++,s++)
				if (*s == '\r') *s = '\n';
		}
	}

	/* New data has become available or there is new room in the output buffer. */
	if (FD_ISSET(infd, &rfds) || (output_blocked && FD_ISSET(outfd, &wfds))) {
		output_blocked = 0;
		if (send_bytes <= 0) return(0);
		cnt = 0;
		while (!output_blocked && send_bytes > 0) {
			cnt = write(outfd, send_bufp, send_bytes);
			if (cnt < 0) switch (errno) {
				case EWOULDBLOCK:
					output_blocked = 1;
					break;
				case EINTR:
					continue;
				default:
					return(-1);
			}
			send_bytes -= cnt;
			send_bufp += cnt;
		}
	}

	/* Data is ready on the outputdevice. */
	if (FD_ISSET(outfd, &rfds)) {
		rcv_bytes = read(outfd, rcv_buffer, sizeof(rcv_buffer));
		rcv_bufp = rcv_buffer;
	}

	/* New data has become available or there is new room in the output buffer. */
	if (FD_ISSET(outfd, &rfds) || (input_blocked && FD_ISSET(infd, &wfds))) {
		input_blocked = 0;
		if (rcv_bytes <= 0) return(0);
		cnt = 0;
		while (!input_blocked && rcv_bytes > 0) {
			cnt = write(infd, rcv_bufp, rcv_bytes);
			if (cnt < 0) switch (errno) {
				case EWOULDBLOCK:
					input_blocked = 1;
					break;
				case EINTR:
					continue;
				default:
					return(-1);
			}
			rcv_bytes -= cnt;
			rcv_bufp += cnt;
		}
	}
   }
}

/* Copy data across a TCP connection. */
int tty_copy(int crnl, int infd, int outfd)
{
  int flags, iflags, oflags;
  void (*osig)(int);
  int ret;

  /* We don't need SIGIO */
  osig = signal(SIGIO, SIG_IGN);

  /* Set file descriptors in non-blocking mode. */
  iflags = fcntl(infd, F_GETFL);
  oflags = fcntl(outfd, F_GETFL);
  flags = iflags | O_NDELAY;
  fcntl(infd, F_SETFL, flags);
  flags = oflags | O_NDELAY;
  fcntl(outfd, F_SETFL, flags);

  ret = _tty_copy(crnl, infd, outfd);

  fcntl(infd, F_SETFL, iflags);
  fcntl(outfd, F_SETFL, oflags);
  signal(SIGIO, osig);

  return ret;
}
