/*
 * conf.c	Read the configuration file.
 *
 * Version:	@(#)conf.c  1.25  08-Nov-1997  MvS.
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>
#include <ctype.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <stddef.h>

#define EXTERN
#include "server.h"

#define C_INT	1
#define C_STR	2
#define C_HOST	3
#define C_LIST	4
#define C_IPNO	5
#define C_IPDY	6
#define C_CHAT	7

/*
 *	For option lists.
 */
struct lst {
  char *name;
  int value;
};

/*
 *	Define type of config variables.
 */
struct conf {
  char *name;
  int type;
  struct lst *opts;
  int offs;
};

/*
 *	Types of authentication.
 */
static struct lst radlst[] = {
  { "none",    0  },
  { "radius",  1  },
  { NULL,      0  },
};

/*
 *	Types of flow control.
 */
static struct lst flowlst[] = {
  { "none",	FLOW_NONE },
  { "hard",	FLOW_HARD },
  { "soft",	FLOW_SOFT },
  { NULL,	0	  },
};

/*
 *	Protocols.
 */
static struct lst prlst[] = {
  { "login",	P_LOCAL    },
  { "rlogin",	P_RLOGIN   },
  { "slip",	P_SLIP     },
  { "cslip",	P_CSLIP    },
  { "ppp",	P_PPP      },
  { "tcpclear",	P_TCPCLEAR },
  { "tcplogin",	P_TCPLOGIN },
  { "console",	P_CONSOLE  },
  { NULL,	0 },
};

/*
 *	Define structure for the main configuration.
 */
static struct conf main_cfg[] = {
  { "hostname",		C_STR,	NULL,	offsetof(struct main_cfg, hostname)	},
  { "ipno",		C_IPNO,	NULL,	offsetof(struct main_cfg, ipno)		},
  { "lockdir",		C_STR,	NULL,	offsetof(struct main_cfg, lockdir)	},
  { "rlogin",		C_STR,	NULL,	offsetof(struct main_cfg, rlogin)	},
  { "telnet",		C_STR,	NULL,	offsetof(struct main_cfg, telnet)	},
  { "pppd",		C_STR,	NULL,	offsetof(struct main_cfg, pppd)		},
  { "locallogins",	C_STR,	NULL,	offsetof(struct main_cfg, locallogins)	},
  { "syslog",		C_HOST,	NULL,	offsetof(struct main_cfg, syslog)	},
  { "facility",		C_INT,	NULL,	offsetof(struct main_cfg, facility)	},
  { NULL,									},
};

/*
 *	Define structure for the per-line configuration.
 */
static struct conf line_cfg[] = {
  { "debug",		C_INT,	NULL,	offsetof(struct line_cfg, debug)		},
  { "sysutmp",		C_INT,	NULL,	offsetof(struct line_cfg, sysutmp)		},
  { "syswtmp",		C_INT,	NULL,	offsetof(struct line_cfg, syswtmp)		},
  { "emumodem",		C_INT,	NULL,	offsetof(struct line_cfg, emumodem)		},
  { "porttype",		C_INT,	NULL,	offsetof(struct line_cfg, porttype)		},
  { "authtype",		C_LIST,	radlst,	offsetof(struct line_cfg, authtype)	},
  { "authhost1",	C_HOST,	NULL,	offsetof(struct line_cfg, authhost1)	},
  { "authhost2",	C_HOST,	NULL,	offsetof(struct line_cfg, authhost2)	},
  { "accthost1",	C_HOST,	NULL,	offsetof(struct line_cfg, accthost1)	},
  { "accthost2",	C_HOST,	NULL,	offsetof(struct line_cfg, accthost2)	},
#ifdef UUCPHACK
  { "uauthhost1",	C_HOST,	NULL,	offsetof(struct line_cfg, uauthhost1)	},
  { "uauthhost2",	C_HOST,	NULL,	offsetof(struct line_cfg, uauthhost2)	},
  { "uaccthost1",	C_HOST,	NULL,	offsetof(struct line_cfg, uaccthost1)	},
  { "uaccthost2",	C_HOST,	NULL,	offsetof(struct line_cfg, uaccthost2)	},
#endif
  { "protocol",		C_LIST,	prlst,	offsetof(struct line_cfg, protocol)	},
  { "secret",		C_STR,	NULL,	offsetof(struct line_cfg, secret)	},
  { "host",		C_HOST,	NULL,	offsetof(struct line_cfg, host)		},
  { "ipno",		C_IPDY,	NULL,	offsetof(struct line_cfg, ipno)		},
  { "netmask",		C_IPNO,	NULL,	offsetof(struct line_cfg, netmask)	},
  { "mtu",		C_INT,	NULL,	offsetof(struct line_cfg, mtu)		},
  { "mru",		C_INT,	NULL,	offsetof(struct line_cfg, mru)		},
  { "autoppp",		C_STR,	NULL,	offsetof(struct line_cfg, autoppp)	},
  { "pppopt",		C_STR,	NULL,	offsetof(struct line_cfg, pppopt)	},
  { "tty",		C_STR,	NULL,	offsetof(struct line_cfg, tty)		},
  { "issue",		C_STR,	NULL,	offsetof(struct line_cfg, issue)	},
  { "prompt",		C_STR,	NULL,	offsetof(struct line_cfg, prompt)	},
  { "term",		C_STR,	NULL,	offsetof(struct line_cfg, term)		},

  { "speed",		C_INT,	NULL,	offsetof(struct line_cfg, speed)	},
  { "dcd",		C_INT,	NULL,	offsetof(struct line_cfg, dcd)		},
  { "flow",		C_LIST, flowlst,offsetof(struct line_cfg, flow)		},
  { "aa",		C_INT,	NULL,	offsetof(struct line_cfg, aa)		},
  { "checktime",	C_INT,	NULL,	offsetof(struct line_cfg, checktime)	},
  { "checkchat",	C_CHAT,	NULL,	offsetof(struct line_cfg, checkchat)	},
  { "initchat",		C_CHAT,	NULL,	offsetof(struct line_cfg, initchat)	},
  { "waitfor",		C_CHAT,	NULL,	offsetof(struct line_cfg, waitfor)	},
  { "answer",		C_CHAT,	NULL,	offsetof(struct line_cfg, answer)	},
  { NULL,									},
};

/*
 *	Set an integer.
 */
static int setint(char *ptr, char *val, int offs)
{
  int n, i;

  if (sscanf(val, "%d", &n) != 1)
	return -1;
  if (ptr == (char *)&allconf) {
	for(i = 0; i < MAXLINES; i++)
		*(int *)((char *)&lineconf[i] + offs) = n;
  }
  *(int *)(ptr + offs) = n;

  return 0;
}

/*
 *	Set a string.
 */
static int setstr(char *ptr, char *val, int offs, int doun)
{
  int i;
  char **s;

  /*
   *	Specialcase "" as the empty string.
   */
  if (doun)
	unescape(val);
  else if (strcmp(val, "\"\"") == 0) val = "";

  if (ptr == (char *)&allconf) {
	for(i = 0; i < MAXLINES; i++) {
		s =(char **)((char *)&lineconf[i] + offs);
		if (*s) free(*s);
		*s = strdup(val);
	}
  }
  s =(char **)(ptr + offs);
  if (*s) free(*s);
  *s = strdup(val);

  return 0;
}

/*
 *	Set a hostname. Return ipno.
 */
static int sethost(char *ptr, char *val, int offs)
{
  int i;
  unsigned int n;
  struct hostent *h;

  if (val[0] == 0)
	n = 0;
  else if ((int)(n = inet_addr(val)) == -1) {
	if ((h = gethostbyname(val)) == NULL)
		return -1;
	n = *(unsigned int *)h->h_addr_list[0];
  }

  if (ptr == (char *)&allconf) {
	for(i = 0; i < MAXLINES; i++)
		*(unsigned int *)((char *)&lineconf[i] + offs) = n;
  }
  *(unsigned int *)(ptr + offs) = n;

  return 0;
}

/*
 *	Set an IP address.
 */
static int setipno(char *ptr, char *val, int offs)
{
  int i;
  unsigned int n;

  if ((int)(n = inet_addr(val)) == -1 && strcmp(val, "255.255.255.255") != 0)
		return -1;

  if (ptr == (char *)&allconf) {
	for(i = 0; i < MAXLINES; i++)
		*(unsigned int *)((char *)&lineconf[i] + offs) = n;
  }
  *(unsigned int *)(ptr + offs) = n;

  return 0;
}

/*
 *	Set a dynamic IP address.
 */
static int setipdy(char *ptr, char *val, int offs, int port)
{
  int i;
  unsigned int n, m;
  int dy = 0;

  i = strlen(val);
  if (i > 0 && val[i - 1] == '+') {
	dy = 1;
	val[i - 1] = 0;
  }

  if ((int)(n = inet_addr(val)) == -1 && strcmp(val, "255.255.255.255") != 0)
		return -1;

  if (ptr == (char *)&allconf) {
	for(i = 0; i < MAXLINES; i++) {
		m = n;
		if (dy) m = htonl(ntohl(m) + i);
		*(unsigned int *)((char *)&lineconf[i] + offs) = m;
	}
  }
  if (port >= 0 && dy) n = htonl(ntohl(n) + i);
  *(unsigned int *)(ptr + offs) = n;

  return 0;
}

/*
 *	Set an integer, from a list of string values.
 */
static int setlist(char *ptr, char *val, struct lst *lst, int offs)
{
  int n, i;

  for(i = 0; lst[i].name; i++) {
	if (strcmp(val, lst[i].name) == 0) {
		n = lst[i].value;
		break;
	}
  }
  if (lst[i].name == NULL) return -1;

  if (ptr == (char *)&allconf) {
	for(i = 0; i < MAXLINES; i++)
		*(int *)((char *)&lineconf[i] + offs) = n;
  }
  *(int *)(ptr + offs) = n;

  return 0;
}

/*
 *	Parse one line from the configuration file.
 */
int parseline(char *line)
{
  char *dup = strdup(line);
  char *s = dup;
  char *w, *a, *c;
  char *ptr;
  struct conf *cfg, *x;
  int n = -1;

  /*
   *	Remove the trailing newline.
   */
  for(w = s; *w; w++)
	;
  while(--w >= s && *w == '\n')
	*w = 0;

  /*
   *	Get word (w), command (c) and argument (a).
   */
  while(isspace(*s)) s++;
  w = s;
  while(*s && !isspace(*s)) s++;
  if (*w == 0) {
parseline_error:
	free(dup);
	return -1;
  }
  if (*s) *s++ = 0;
  while(isspace(*s)) s++;
  a = s;
  if ((c = strchr(w, '.')) == NULL || c[1] == 0)
	goto parseline_error;
  *c++ = 0;

  /*
   *	See what class of command this is.
   */
  if (strcmp(w, "conf") == 0) {
	ptr = (char *)&mainconf;
	cfg = main_cfg;
  } else if (strcmp(w, "all") == 0) {
	ptr = (char *)&allconf;
	cfg = line_cfg;
  } else if (w[0] == 's' && ((n = atoi(w + 1)) || w[1] == '0') && n < MAXLINES) {
	ptr = (char *)&lineconf[n];
	cfg = line_cfg;
  } else
	goto parseline_error;

  /*
   *	Parse the line and fill in the right structure(s).
   */
  for(x = cfg; x->name != NULL; x++) {
	if (strcmp(c, x->name) != 0) continue;
	switch(x->type) {
		case C_INT:
			n = setint((char *)ptr, a, x->offs);
			n = 0;
			break;
		case C_STR:
			n = setstr((char *)ptr, a, x->offs, 1);
			break;
		case C_CHAT:
			n = setstr((char *)ptr, a, x->offs, 0);
			break;
		case C_HOST:
			n = sethost((char *)ptr, a, x->offs);
			break;
		case C_LIST:
			n = setlist((char *)ptr, a, x->opts, x->offs);
			break;
		case C_IPNO:
			n = setipno((char *)ptr, a, x->offs);
			break;
		case C_IPDY:
			n = setipdy((char *)ptr, a, x->offs, n);
			break;
		default:
			n = -1;
			break;
	}
	break;
  }
  if (x == NULL || n < 0) goto parseline_error;

  return 0;
}

/*
 *	Read the configuration file.
 */
int readcfg(void)
{
  FILE *fp;
  char buf[2048];
  int lineno = 0;
  char *s, *p;

  if ((fp = fopen(CONFFILE, "r")) == NULL) {
	nsyslog(LOG_ERR, "%s: %m", CONFFILE);
	return -1;
  }
  p = buf;
  /*
   *	FIXME: this style of continuation lines
   *	does not work on the last line of a file.
   */
  while(fgets(p, sizeof(buf) - (p - buf), fp)) {
	lineno++;
	if (p[0] == '#') continue;
	s = p + strlen(p);
	if (s > p && *--s == '\n' && *--s == '\\') {
		p = s;
		continue;
	}
	if (buf[0] == '\n' || buf[0] == 0) {
		p = buf;
		continue;
	}
	if (parseline(buf) < 0) {
		nsyslog(LOG_WARNING, "%s[%d]: syntax error\n", CONFFILE, lineno);
	}
	p = buf;
  }
  return 0;
}

#if 0 /* XXX - notyet (writecfg) */
/*
 *	Write the configuration file.
 */
int writecfg(void)
{
  FILE *in, out;
  char buf[128];
  char outfile[128];
  int lineno = 0;

  sprintf(outfile, "%s.tmp", CONFFILE);
  if ((in = fopen(CONFFILE, "r")) == NULL) {
	nsyslog(LOG_ERR, "%s: %m", CONFFILE);
	return -1;
  }
  if ((out = fopen(outfile, "w")) == NULL) {
	nsyslog(LOG_ERR, "%s: %m", outfile);
	return -1;
  }
  while(fgets(buf, sizeof(buf), in)) {
	/*
	 *	See which configuration this is.
	 */
#endif

/*
 *	Initialize the configuration stuff with defaults.
 */
void initcfg(void)
{
  struct hostent *h;
  char buf[128];

  (void)gethostname(buf, sizeof(buf));
  mainconf.hostname = strdup(buf);
  if (( h = gethostbyname(buf)) != NULL)
	mainconf.ipno = *(unsigned int *)(h->h_addr);
  allconf.prompt = strdup("login: ");
}

