/*
 * radclient	A client-side implementation of the RADIUS protocol.
 *
 * Version:	@(#)radclient.c  1.43  08-Nov-1997  miquels@cistron.nl
 *
 */
#include <sys/types.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/file.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <time.h>

#define NO_FIX_MALLOC

#include "radius.h"
#include "server.h"

extern void radius_md5_calc(unsigned char *, unsigned char *, unsigned int);

/*
 *	This is an "attribute".
 */
typedef struct _attr_ {
	unsigned char type;
	unsigned char len;
	union {
		char str[254];
		unsigned int num;
	} val;
	unsigned char pwtype;
	struct _attr_ *next;
} ATTR;

/*
 *	This is a complete request or reply.
 */
typedef struct {
	unsigned char code;
	unsigned char id;
	unsigned char vector[AUTH_VECTOR_LEN];
	time_t timestamp;
	time_t lastsent;
	ATTR *list;
	unsigned char secret[100];
} RADPKT;


/*
 *	Create a new id.
 */
int rad_id()
{
	int fd, i, n;
	char id[8];

	if ((fd = open(RAD_ID_FILE, O_RDWR|O_CREAT, 0644)) < 0) {
		nsyslog(LOG_ERR, "%s: %m", RAD_ID_FILE);
		return -1;
	}
	for(i = 0; i < 10; i++) {
		if (i > 0) usleep(200000);
		if (flock(fd, LOCK_EX) == 0)
		        break;
	}
	if (i == 10) {
		nsyslog(LOG_ERR, "rad_id: failed to lock %s\n", RAD_ID_FILE);
		return -1;
	}

	n = read(fd, id, 7);
	if (n < 0) n = 0;
	id[n] = 0;
	i = (atoi(id) + 1) & 0xff;
	sprintf(id, "%d\n", i);
	(void)ftruncate(fd, 0L);
	(void)lseek(fd, 0L, SEEK_SET);
	(void)write(fd, id, strlen(id));

	flock(fd, LOCK_UN);
	close(fd);

	return i;
}

/*
 *	Create a new session id.
 */
unsigned int rad_sessionid(char *s)
{
	int fd, n;
	unsigned int i;
	char id[32];

	if ((fd = open(RAD_SESSIONID_FILE, O_RDWR|O_CREAT, 0644)) < 0) {
		nsyslog(LOG_ERR, "%s: %m", RAD_SESSIONID_FILE);
		return -1;
	}
	for(i = 0; i < 10; i++) {
		if (i > 0) usleep(200000);
		if (flock(fd, LOCK_EX) == 0)
		        break;
	}
	if (i == 10) {
		nsyslog(LOG_ERR, "rad_sessionid: failed to lock %s\n",
			RAD_SESSIONID_FILE);
		return -1;
	}

	n = read(fd, id, 31);
	if (n < 0) n = 0;
	id[n] = 0;
	i = 0;
	sscanf(id, "%x", &i);
	if (s == NULL)
		i += 0x01000000;
	else
		i++;
	i &= 0xFFFFFFFF;
	if (i == 0) i++;
	sprintf(id, "%08x\n", i);
	(void)ftruncate(fd, 0L);
	(void)lseek(fd, 0L, SEEK_SET);
	(void)write(fd, id, strlen(id));

	flock(fd, LOCK_UN);
	close(fd);

	if (s) {
		strcpy(s, id);
		s[8] = 0;
	}
	return 0;
}

/*
 *	Create a random vector. We could use /dev/urandom
 *	for this under Linux for increased security, but..
 */
void rad_vector(char *vector)
{
	int	randno;
	int	i;

	srand(time(0) + getpid());
	for(i = 0;i < AUTH_VECTOR_LEN;) {
		randno = rand();
		memcpy(vector, &randno, sizeof(int));
		vector += sizeof(int);
		i += sizeof(int);
	}
}


/*
 *	Build an attribute.
 *	Integers are passed through without conversion...
 */
ATTR *rad_addattr(ATTR *list, int type, int val_len, unsigned int num, char *str)
{
	ATTR *a, *r;

	/*
	 * Ask for space.
	 */
	if ((r = (ATTR *)malloc(sizeof(ATTR))) == NULL) {
		nsyslog(LOG_ERR, "rad_addattr: Out of memory\n");
		return r;
	}
	memset(r, 0, sizeof(ATTR));

	/*
	 * Find end of the list (if any) and add us to it.
	 */
	for(a = list; a && a->next; a = a->next)
		;
	if (a) a->next = r;

	/*
	 * Fill out fields.
	 */
	if (str) {
		r->pwtype = PW_TYPE_STRING;
		r->len = val_len >= 0 ? val_len : strlen(str) + 1;
		memcpy(r->val.str, str, r->len);
	} else {
		r->pwtype = PW_TYPE_INTEGER;
		r->len = val_len > 0 ? val_len : 6;
		r->val.num = num;
	}
	r->len += 2;
	r->type = type;

	return r;
}

/*
 *	Build a request.
 */
int rad_buildreq(char *buf, int bufsz, RADPKT *pkt)
{
	char *p = buf;
	ATTR *a;
	int attr_len = 0;
	int tot_len;
	int tm;
	time_t now;

	/*
	 *	Find out how much space we need for the value pairs.
	 */
	for(a = pkt->list; a; a = a->next)
		attr_len += a->len;
	tot_len = attr_len + AUTH_HDR_LEN;
	if (pkt->timestamp) tot_len += 8;
	if (tot_len > bufsz) {
		nsyslog(LOG_ERR, "rad_buildreq: buffer overflow\n");
		return -1;
	}
	memset(p, 0, tot_len);

	/*
	 *	See if we're going to use a new timestamp -
	 *	if so, we need a new ID.
	 */
	if (pkt->timestamp) {
		time(&now);
		if (pkt->lastsent != now) {
			pkt->id = rad_id();
			pkt->lastsent = now;
		}
	}

	/*
	 *	Fill in header fields.
	 */
	*p++ = pkt->code;
	*p++ = pkt->id;
	*(unsigned short *)p = htons(tot_len);
	p += 2;
	memcpy(p, pkt->vector, AUTH_VECTOR_LEN);
	p += AUTH_VECTOR_LEN;

	/*
	 *	Build complete request.
	 */
	for(a = pkt->list; a; a = a->next) {
		p[0] = a->type;
		p[1] = a->len;
		if (a->pwtype == PW_TYPE_STRING)
			memcpy(p+2, a->val.str, a->len - 2);
		else
			memcpy(p+2, &(a->val.num), 4);
		p += a->len;
	}

	/*
	 *	And add a time stamp.
	 */
	if (pkt->timestamp) {
		*p++ = PW_ACCT_DELAY_TIME;
		*p++ = 8;
		tm = htonl(now - pkt->timestamp);
		memcpy(p, &tm, 4);
	}

	return tot_len;
}

/*
 *	Free a list of attributes.
 */
void rad_attrfree(ATTR *l)
{
	ATTR *next;

	while (l != NULL) {
		next = l->next;
		free(l);
		l = next;
	}
}

/*
 *	Build an authentication request.
 */
RADPKT *rad_buildauth(char *secret_key, char *login,
	char *passwd, int port, int ppp, int porttype)
{
	RADPKT *pkt;
	ATTR *list = NULL;
	int secret_len;
	char  md5buf[256];
	char passtmp[256];
	int i;

	if ((pkt = malloc(sizeof(RADPKT))) == NULL)
		return NULL;
	memset(pkt, 0, sizeof(RADPKT));

	/*
	 *	Build the header.
	 */
	pkt->code = PW_AUTHENTICATION_REQUEST;
	pkt->id   = rad_id();
	rad_vector(pkt->vector);

	/*
	 *	Add the login name.
	 */
	if ((pkt->list = rad_addattr(NULL, PW_USER_NAME, -1, 0, login)) == NULL)
	{
		free(pkt);
		return NULL;
	}

	/*
	 *	And the password - this one is hard.
	 */
	memset(passtmp, 0, sizeof(passtmp));
	secret_len = strlen(secret_key);
	strcpy(md5buf, secret_key);
	memcpy(md5buf + secret_len, pkt->vector, AUTH_VECTOR_LEN);
	radius_md5_calc(passtmp, md5buf, secret_len + AUTH_VECTOR_LEN);
	for(i = 0; passwd[i] && i < AUTH_PASS_LEN; i++)
		passtmp[i] ^= passwd[i];

#if 0 /* XXX - DEBUG */
	nsyslog(LOG_DEBUG, "passwd/secret: %s/%s", passwd, secret_key);
#endif
	if (rad_addattr(pkt->list, PW_PASSWORD, AUTH_PASS_LEN, 0, passtmp) == NULL) {
		rad_attrfree(pkt->list);
		free(pkt);
		return NULL;
	}

	/*
	 *	Now we finish off with the client id (our ip address),
	 *	and the port id (tty number).
	 */
	if (rad_addattr(pkt->list, PW_NAS_IP_ADDRESS, 0, mainconf.ipno, NULL) == NULL) {
		rad_attrfree(list);
		free(pkt);
		return NULL;
	}
	if (rad_addattr(pkt->list, PW_NAS_PORT_ID, 0, htonl(port), NULL) == NULL) {
		rad_attrfree(list);
		free(pkt);
		return NULL;
	}
	if (rad_addattr(pkt->list, PW_NAS_PORT_TYPE, 0, htonl(porttype), NULL) == NULL) {
		rad_attrfree(pkt->list);
		free(pkt);
		return NULL;
	}

	/*
	 *	We add the protocol type if this is PPP.
	 */
	if (ppp &&
	rad_addattr(pkt->list, PW_FRAMED_PROTOCOL, 0, htonl(PW_PPP), NULL) == NULL) {
		rad_attrfree(list);
		free(pkt);
		return NULL;
	}

	if (ppp &&
	rad_addattr(pkt->list, PW_SERVICE_TYPE, 0, htonl(PW_FRAMED_USER), NULL) == NULL) {
		rad_attrfree(list);
		free(pkt);
		return NULL;
	}

	return pkt;
}

/*
 *	Build an accounting request.
 */
RADPKT *rad_buildacct(char *secret_key, struct auth *ai,
	int port, int islogin)
{
	RADPKT *pkt;
	int i, s, p, c;
	unsigned int h, a;

	if ((pkt = malloc(sizeof(RADPKT))) == NULL)
		return NULL;
	memset(pkt, 0, sizeof(RADPKT));

	/*
	 *	Build the header.
	 */
	pkt->code = PW_ACCOUNTING_REQUEST;;
	pkt->id   = rad_id();
	memset(pkt->vector, 0, AUTH_VECTOR_LEN);
	strcpy(pkt->secret,secret_key);

	pkt->timestamp = time(NULL);

	/*
	 *	Tell the server what kind of request this is.
	 */
	i = islogin ? PW_STATUS_START : PW_STATUS_STOP;
	if ((pkt->list = rad_addattr(pkt->list, PW_ACCT_STATUS_TYPE, 0, htonl(i), NULL)) == NULL) {
		free(pkt);
		return NULL;
	}

	/*
	 *	Add the login name.
	 */
	if (ai->login && ai->login[0]) {
		if (rad_addattr(pkt->list, PW_USER_NAME, -1, 0, ai->login) == NULL) {
			rad_attrfree(pkt->list);
			free(pkt);
			return NULL;
		}
	}

	/*
	 *	Now we add the client id (our ip address),
	 *	and the port id (tty number).
	 */
	if (rad_addattr(pkt->list, PW_NAS_IP_ADDRESS, 0, mainconf.ipno, NULL) == NULL) {
		rad_attrfree(pkt->list);
		free(pkt);
		return NULL;
	}

	if (rad_addattr(pkt->list, PW_NAS_PORT_ID, 0, htonl(port), NULL) == NULL) {
		rad_attrfree(pkt->list);
		free(pkt);
		return NULL;
	}
	if (rad_addattr(pkt->list, PW_NAS_PORT_TYPE, 0, htonl(ai->porttype), NULL) == NULL) {
		rad_attrfree(pkt->list);
		free(pkt);
		return NULL;
	}

	/*
	 *	Add input and output octets.
	 */
	if ((!islogin && (ai->sent_bytes || ai->recv_bytes)) &&
	     rad_addattr(pkt->list,
	     PW_ACCT_OUTPUT_OCTETS, 0, htonl(ai->sent_bytes), NULL) == NULL) {
		rad_attrfree(pkt->list);
		free(pkt);
		return NULL;
	}
	if ((!islogin && (ai->sent_bytes || ai->recv_bytes)) &&
	     rad_addattr(pkt->list,
	     PW_ACCT_INPUT_OCTETS, 0, htonl(ai->recv_bytes), NULL) == NULL) {
		rad_attrfree(pkt->list);
		free(pkt);
		return NULL;
	}


	/*
	 *	We have to add a unique identifier, the same
	 *	for login as for logout.
	 */
	if (rad_addattr(pkt->list, PW_ACCT_SESSION_ID, -1, 0, ai->session) == NULL) {
		rad_attrfree(pkt->list);
		free(pkt);
		return NULL;
	}

	/*
	 *	Add connection info on login AND logout. That's what
	 *	a portmaster does too..
	 */
	if (ai->conn_info[0]) {
		if (rad_addattr(pkt->list, PW_CONNECT_INFO, -1, 0, ai->conn_info) == NULL) {
			rad_attrfree(pkt->list);
			free(pkt);
			return NULL;
		}
	}

	/*
	 *	Include session time if this is a logout.
	 */
	if (!islogin &&
	    rad_addattr(pkt->list, PW_ACCT_SESSION_TIME, 0,
	    htonl(time(NULL) - ai->start), NULL) == NULL) {
		rad_attrfree(pkt->list);
		free(pkt);
		return NULL;
	}

	/*
	 *	We'll have to add some more fields here:
	 *	- service type
	 *	- login service
	 *	- framed protocol
	 *	- framed IP address
	 *	- framed compression
	 *	- login IP host
	 *
	 */
	i = -1;
	s = -1;
	p = -1;
	c = -1;
	h = 0;
	a = 0;
	switch(ai->proto) {
		case P_SHELL:
			s = PW_SHELL_USER;
			break;
		case P_TELNET:
			s = PW_LOGIN_USER;
			i = PW_TELNET;
			h = ai->address;
			break;
		case P_RLOGIN:
			s = PW_LOGIN_USER;
			i = PW_RLOGIN;
			h = ai->address;
			break;
		case P_TCPCLEAR:
		case P_TCPLOGIN:
			s = PW_LOGIN_USER;
			i = PW_TCP_CLEAR;
			h = ai->address;
			break;
		case P_PPP:
		case P_SLIP:
		case P_CSLIP:
			s = PW_FRAMED_USER;
			a = ai->address;
			break;
	}
	switch(ai->proto) {
		case P_PPP:
			p = PW_PPP;
			c = PW_VAN_JACOBSEN_TCP_IP;
			break;
		case P_SLIP:
			p = PW_SLIP;
			break;
		case P_CSLIP:
			p = PW_SLIP;
			c = PW_VAN_JACOBSEN_TCP_IP;
	}
	if (s > 0 && rad_addattr(pkt->list, PW_SERVICE_TYPE, 0, htonl(s), NULL) == NULL) {
buildacct_error:
		rad_attrfree(pkt->list);
		free(pkt);
		return NULL;
	}
	if (i >= 0 && rad_addattr(pkt->list, PW_LOGIN_SERVICE, 0, htonl(i), NULL) == NULL)
		goto buildacct_error;

	if (p >= 0 && rad_addattr(pkt->list, PW_FRAMED_PROTOCOL, 0, htonl(p), NULL) == NULL)
		goto buildacct_error;

	if (a > 0 && rad_addattr(pkt->list, PW_FRAMED_IP_ADDRESS, 0, a, NULL) == NULL)
		goto buildacct_error;

	if (c >= 0 && rad_addattr(pkt->list, PW_FRAMED_COMPRESSION, 0, htonl(c), NULL) == NULL)
		goto buildacct_error;

	if (h > 0 && rad_addattr(pkt->list, PW_LOGIN_IP_HOST, 0, h, NULL) == NULL)
		goto buildacct_error;

	return pkt;
}


static int rad_send(int port, unsigned int host1, unsigned int host2,
	char *recvbuf, int size, RADPKT *req,int acctpkt)
{
	char sendbuf[4096];
	int sendlen;
	struct sockaddr_in salocal, saremote, sa_replied;
	int local_port;
	int s, len, i;
	unsigned int remote = host1;
	fd_set readfds;
	struct timeval tv;

	sendlen = rad_buildreq(sendbuf, sizeof(sendbuf), req);
	if (sendlen < 0) return -2;

	/* if an acct pkt, create the authenticator */
	if (acctpkt == 1) {
		i = strlen(req->secret);
		memcpy(&sendbuf[sendlen],req->secret,i);
		radius_md5_calc(&sendbuf[4],sendbuf,sendlen+i);
	}

	/*
	 *	We have to bind the socket in order to
	 *	receive an answer back...
	 */
	if ((s = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		nsyslog(LOG_ERR, "socket: %m");
		return -2;
	}

	memset(&salocal, 0, sizeof(salocal));
	salocal.sin_family = AF_INET;
	salocal.sin_addr.s_addr = INADDR_ANY;
	local_port = 1025;
	do {
		local_port++;
		salocal.sin_port = htons((unsigned short)local_port);
	} while ((bind(s, (struct sockaddr *)&salocal, sizeof (salocal)) < 0)
			&& local_port < 64000);

	if (local_port >= 64000) {
		close(s);
		nsyslog(LOG_ERR, "bind: %m");
		return -2;
	}

	/*
	 *	saremote is the remote radius server address.
	 */
	memset(&saremote, 0, sizeof(saremote));
	saremote.sin_family = AF_INET;
	saremote.sin_port = htons(port);

	if (!host1 && host2) {
		host1 = host2;
		host2 = 0;
	}

	/*
	 *	Try at most 10 times. We time out after 3 seconds,
	 *	and after 3 tries we switch to an alternate server, and
	 *	keep switching after every timeout.
	 */
	for(i = 0; i < 10; i++) {
		if (i > 2) {
			nsyslog(LOG_INFO, "radius@%s not responding",
				dotted(remote));
			if (host2) {
				if (remote == host1)
					remote = host2;
				else
					remote = host1;
			}
		}

		/*
		 *	Connect to the radius server, and
		 *	send the request.
		 */
		saremote.sin_addr.s_addr = remote;
		sendto(s, sendbuf, sendlen, 0,
			(struct sockaddr *)&saremote, sizeof(saremote));

		/*
		 *	Now wait for an answer.
		 */
		tv.tv_sec = 3;
		tv.tv_usec = 0;
		FD_ZERO(&readfds);
		FD_SET(s, &readfds);
		len = sizeof(saremote);
		if (select(s + 1, &readfds, NULL, NULL, &tv) > 0 &&
		    recvfrom(s, recvbuf, size, 0,
				(struct sockaddr *)&sa_replied, &len) > 0)
					break;
	}
	close(s);

	if (i == 10) {
		sprintf(sendbuf, "Radius server%s %s",
			host2 ? "s" : "", dotted(host1));
		if (host2)
			sprintf(sendbuf + strlen(sendbuf),
				" and %s", dotted(host2));
		strcat(sendbuf, " not responding");
		nsyslog(LOG_ERR, sendbuf);
		return -1;
	}
	if (i > 2 && host2) nsyslog(LOG_INFO, "radius@%s replied eventually",
		dotted(sa_replied.sin_addr.s_addr));

	return len;
}

/*
 *	Ask a radius server to authenticate us,
 */
int rad_client(int port, struct auth *ai, int ppp)
{
	int ret, len, value;
	unsigned int i;
	int islogin = 0;
	int isframed = 0;
	AUTH_HDR *reply;
	RADPKT *req;
	int a_len, a_type;
	char *a_val;
	char recvbuf[4096];
	char *ptr = recvbuf, *maxptr;
	int oldproto;

	/*
	 *	Set the message to some error in case we fail.
	 */
	strcpy(ai->message, "Internal error\r\n");

        /*
         *      First, build the request.
         */
	if ((req = rad_buildauth(lineconf[port].secret, ai->login, ai->passwd,
	     port, ppp, ai->porttype)) == NULL)
		return -1;

	/*
	 *	Now connect to the server.
	 */
#ifdef UUCPHACK
	if (ai->login[0] == 'U') {
		if (lineconf[port].uauthhost1 == 0)
		    lineconf[port].uauthhost1 = lineconf[port].authhost1;
		if (lineconf[port].uauthhost2 == 0)
		    lineconf[port].uauthhost2 = lineconf[port].authhost2;
		len = rad_send(PW_AUTH_UDP_PORT, lineconf[port].uauthhost1,
			lineconf[port].uauthhost2,
			recvbuf, sizeof(recvbuf), req,0);
	} else
#endif
	len = rad_send(PW_AUTH_UDP_PORT, lineconf[port].authhost1,
			lineconf[port].authhost2,
			recvbuf, sizeof(recvbuf), req,0);
	rad_attrfree(req->list);
	free(req);

	if (len < 0) {
		if (len == -1) strcpy(ai->message,
			"RADIUS server not responding\r\n");
		return -1;
	}
	reply = (AUTH_HDR *)ptr;
#if 0 /* XXX Debug */
	nsyslog(LOG_DEBUG, "Got a reply, code %d, length %d", reply->code, ntohs(reply->length));
#endif
	ret = (reply->code == PW_AUTHENTICATION_ACK) ? 0 : -1;
	ai->message[0] = 0;
	oldproto = ai->proto;
	ai->proto = 0;

	/*
	 *	Put the reply into our auth struct.
	 */
	maxptr = ptr + ntohs(reply->length);
	ptr += 20;

	while(ptr < maxptr) {
		a_type = ptr[0];
		a_len = ptr[1];
		a_val = &ptr[2];
		ptr += a_len;
#if 0 /* XXX - Debug */
		nsyslog(LOG_DEBUG, "Attribute type %d, length %d", a_type, a_len);
#endif
		if (a_type == 0 || a_len < 2) break;
		a_len -= 2;
		value = ntohl(*(int *)a_val);
		switch(a_type) {
			case PW_SERVICE_TYPE:
				/* Framed or login. */
				switch(value) {
					case PW_SHELL_USER:
						ai->proto = P_SHELL;
						break;
					case PW_LOGIN_USER:
						islogin = 1;
						break;
					case PW_FRAMED_USER:
						isframed = 1;
						break;
					default:
						break;
				}
				break;
			case PW_LOGIN_SERVICE:
				islogin = 1;
				switch(value) {
					case PW_TELNET:
						ai->proto = P_TELNET;
						break;
					case PW_RLOGIN:
						ai->proto = P_RLOGIN;
						break;
					case PW_TCP_CLEAR:
						ai->proto = P_TCPCLEAR;
						break;
					case PW_PORTMASTER:
					default:
						islogin = 0;
						/* Unknown to us. */
						break;
				}
				break;
			case PW_LOGIN_IP_HOST:
				ai->address = *(unsigned int *)a_val;
				break;
			case PW_LOGIN_PORT:
				ai->port = value;
				break;
			case PW_FRAMED_IP_ADDRESS:
				isframed = 1;
				i = *(unsigned int *)a_val;
				if (value != 0xFFFFFFFE)
					ai->address = i;
				break;
			case PW_FRAMED_IP_NETMASK:
				ai->netmask = *(unsigned int *)a_val;
				break;
			case PW_FRAMED_MTU:
				ai->mtu = value;
				break;
			case PW_IDLE_TIMEOUT:
				ai->idletime = value;
				break;
			case PW_FRAMED_COMPRESSION:
				if (value != PW_VAN_JACOBSEN_TCP_IP)
					break;
				if (ai->proto == 0 || ai->proto == P_SLIP)
					ai->proto = P_CSLIP;
				break;
			case PW_FRAMED_PROTOCOL:
				isframed = 1;
				if (value == PW_PPP)
					ai->proto = P_PPP;
				else if (ai->proto == 0)
					ai->proto = P_SLIP;
				break;
			case PW_REPLY_MESSAGE:
				len = a_len;
				if (len >= sizeof(ai->message))
					len = sizeof(ai->message) - 1;
				strncpy(ai->message, a_val, len);
				ai->message[len] = 0;
				break;
		}
	}

	if (isframed && ai->address == 0 && lineconf[port].ipno)
		ai->address = lineconf[port].ipno;

	if (islogin && ai->address == 0 && lineconf[port].host)
		ai->address = lineconf[port].host;

	if (ai->proto == 0)
		ai->proto = oldproto;

	if (ret == 0) {
		rad_sessionid(ai->session);
		ai->start = time(NULL);
	} else {
		nsyslog(LOG_INFO, "authentication failed (%s/%s) %s",
			ai->login, ai->passwd, ai->message);
	}

	return ret;
}

/*
 *	Send accounting info to a RADIUS server.
 */
int rad_acct(int port, struct auth *ai, int islogin)
{
	int len;
	AUTH_HDR *reply;
	RADPKT *req;
	char recvbuf[4096];
	char *ptr = recvbuf;

	/*
	 *	HACK/FIXME: this should not be here, but it's easier..
	 */
	update_utmp(port, ai, islogin);

        /*
         *      First, build the request.
         */
	if ((req = rad_buildacct(lineconf[port].secret, ai, port, islogin)) == NULL)
		return -1;

	/*
	 *	Now connect to the server.
	 */
#ifdef UUCPHACK
	if (ai->login[0] == 'U') {
		if (lineconf[port].uaccthost1 == 0)
		    lineconf[port].uaccthost1 = lineconf[port].accthost1;
		if (lineconf[port].uaccthost2 == 0)
		    lineconf[port].uaccthost2 = lineconf[port].accthost2;
		len = rad_send(PW_ACCT_UDP_PORT, lineconf[port].uaccthost1,
			lineconf[port].uaccthost2,
			recvbuf, sizeof(recvbuf), req,1);
	} else
#endif
	len = rad_send(PW_ACCT_UDP_PORT, lineconf[port].accthost1,
			lineconf[port].accthost2,
			recvbuf, sizeof(recvbuf), req,1);
	rad_attrfree(req->list);
	free(req);

	if (len < 0)
		return -1;

	reply = (AUTH_HDR *)ptr;
#if 0 /* XXX - Debug */
	nsyslog(LOG_DEBUG, "Got a reply, code %d, length %d", reply->code, ntohs(reply->length));
#endif

	return 0;
}
