/*
 * lib.c         Generic functions.
 *
 * Version:      @(#)lib.c  1.33  20-May-1997  MvS.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <errno.h>
#include <stdarg.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "server.h"

/*
 * Malloc and die if failed.
 */
void *xmalloc(int size)
{
  void *p;

  if ((p = malloc(size)) == NULL) {
	nsyslog(LOG_ERR, "Virtual memory exhausted.\n");
	exit(1);
  }
  memset(p, 0, size);
  return p;
}

/*
 * Signal functions.
 */
void block(int sig)
{
  sigset_t set;

  sigemptyset(&set);
  sigaddset(&set, sig);
  sigprocmask(SIG_BLOCK, &set, NULL);
}
void unblock(int sig)
{
  sigset_t set;

  sigemptyset(&set);
  sigaddset(&set, sig);
  sigprocmask(SIG_UNBLOCK, &set, NULL);
}


/*
 * This part is for the automatic PPP detection.
 * getline() can check for it.
 */
#define PPP1 "\x7e\xff\x03\xc0\x21"
#define PPP2 "\x7e\xff\x7d\x23\xc0\x21"
#define PPP3 "\x7e\x7d\xdf\x7d\x23\xc0\x21"
static int pppseen(int ch)
{
	static char buf[16];

	if (ch == -1) {
	    memset(buf, 0, sizeof(buf));
	    return 0;
	}

	memmove(buf, buf + 1, 15);
	buf[14] = ch;
	if (strcmp(buf + 15 - 5, PPP1) == 0 ||
	    strcmp(buf + 15 - 6, PPP2) == 0 ||
	    strcmp(buf + 15 - 7, PPP3) == 0)
		return 1;

	return 0;
}

/*
 * Read and echo a line.
 *
 * Returns -1 on EOF, 0 on success and 1 on PPP detect.
 */
int getline(int ppp, char *prompt, int echo, char *buffer, int size)
{
  int pos = 0;
  unsigned char c;
  int echonl = 1;
  static int pppinit = 0;
  static int crseen = 0;
  int len;

  if (ppp && !pppinit) pppseen(-1);

  if (prompt && prompt[0]) {
	len = strlen(prompt);
	write(0, prompt, len);
	if (prompt[len - 1] != ' ')
		write(0, " ", 1);
  }
  buffer[0] = 0;

  while(read(0, &c, 1) == 1) {

	if (ppp && pppseen(c)) {
		buffer[0] = 0;
		return 1;
	}

	if (c == '\n' && crseen) {
		crseen = 0;
		continue;
	}
	crseen = 0;

	switch(c) {
		case '\n':
		case '\r':
			if (c == '\r') crseen = 1;
			buffer[pos] = 0;
			if (echonl) write(1, "\r\n", 2);
			return 0;
		case 4: /* Control-D */
			if (pos == 0) {
				if (echonl) write(1, "\r\n", 2);
				return -1;
			}
			break;
		case 5: /* Control-E */
			echo = echonl = 0;
			break;
		case 127:
		case 8: /* Backspace. */
			if (pos == 0) {
				write(1, "\007", 1);
				break;
			}
			if (echo) write(1, "\010 \010", 3);
			pos--;
			break;
		default:
			if (c < 32 || c > 127 || pos >= size - 1) {
				write(1, "\007", 1);
				break;
			}
			if (echo) write(1, &c, 1);
			buffer[pos++] = c;
			break;
	}
  }
  buffer[pos] = 0;
  return -1;
}

/*
 *	Print number in static buffer (itoa).
 */
char *num(int i)
{
  static char buf[16];

  sprintf(buf, "%d", i);
  return buf;
}

/*
 *	Print dotted IP address.
 */
char *dotted(unsigned int a)
{
	struct in_addr ia;

	ia.s_addr = a;
	return inet_ntoa(ia);
}

/*
 *	Expand `%' stuff.
 *
 *	%l: login name
 *	%p: port number
 *	%b: speed
 *	%i: Local IP
 *	%j: remote IP
 *	%m: netmask
 *	%t: MTU
 *	%r: MRU
 */
char *percent(int port, struct auth *ai, char *in)
{
  char *p, *s, *t, *out;
  int l, max;

  if (in == NULL) return NULL;

  max = strlen(in) + 100;
  out = malloc(max);
  if (out == NULL) {
	nsyslog(LOG_ERR, "out of memory");
	return NULL;
  }
  s = out;

  for(p = in; *p && max > 0; p++) {
	if (*p != '%') {
		*s++ = *p;
		max--;
		continue;
	}
	p++;
	t = "";
	switch(*p) {
		case 'l':
			if (ai) t = ai->login;
			break;
		case 'p':
			t = num(port);
			break;
		case 'b':
			t = num(lineconf[port].speed);
			break;
		case 'i':
			t = dotted(mainconf.ipno);
			break;
		case 'j':
			if (ai) t = dotted(ai->address);
			break;
		case 'm':
			if (ai) t = dotted(ai->netmask);
			break;
		case 't':
			if (ai) t = num(ai->mtu);
			break;
		case 'r':
			if (ai) t = num(ai->mru);
			break;
		case 'I':
			if (ai) t = num(ai->idletime);
			break;
		case 'h':
			t = mainconf.hostname;
			break;
		case '%':
			t = "%";
			break;
		default:
			*s++ = *--p;
			max--;
			continue;
	}
	l = strlen(t);
	if (max <= l)
		break;
	strcpy(s, t);
	s += l;
	max -= l;
  }
  *s = 0;

  return out;
}


/*
 *	Initialize struct and read config file.
 */
int rad_init(int port, struct auth *ai, char *tty)
{
  extern void setlogremote(unsigned int ipaddr, int block_open);
  static char ident[64];
  int fac, oport;

  /*
   *	Read the config file.
   */
  initcfg();
  if (readcfg() < 0)
	return -1;

  /*
   *	Find out port number if needed.
   */
  oport = port;
  if (tty && port == -2) /* PPP daemon */
	port = rad_portno(tty);

  /*
   *	Open the logfile.
   */
  if (mainconf.syslog) setlogremote(mainconf.syslog, 1);
  sprintf(ident, "port[S%d]", (port < 0) ? 9999 : port);
  fac = LOG_FAC(LOG_LOCAL0) + mainconf.facility;
  nopenlog(ident, LOG_CONS|LOG_NDELAY, LOG_MAKEPRI(fac, 0));
  if (port >= 0 && lineconf[port].debug == 0) {
	if (oport == -2) /* PPP daemon */
		nsetlogmask(LOG_UPTO(LOG_NOTICE));
	else
		nsetlogmask(LOG_UPTO(LOG_INFO));
  }

  if (port < 0) {
	nsyslog(LOG_ERR, "%s: not in config file", tty);
	return -1;
  }

  /*
   *	Fill in the auth struct for now.
   */
  memset(ai, 0, sizeof(struct auth));
  strcpy(ai->login, "NONE");
  ai->proto   = lineconf[port].protocol;
  ai->address = lineconf[port].ipno;
  ai->netmask = lineconf[port].netmask;
  ai->mtu     = lineconf[port].mtu;
  ai->mru     = lineconf[port].mru;
  ai->localip = mainconf.ipno;

  if (ai->netmask == 0) ai->netmask = 0xFFFFFFFF;

  return port;
}

/*
 *	Get the port number belonging to this tty
 *	(the pppd uses this).
 */
int rad_portno(char *tty)
{
  int i;

  if (strncmp(tty, "/dev/", 5) == 0) tty += 5;
  for(i = 0; i < MAXLINES; i++) {
	if (lineconf[i].tty && strcmp(lineconf[i].tty, tty) == 0)
		return i;
  }
  return -1;
}

