/*
 * getty.c	This is the part that talks with the modem,
 *		does the login stuff etc.
 *
 * Version:	@(#)getty.c  1.33  27-Jun-1997  MvS.
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/time.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>
#include <ctype.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <termios.h>
#include <errno.h>

#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "server.h"

/*
 *	Turn CR translation on or off.
 */
static void crtrans(int inon, int outon)
{
  struct termios tty;

  tcgetattr(0, &tty);
  if (inon) {
	tty.c_iflag |= ICRNL|IGNCR;
  } else {
	tty.c_iflag &= ~(ICRNL|IGNCR);
  }
  if (outon) {
	tty.c_oflag |= ONLCR|OPOST;
  } else {
	tty.c_oflag &= ~(ONLCR|OPOST);
  }
  tcsetattr(0, TCSANOW, &tty);
}


/*
 *	Get the lockfile.
 */
static int lockfile(char *lockdir, char *tty, char *lock)
{
  struct stat st;

  /*
   *	First some checks to see if we do want to
   *	use a lockdir and if it exists.
   */
  if (lockdir == NULL || lockdir[0] != '/')
	return -1;
  if (stat(lockdir, &st) < 0 || !S_ISDIR(st.st_mode))
	return -1;

  if (tty[0] == '/') tty += 5;
  sprintf(lock, "%s/LCK..%s", lockdir, tty);

  return 0;
}


/*
 *	See if a device is locked.
 */
static int islocked(char *lockdir, char *tty)
{
  char lock[1024];
  FILE *fp;
  int pid;

  if (lockfile(lockdir, tty, lock) < 0)
	return 0;
  if ((fp = fopen(lock, "r")) == NULL)
	return 0;
  if (fscanf(fp, "%d", &pid) != 1 || (kill(pid, 0) < 0 && errno == ESRCH)) {
	fclose(fp);
	unlink(lock);
	return 0;
  }
  fclose(fp);

  return 1;
}

/*
 *	Unlock.
 */
static void unlock(char *lockdir, char *tty)
{
  char lock[1024];

  if (lockfile(lockdir, tty, lock) == 0)
	unlink(lock);
}

/*
 *	Create a lockfile.
 *	Returns: -1   some error
 *		  0   success
 *		  1   already locked.
 */
static int dolock(char *lockdir, char *tty)
{
  char tmp[1024];
  char lock[1024];
  FILE *fp;
  int i;

  if (lockfile(lockdir, tty, lock) < 0)
	return 0;

  strcpy(tmp, lock);
  sprintf(tmp + strlen(tmp), ".%d", getpid());
  unlink(tmp);

  /*
   *	Already locked?
   */
  if (islocked(lockdir, tty))
	return 1;

  /*
   *	Try to lock.
   */
  if ((fp = fopen(tmp, "w")) == NULL) {
	nsyslog(LOG_ERR, "%s: %m", tmp);
	return -1;
  }
  fprintf(fp, "%10d portslave root\n", getpid());
  fclose(fp);

  for(i = 0; i < 5; i++) {
	if (i > 0) sleep(5);
	(void)islocked(lockdir, tty);
	if (link(tmp, lock) == 0) break;
  }
  unlink(tmp);

  return (i == 5) ? 1 : 0;
}

/*
 *	Disassociate from our controlling tty.
 */
static void disass_ctty()
{
  pid_t pid;
  int fl;

  /*
   *	First let go of our old tty.
   *	This looks like black magic. It is ;)
   */
  pid = getpid();
#ifndef __ELF__
  /*
   *	a.out systems generally don't have getsid()
   */
  (void)setpgid(0, getpgid(getppid()));
  (void)setsid();
#else
  if (getsid(pid) != pid) {
	if (setpgid(0, getpgid(getppid())) < 0)
		nsyslog(LOG_WARNING, "setpgid: %m");
	if (setsid() < 0)
		nsyslog(LOG_WARNING, "setsid: %m");
  }
#endif
  if ((fl = open("/dev/tty", O_RDWR)) >= 0) {
	signal(SIGHUP, SIG_IGN);
	if (ioctl(fl, TIOCNOTTY, (char *)1) < 0)
		nsyslog(LOG_WARNING, "ioctl TIOCNOTTY: %m");
	signal(SIGHUP, SIG_DFL);
	close(fl);
  }
}


/*
 *	Open the serial line and put it into raw/no echo mode.
 */
static int serial_open(int port, int usedcd)
{
  int fd, fl;
  struct termios tio;
  char *ltty = lineconf[port].tty;
  char tty[64];

  if (ltty == NULL || ltty[0] == 0) {
	nsyslog(LOG_ERR, "no tty specified");
	return -1;
  }
  if (ltty[0] != '/')
	strcpy(tty, "/dev/");
  else
	tty[0] = 0;
  strcat(tty, ltty);

  if ((fd = open(tty, O_RDWR|O_NONBLOCK|O_NOCTTY)) < 0) {
	nsyslog(LOG_ERR, "%s: %m", tty);
	return -1;
  }

  fl = fcntl(fd, F_GETFL, 0);
  fl &= ~O_NONBLOCK;
  fcntl(fd, F_SETFL, fl);

  (void)dup2(fd, 0);
  (void)dup2(fd, 1);
  (void)dup2(fd, 2);

  if (fd > 2) close(fd);

  (void)tcgetattr(0, &tio);
  tio.c_iflag = IGNPAR|ICRNL;
  tio.c_oflag = OPOST|ONLCR;
  tio.c_lflag = 0; /* ICANON|ECHO|ECHONL*/

  tio.c_cflag = CREAD|CS8|HUPCL;
  if (!usedcd) tio.c_cflag |= CLOCAL;

  switch (lineconf[port].flow) {
	case FLOW_NONE:
		break;
	case FLOW_HARD:
		tio.c_cflag |= CRTSCTS;
		break;
	case FLOW_SOFT:
		tio.c_iflag |= IXON|IXOFF;
		tio.c_cc[VSTART] = 17;
		tio.c_cc[VSTOP]  = 19;
		break;
  }
  switch(lineconf[port].speed) {
	case 1200:
		tio.c_cflag |= B1200;
		break;
	case 2400:
		tio.c_cflag |= B2400;
		break;
	case 4800:
		tio.c_cflag |= B4800;
		break;
	case 9600:
		tio.c_cflag |= B9600;
		break;
	case 19200:
		tio.c_cflag |= B19200;
		break;
	case 38400:
		tio.c_cflag |= B38400;
		break;
	case 57600:
		tio.c_cflag |= B57600;
		break;
	case 115200:
		tio.c_cflag |= B115200;
		break;
	case 230400:
		tio.c_cflag |= B230400;
		break;
  	default:
		nsyslog(LOG_WARNING, "%d: line speed not supported (using 9600)",
			lineconf[port].speed);
		tio.c_cflag |= B9600;
		break;
  }

  (void)tcsetattr(0, TCSANOW, &tio);

  return 0;
}


/*
 *	Run a getty on a port.
 */
int getty(int port, char *conn_info)
{
  struct line_cfg *lc = &lineconf[port];
  char *s;
  fd_set readfds;

  disass_ctty();

  while(1) {
	close(0);
	close(1);
	close(2);

	/*
	 *	First lock or wait if the port is busy.
	 */
	if (dolock(mainconf.lockdir, lc->tty) == 1) {
		while(islocked(mainconf.lockdir, lc->tty))
			sleep(10);
		sleep(1);
		continue;
	}

	if (serial_open(port, lc->dcd) < 0) {
		unlock(mainconf.lockdir, lc->tty);
		return -1;
	}

	/*
	 *	See if we need to initialize the modem.
	 */
	crtrans(0,0 );
	if (chat(1, lc->initchat, NULL) < 0) {
		crtrans(1, 1);
		nsyslog(LOG_ERR, "initchat failed.");
		unlock(mainconf.lockdir, lc->tty);
		return -1;
	}
	crtrans(1, 1);

	/*
	 *	Now, do we need to wait for anything to happen?
	 */
	if (lc->aa) {
		/*
		 *	Wait for DCD. This is pretty linux specific.
		 */
		nsyslog(LOG_INFO, "waiting for DCD");
		unlock(mainconf.lockdir, lc->tty);
		if (ioctl(0, TIOCMIWAIT, TIOCM_CD) < 0) {
			nsyslog(LOG_WARNING, "waiting for DCD: %m");
			unlock(mainconf.lockdir, lc->tty);
			return -1;
		}
		if (dolock(mainconf.lockdir, lc->tty) == 1) {
			sleep(5);
			continue;
		}
		
	} else if (lc->waitfor && lc->waitfor[0]
			&& strcmp(lc->waitfor, "\"\"") != 0) {
		/*
		 *	First wait for something to happen on
		 *	our input, then check for a lockfile.
		 */
waitfor_loop:
		nsyslog(LOG_INFO, "waiting for %s", lc->waitfor);
		usleep(250000);
		tcflush(0, TCIOFLUSH);
		FD_ZERO(&readfds);
		FD_SET(0, &readfds);
		unlock(mainconf.lockdir, lc->tty);
		while(select(1, &readfds, NULL, NULL, NULL) < 0 &&
		      errno == EINTR)
			;
		if (dolock(mainconf.lockdir, lc->tty) == 1) {
			sleep(5);
			continue;
		}
		chat_timeout = 20;
		if (chat_expect(0, lc->waitfor, NULL) < 0) {
			if (errno == ETIMEDOUT) {
				nsyslog(LOG_INFO, "did not see %s - retry",
					lc->waitfor);
				goto waitfor_loop;
			}
			nsyslog(LOG_WARNING, "waiting for %s: %m", lc->waitfor);
			unlock(mainconf.lockdir, lc->tty);
			return -1;
		}
	}
	break;
  }

  /*
   *	Now force the new tty into becoming our
   *	controlling tty. This will also kill other
   *	processes hanging on this tty.
   */
  if (ioctl(0, TIOCSCTTY, 1) < 0)
	nsyslog(LOG_WARNING, "TIOCSCTTY: %m");

  /*
   *	Okay, something has happened. Do we need to talk to the modem?
   */
  if (lc->answer && lc->answer[0]) {
	chat_timeout = 60;
	crtrans(0, 0);
	if (chat(0, lc->answer, conn_info) < 0) {
		crtrans(1, 1);
		nsyslog(LOG_WARNING, "answering modem: %m");
		unlock(mainconf.lockdir, lc->tty);
		return -1;
	}
	crtrans(1, 1);
  }

  /*
   *	Delay just a little bit and flush junk.
   */
  usleep(250000);
  tcflush(0, TCIOFLUSH);

  /*
   *	Show the banner and say hello.
   */
  if (lc->issue && lc->issue[0]) {
	s = percent(port, NULL, lc->issue);
	fprintf(stdout, "%s\n", s);
	fflush(stdout);
	if (s) free(s);
  }

  return 0;
}


/*
 *	Imitate a modem on a port.
 */
int emumodem(int port, char *conn_info)
{
  struct line_cfg *lc = &lineconf[port];
  char *s;
  char buf[128];

  disass_ctty();

  while(1) {
	close(0);
	close(1);
	close(2);

	/*
	 *	First lock or wait if the port is busy.
	 */
	if (dolock(mainconf.lockdir, lc->tty) == 1) {
		while(islocked(mainconf.lockdir, lc->tty))
			sleep(10);
		sleep(1);
		continue;
	}
	break;
  }
  if (serial_open(port, lc->dcd) < 0) {
	unlock(mainconf.lockdir, lc->tty);
	return -1;
  }

  /*
   *	Now force the new tty into becoming our
   *	controlling tty. This will also kill other
   *	processes hanging on this tty.
   */
  if (ioctl(0, TIOCSCTTY, 1) < 0)
	nsyslog(LOG_WARNING, "TIOCSCTTY: %m");

  /*
   *	Loop, and emulate a modem.
   */
  crtrans(0, 0);
  printf("\r\nNO CARRIER\r\n");
  while(1) {
	fflush(stdout);
	getline(0, "", 1, buf, 128);
	if (strncasecmp(buf, "ATD", 3) == 0) {
		sleep(3);
		printf("CONNECT 57600\r\n");
		strcpy(conn_info, "57600/DIRECT");
		fflush(stdout);
		break;
	}
	if (strncasecmp(buf, "AT&V", 4) == 0) {
		printf("OK - Portslave Modem emulator\r\n");
		continue;
	}
	if (strncasecmp(buf, "AT", 2) == 0) {
		printf("OK\r\n");
		continue;
	}
	if (buf[0]) printf("ERROR\r\n");
  }
  crtrans(1, 1);

  /*
   *	Delay just a little bit and flush junk.
   */
  usleep(250000);
  tcflush(0, TCIOFLUSH);

  /*
   *	Show the banner and say hello.
   */
  if (lc->issue && lc->issue[0]) {
	s = percent(port, NULL, lc->issue);
	fprintf(stdout, "%s\n", s);
	fflush(stdout);
	if (s) free(s);
  }

  return 0;
}


/*
 *	Alarm handler for the login timeout.
 */
static int got_alrm;
static void alrm_handler()
{
  got_alrm = 1;
}

/*
 *	We've got a connection. Now let the user login.
 */
int do_login(int port, char *login, char *pass)
{
  struct line_cfg *lc = &lineconf[port];
  int r;
  char *p = lc->prompt ? lc->prompt : "login: ";
  char *s;
  int to = 60;

  got_alrm = 0;
  signal(SIGALRM, alrm_handler);
  if (lineconf[port].dcd == 0) to = 0;

  s = percent(port, NULL, p);

  alarm(to);
  crtrans(0, 1);
  do {
	r = getline(1, s, 1, login, 64);
  } while(login[0] == 0 && r == 0);
  alarm(0);
  crtrans(1, 1);

  if (r == 1) {
	strcpy(login, "AutoPPP");
	pass[0] = 0;
  }

  if (r >= 0)
	nsyslog(LOG_INFO, "Detected login for %s", login);
  else
	nsyslog(LOG_INFO, "Login timed out / quit");

  if (r == 1) return 0;

  if (mainconf.locallogins && login[0] == '!') {
	if (s) free(s);
	return 0;
  }

  alarm(to);
  crtrans(0, 1);
  if (r != 0 ||
      (r = getline(1, "Password: ", 0, pass, 64)) != 0) {
	if (r == 1) {
		strcpy(login, "AutoPPP");
		pass[0] = 0;
		r = 0;
	}
  }
  alarm(0);
  crtrans(1, 1);
  if (s) free(s);

  return r;
}

