/*
 * chat.c        This is a generic chat handler
 *               for initialization of the modems.
 *
 * Version:      (@)#chat.c  1.00  12-Sep-1995  MvS.
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <fcntl.h>
#include <signal.h>
#include <termios.h>
#include <sys/ioctl.h>
#include "server.h"

static int timed_out;
int chat_timeout = 10;      /* Seconds */
int chat_send_delay = 1;  /* Tenths  */

#if 1
  #define CDEBUG1(x) nsyslog(LOG_DEBUG, x)
  #define CDEBUG2(x, y) nsyslog(LOG_DEBUG, x, y)
#else
  #define CDEBUG1(x)
  #define CDEBUG2(x, y)
#endif

/*
 *	Set timeout flag on alarm().
 */
static void alrm_handler()
{
  timed_out++;
}

/*
 *	Parse string and replace escape sequences.
 */
static void tstr(int sending, char *s)
{
  char *p = s;

  while(*s) {

	/* Skip quotes. */
	if (*s == '"') {
		s++;
		continue;
	}

	/* ^ for control character. */
	if (*s == '^') {
		*p++ = *++s & 0x1f;
		if (*s) s++;

	/* Some escaped character. */
	} else if (*s == '\\') {
	    switch(*++s) {
		case '\n':
			break;
		case 'r':
			*p++ = '\r';
			break;
		case 'n':
			*p++ = '\n';
			break;
		case 'b':
			*p++ = '\b';
			break;
		case 's':
			*p++ = ' ';
			break;
		case 't':
			*p++ = '\t';
			break;
		case '-':
			/* Keep minus sign escaped. */
			*p++ = '\\';
			*p++ = *s;
			break;
		case '\\':
		case 'd':
		case 'p':
		case 'l':
		case 'c':
		case 'K':
			/* These are double escaped for chat_send(). */
			if (sending) {
				*p++ = '\\';
				*p++ = *s;
				break;
			}
			/*FALLTHRU*/
		default:
			/* See if its an octal number. */
			if (*s >= '0' && *s <= '7') {
				int val = *s - '0';
				while(*++s >= '0' && *s <= '7')
					val = 8*val + (*s - '0');
				s--;
				/* '\0' and '\\' are special. */
				if (val == 0 || val == '\\') {
					*p++ = '\\';
					*p++ = val ? '\\' : 'N';
				} else
					*p++ = val;
			} else
				*p++ = *s;
	    }
	    if (*s) s++;
	} else
		/* Just a normal character. */
		*p++ = *s++;
  }
  *p = 0;
}

/*
 *	The tstr stuff is so usefull that we export it.
 */
void unescape(char *str)
{
  tstr(0, str);
}

/*
 *	Send a string to the remote side. Process any
 *	left-over escape sequences.
 */
static void chat_send(int fd, char *p)
{
  char *s, *start, *end;
  int addcr = 1;
  int flags, val;
  char c;

  CDEBUG2("chat_send(%s)", p);

  /* Translate escape characters, pass 1 */
  if ((s = (char *)malloc(strlen(p) + 1)) == NULL)
	return;
  strcpy(s, p);
  tstr(1, s);

  /* Delay a bit and then flush the input. */
  usleep(chat_send_delay * 100000);
  tcflush(0, TCIFLUSH);

  /* Now send the string. */
  start = end = s;
  while(1) {
	/* Collect characters till backslash or end of string. */
	while(*end != '\\' && *end)
		end++;
  	if (end != start) write(fd, start, end - start);
	if (*end == 0) {
		if (addcr) {
			c = '\r';
			write(fd, &c, 1);
		}
		free(s);
		return;
	}

	/* Special escape sequence. */
	val = -1;
	switch (*++end) {
		case 'd': /* delay */
			sleep(1);
			break;
		case 'p': /* pause */
			usleep(100000);
			break;
		case 'l': /* lower DTR */
			ioctl(fd, TIOCMGET, &flags);
			flags &= ~TIOCM_DTR;
			ioctl(fd, TIOCMSET, &flags);
			sleep(1);
			flags |= TIOCM_DTR;
			ioctl(fd, TIOCMSET, &flags);
			break;
		case 'c': /* Don't add CR */
			addcr = 0;
			break;
		case 'K': /* Send a break */
			tcsendbreak(fd, 0);
			break;
		default: /* Just fill in character. */
			val = *end;
			break;
	}
	/* Write character if needed. */
	if (val >= 0) {
		c = val;
		write(fd, &c, 1);
	}

	/* And continue after escape character. */
	if (*end) end++;
	start = end;
  }
}

/*
 * Skip string until the next '-'.
 */
static char *skip(char *s)
{

  for(; *s; s++) {
	if (*s == '\\' && *(s+1)) s++;
	if (*s == '-') {
		*s++ = 0;
		return s;
	}
  }
  return NULL;
}

/*
 * Compare 2 strings where '@' means ".*[\r\n]"
 */
int chatncmp(char *exp, char *str, int len, char *matchpart)
{
  char *p;
  int r, l;

  nsyslog(LOG_DEBUG, "chatncmp 1: (%s, %s, %d, %s)", exp, str, len,
	matchpart ? "mp" : "--");

  /* No `@': return normal strncmp */
  if (matchpart == NULL || (p = strchr(exp, '@')) == NULL)
	return strncmp(exp, str, len);

  nsyslog(LOG_DEBUG, "chatncmp 2");

  /* Find the `@'. See if we need to look behind the `@' */
  if ((l = p - exp) >= len)
	return -1;

  nsyslog(LOG_DEBUG, "chatncmp 3");

  /* Compare up to the `@' */
  if ((r = strncmp(exp, str, l)) != 0)
	return r;

  nsyslog(LOG_DEBUG, "chatncmp 4");

  /* Now see if the rest of the string contains ".*[\r\n]" */
  for(p = str + l; *p; p++)
	if (*p == '\r' || *p == '\n')
		break;
  if (*p) {
	strncpy(matchpart, str + l, p - (str + l));
	matchpart[p - (str + l)] = 0;
  }

  nsyslog(LOG_DEBUG, "chatncmp 5: *p = %d", *p);

  return *p == 0 ? -1 : 0;
}



/*
 * Expect a string.
 */
int chat_expect(int fd, char *p, char *matchpart)
{
  char *s, *q;
  char *send, *expect;
  char buf[64];
  int len, conn_hack;
  char c;
  int retries = 0;

  /* Copy argument. */
  if ((s = (char *)malloc(strlen(p) + 1)) == NULL) {
	nsyslog(LOG_CRIT, "malloc: %m");
	return -1;
  }
  strcpy(s, p);
  expect = s;

  while(1) {

	/* Prepare expect string. */
	send = skip(expect);
	CDEBUG2("chat_expect(%s)", expect);
  	tstr(0, expect);
	len = strlen(expect);
	if (len > 63) len = 63;

	/* Empty expects always succeed. */
	if (*expect == 0) {
		CDEBUG1("chat_expect - got it");
		free(s);
		return 0;
	}

	/*
	 *	Hack for Connect-Info (a bit ugly tho')
	 */
	conn_hack = 0;
	if (matchpart && (q = strchr(p, '@')) != NULL && q[1] == 0) {
		len--;
		conn_hack = 1;
	}

	/* Wait for string to arrive, or timeout. */
	signal(SIGALRM, alrm_handler);
	siginterrupt(SIGALRM, 1);
	alarm(chat_timeout);
	timed_out = 0;
	memset(buf, 0, 64);

	/* Put every char in a buffer and shift. */
	while(1) {
		if (read(fd, &c, 1) != 1) {
			if (timed_out)
				break;
			nsyslog(LOG_DEBUG,
		"chat_expect error (%s) - got (%s) with error code %d",
				expect, (buf + 63 - len), errno);
			if (retries++ >= 3)
				break;
			continue;
		}
		memmove(buf, buf + 1, 62);
		buf[62] = c;
		/* See if we got it. */
		if (strncmp(expect, buf + 63 - len, len) == 0) {
			/*
			 *	Connect-Info hack again.
			 */
			if (conn_hack) {
				len = 0;
				while(1) {
					if (read(fd, &c, 1) != 1)
						break;
					if (!len && c == ' ')
						continue;
					if (c == '\r' || c == '\n')
						break;
					matchpart[len++] = c;
				}
				if (c != '\r' && c != '\n')
					break;
				matchpart[len] = 0;
			}
			CDEBUG1("chat_expect - got it");
			alarm(0);
			free(s);
			return 0;
		}
	}

	if (!timed_out) {
		CDEBUG2("chat_expect(%s): interrupted", expect);
		alarm(0);
		free(s);
		return -1;
	}

	/* Is there a send string ? */
	if (send == NULL) {
		CDEBUG2("chat_expect(%s): timeout", expect);
		errno = ETIMEDOUT;
		free(s);
		return -1;
	}

	CDEBUG2("chat_expect(%s): timeout (retry)", expect);

	expect = skip(send);
	chat_send(fd, send);

	/* If there's no expect string now - I guess we succeeded? */
	if (expect == NULL) {
		free(s);
		return 0;
	}

  }
  /*NOTREACHED*/
  return -1;
}


/*
 *	This is the main chat loop;
 *	you have to feed it with an argument
 *	vector and a count.
 */
int chatarray(int fd, int argc, char **argv, char *matchpart)
{
  int c, send;

  /* Now do alternative send-expects. */
  for (c = 0; c < argc; c++) {
	send = c & 1;
	if (send) {
		chat_send(fd, argv[c]);
	} else {
		if (chat_expect(fd, argv[c], matchpart) < 0)
			return -1;
	}
  }
  return 0;
}

/*
 *	This is a generic chat; you can
 *	just pass a string to it.
 */
int chat(int fd, char *str, char *matchpart)
{
  char *args[64];
  char *s;
  int i = 0;

  if (str == NULL || str[0] == 0) return 0;

  if ((s = strdup(str)) == NULL) return -1;

  /* Split string up in expect-send pairs. */
  while(*s) {
	if (*s == ' ' || *s == '\t' || i == 0) {
		if (i) {
			/* End of a string. */
			*s++ = 0;
		}
		/* Skip until next non-space. */
		while (*s == ' ' || *s == '\t')
			s++;
		if (*s == 0) continue;
		args[i++] = s;
	}
	if (*s == '"') {
		while(*s && *s != '"') {
			if (*s == '\\' && *(s+1)) s++;
			s++;
		}
	}
	if (*s) s++;
  }
  args[i] = NULL;

  /* Now call the real routine. */
  return chatarray(fd, i, args, matchpart);
}

#ifdef CHATPROG
int main(int argc, char **argv)
{
  argv++;
  argc--;

  return chatarray(1, argc, argv);
}
#endif

