/*
 * libpsr.c	LibPortSlaveRadius.
 *		Shared library to be linked with pppd-2.2.0f
 *		Takes care of the portslave side of things.
 *
 * Version	@(#)libpsr.c 1.01  11-Nov-1997  miquels@cistron.nl
 *
 */
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/socket.h>

#define syslog   nsyslog
#define openlog  nopenlog
#define closelog ncloselog
#define vsyslog  nvsyslog
#include <syslog.h>

#include <linux/if_ppp.h>

#include "pppd.h"
#include "auth.h"

#undef DEBUG
#define DEBUG 1

struct auth thisauth;
int thisport;
static int ppp_unit = -1;

static int get_ppp_stats(int unit, struct ppp_stats *curp)
{
	struct ifpppstatsreq req;
	int s;

	if ((s = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
		return -1;

	memset (&req, 0, sizeof (req));
#ifdef __linux__
    req.stats_ptr = (caddr_t) &req.stats;
#undef ifr_name
#define ifr_name ifr__name
#endif

	sprintf(req.ifr_name, "ppp%d", unit);
	if (ioctl(s, SIOCGPPPSTATS, &req) < 0) {
		syslog(LOG_WARNING, "SIOCGPPPSTATS(ppp%d): %m", unit);
		close(s);
		return -1;
	}
	if (curp) *curp = req.stats;
	close(s);

	if (req.stats.p.ppp_obytes)
		thisauth.sent_bytes = req.stats.p.ppp_obytes;
	if (req.stats.p.ppp_ibytes)
		thisauth.recv_bytes = req.stats.p.ppp_ibytes;

	return 0;
}

/*
 *	Initialize. Called from pppd's main()
 */
int ul_init(char *devnam)
{
	char *s;

#if DEBUG
	syslog(LOG_DEBUG, "ul_init(%s) called", devnam);
#endif

	ppp_syslog  =  &nsyslog;
	ppp_openlog =  &nopenlog;
	ppp_closelog = &ncloselog;

	if ((thisport = rad_init(-2, &thisauth, devnam)) < 0)
		exit(1);

	if ((s = getenv("CONNECT_INFO")) != NULL)
		strcpy(thisauth.conn_info, s);

	return 0;
}

/*
 *	Check the users login name and password.
 */
int ul_login(char *user, char *passwd, char **msg, int *msglen)
{
	char tmp[48];

#if DEBUG
	syslog(LOG_DEBUG, "ul_login(%s) called", user);
#endif
	strncpy(thisauth.login,  user, 32);
	strncpy(thisauth.passwd, passwd, 32);

	if (rad_client(thisport, &thisauth, 1) < 0)
		return UPAP_AUTHNAK;

	/*
	 *	No access if this is no PPP account.
	 */
	if (thisauth.proto != P_PPP)
		return UPAP_AUTHNAK;

	/*
	 * XXX - FIXME: Should we deny access if we fail to
	 *         contact the accounting server ???
	 */
	rad_acct(thisport, &thisauth, 1);

	/*
	 * Now setup IP addresses, netmask etc.
	 */
	if (thisauth.netmask && thisauth.netmask != 0xFFFFFFFF)
		netmask = thisauth.netmask;
	if (thisauth.mtu)
		lcp_allowoptions[0].mru = thisauth.mtu;
	if (thisauth.mru) {
		lcp_wantoptions[0].mru = thisauth.mru;
		lcp_wantoptions[0].neg_mru = 1;
	}
	strcpy(tmp, dotted(thisauth.localip));
	strcat(tmp, ":");
	if (thisauth.address != 0xFFFFFFFF)
		strcat(tmp, dotted(thisauth.address));
	if (setipaddr(tmp) < 0) {
		syslog(LOG_ERR, "bad IP address %s", tmp);
		return UPAP_AUTHNAK;
	}
	if (thisauth.idletime > 0)
		idle_time_limit = thisauth.idletime;

	syslog(LOG_INFO, "user %s logged in", user);
	setenv("LOGNAME", user, 1);
	logged_in = 1;
	return (UPAP_AUTHACK);
}

/*
 *	User has logged out.
 */
int ul_logout(void)
{
#if DEBUG
	syslog(LOG_DEBUG, "ul_logout() called");
#endif
	/*
	 *	Just to be sure - get ppp stats. By now the interface
	 *	is down and ul_ppp_ifdown has already done it for us,
	 *	but oh well...
	 */
	get_ppp_stats(ppp_unit, NULL);
	rad_acct(thisport, &thisauth, 0);

	return 0;
}

/*
 *	This gets called as soon as the ppp interface is "up". Useful
 *	to clear IP rules etc.
 */
int ul_ppp_established(int devno)
{
#if DEBUG
	syslog(LOG_INFO, "ul_ppp_established(%d) called", devno);
#endif

	ppp_unit = devno;
	return 0;
}

/*
 *	This gets called when a ppp interface goes down.
 */
int ul_ppp_disestablished(int devno)
{
#if DEBUG
	syslog(LOG_INFO, "ul_ppp_disestablished(%d) called", devno);
#endif

	ppp_unit = -1;
	return 0;
}

/*
 *	This gets called when the IP layer of a ppp interface goes
 *	down - at this moment the stats _should_ be still valid
 *	(but alas - not always. WHY are the #*&*&^ stats reported as
 *	 zero when the interface is down?)
 */
int ul_ppp_ifdown(int devno)
{
#if DEBUG
	syslog(LOG_INFO, "ul_ppp_ifdown(%d) called", devno);
#endif

	get_ppp_stats(ppp_unit, NULL);
	return 0;
}

