/*
 * spawnit.c	Spawn the appropriate protocol.
 *
 * Version:	@(#)spawnit.c  1.30  24-Oct-1997  MvS.
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <signal.h>
#include <ctype.h>
#include <utmp.h>
#include <time.h>
#include <sys/ioctl.h>
#include <termios.h>
#include <sys/wait.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "server.h"


/*
 *	Get an utmp entry by process id.
 *	This should be a standard function IMHO.
 */
struct utmp *getutpid(struct utmp *in)
{
  struct utmp *u;

  while((u = getutent()) != NULL) {
	if (u->ut_type != INIT_PROCESS &&
	    u->ut_type != LOGIN_PROCESS &&
	    u->ut_type != USER_PROCESS) continue;
	if (u->ut_pid == in->ut_pid) break;
  }
  return u;
}

/*
 *	Relay the SIGHUP to our process group.
 */
static void got_hup(int sig)
{
  struct sigaction sa;
  pid_t pgrp = getpgrp();

  signal(sig, SIG_IGN);
  unblock(sig);

  if (pgrp > 0) kill(-pgrp, sig);

  sa.sa_handler = got_hup;
  sa.sa_flags   = SA_RESTART;
  sigaction(sig, &sa, NULL);
}

/*
 *	Execute but split command line args first.
 */
static int doexec(char *cmd, char *argv0, char *args)
{
  char *argv[32];
  int argc = 0;
  char *s;

  argv[argc++] = argv0;
  s = strtok(args, " \t\n");
  while(s && argc < 31) {
	argv[argc++] = s;
	s = strtok(NULL, " \t\n");
  }
  argv[argc] = NULL;
  return execv(cmd, argv);
}

/*
 *	Execute a rlogin session.
 */
static int do_rlogin(int port, struct auth *ai)
{
  char *host = dotted(ai->address);

  execl(mainconf.rlogin, "rlogin", "-E",
		"-i", ai->login, "-l", ai->login, host, NULL);
  nsyslog(LOG_ERR, "%s: %m", mainconf.rlogin);
  return -1;
}

/*
 *	Execute a telnet session.
 *	Added 1997-03-19 by Stephan Austermuehle <sa@netaccess.de>
 */
static int do_telnet(int port, struct auth *ai)
{
  char *host = dotted(ai->address);

  execl(mainconf.telnet, "telnet", "-8", "-E", host, NULL);
  nsyslog(LOG_ERR, "%s: %m", mainconf.telnet);
  return -1;
}


/*
 *	Execute a PPP session.
 */
static int do_ppp(int port, struct auth *ai)
{
  char *s, *o;

  if (ai->conn_info[0]) setenv("CONNECT_INFO", ai->conn_info, 1);

  s = ai->proto == 'A' ? lineconf[port].autoppp : lineconf[port].pppopt;
  o = percent(port, ai, s);
  doexec(mainconf.pppd, "pppd", o);
  if (o) free(o);
  nsyslog(LOG_ERR, "%s: %m", mainconf.pppd);
  return -1;
}

/*
 *	Execute a local login.
 */
static int do_local(int port, struct auth *ai, char *id, pid_t parent)
{
  struct utmp ut, *u;
  char *tty;

  /*
   *	Prevent hacking.
   */
  if (ai->login[0] == '-') return -1;

  /*
   *	If we forked we have to pull some tricks to
   *	become process group leader etc.
   */
  if (parent) {
	setsid();
	ioctl(0, TIOCSCTTY, (char *)1);
  }

  /*
   *	First fill out an utmp struct.
   */
  tty = ttyname(0);
  if (tty[0] == '/') tty += 5;
  memset(&ut, 0, sizeof(ut));
  ut.ut_type = LOGIN_PROCESS;
  ut.ut_pid = getpid();
  ut.ut_time = time(NULL);
  strncpy(ut.ut_line, tty, sizeof(ut.ut_line));
  strncpy(ut.ut_user, ai->login, sizeof(ut.ut_user));
  strncpy(ut.ut_id, id, sizeof(ut.ut_id));

  /*
   *	Now try to find the existing struct..
   */
  if (parent) ut.ut_pid = parent;
  if ((u = getutpid(&ut)) != NULL)
	strncpy(ut.ut_id, u->ut_id, sizeof(ut.ut_id));
  if (parent) ut.ut_pid = getpid();
  setutent();
  pututline(&ut);

  /* XXXX - ugly to use external stty */
  system("exec stty sane");

  /*
   *	If we are forked it also means we are already authenticated -
   *	so don't ask for a password.
   */
  if (parent)
	execl("/bin/login", "login", "-f", ai->login, NULL);
  else
	execl("/bin/login", "login", ai->login, NULL);

  nsyslog(LOG_ERR, "/bin/login: %m");
  return -1;
}

/*
 *	Spawn an extra process to do the work, if needed.
 */
int spawnit(int port, struct auth *ai)
{
  pid_t pid;
  int st;
  struct sigaction sa;
  char id[8];
  struct utmp ut, *u;

  /*
   *	Make up an id field if needed.
   */
  sprintf(id, "T%d", port);

  /*
   *	We only fork if we are going to exec another program
   *	to do the work and we need to send a RADIUS logout
   *	packets afterwards.
   */
  pid = -1;
  if (ai->proto != P_LOCAL && ai->proto != P_AUTOPPP &&
      ai->proto != P_SLIP  && ai->proto != P_CSLIP) {
	if ((pid = fork()) < 0) {
		nsyslog(LOG_ERR, "fork: %m");
		return -1;
	}

	if (pid > 0) {
		/*
		 *	Catch SIGHUP.
		 */
		sa.sa_handler = got_hup;
		sa.sa_flags   = SA_RESTART;
		sigaction(SIGHUP,  &sa, NULL);
		sigaction(SIGTERM, &sa, NULL);
		sigaction(SIGINT,  &sa, NULL);
		unblock(SIGHUP);
		unblock(SIGTERM);
		unblock(SIGINT);

		/*
		 *	Wait for the child to exit.
		 */
		errno = 0;
		while(waitpid(pid, &st, 0) != pid && errno == EINTR)
			errno = 0;

		/*
		 *	Readjust the utmp entry for P_SHELL.
		 */
		if (ai->proto == P_SHELL) {
			setutent();
			ut.ut_pid = pid;
			if ((u = getutpid(&ut)) != NULL) {
				u->ut_pid = getpid();
				setutent();
				pututline(u);
			}
		}

		return 0;
	}
	/*
	 *	If a signal was pending, it will be delivered now.
	 */
	unblock(SIGHUP);
	unblock(SIGTERM);
	unblock(SIGINT);
  }

  /*
   *	Set the TERM environment variable.
   */
  if (lineconf[port].term && lineconf[port].term[0])
	setenv("TERM", lineconf[port].term, 1);

  /*
   *	Catch sighup so that we can be killed.
   */
  signal(SIGHUP, SIG_DFL);

  /*
   *	Find the protocol.
   */
  switch(ai->proto) {
	case P_TELNET:
		do_telnet(port, ai);
		break;
	case P_TCPCLEAR:
#if 0 /* XXX - unimplemented protocols */
	case P_TCPLOGIN:
		do_tcp(port, ai);
		break;
	case P_CONSOLE:
		do_console(port, ai);
		break;
#endif
	case P_SLIP:
	case P_CSLIP:
		do_slip(port, ai);
		break;
	case P_PPP:
	case P_AUTOPPP:
		do_ppp(port, ai);
		break;
	case P_RLOGIN:
		do_rlogin(port, ai);
		break;
	case P_LOCAL:
	case P_SHELL:
		do_local(port, ai, id, pid == 0 ? getppid() : 0);
		break;
	default:
		printf("protocol `%c' unsupported\n", ai->proto);
		fflush(stdout);
		usleep(500000);
		break;
  }
  if (pid == 0) exit(0);

  return 0;
}

