/*
 * ctlportslave.c	Command line interface to running portslaves.
 *			Basically this is to accomodate the pmmon
 *			program, but it might have some more uses too.
 *
 * Version:		@(#)ctlportslave  1.00  12-Nov-1997  miquels@cistron.nl
 *
 */
#include <sys/types.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <fcntl.h>
#include <utmp.h>
#include <signal.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <net/if.h>
#include <arpa/inet.h>

#include "conf.h"

int do_show(int, char **);
int do_reset(int, char **);
int do_quit(int, char **);
int do_help(int, char **);
int read_utmp(void);


struct utmp lines[MAXLINES];
int max_port_ever = 0;
char *eol = "\n";
int finger_prog = 0;
char eth0_classB[32];

int sockfd;
extern int traffic_stats(struct in_addr ina, int *in, int *out);
extern int idle_stat(char *dev, int *idle);

/*
 *	Header for the 2 formats: sessions and who.
 */
char *hdr1 =
"Port User            Host/Inet/Dest   Type    Dir Status         Start   Idle";
char *sep1 =
"---- --------------- ---------------- ------- --- ------------- ------ ------";
char *idle1 = 
"S%-3d -               -                Log/Net In  IDLE               0      0%s";
char *hdr2 =
"Port User         Host/Inet/Dest  Type    Stat Start        In       Out  Idle";
char *sep2 =
"---- ------------ --------------- ------- ---- ----- --------- --------- -----";
char *idle2 =
"S%-3d -            -               -       IDLE     0         0         0     0%s";

/*
 *	Commands we support.
 */
struct commands {
  char *cmd;
  int (*fun)(int, char **);
} commands[] = {
  { "show",	do_show		},
  { "reset",	do_reset	},
  { "help",	do_help		},
  { "exit",	do_quit		},
  { "quit",	do_quit		},
  { NULL,	NULL		},
};


/*
 *	Return extended type.
 */
char *xtype(char t)
{
	if (t == 'x')  return("Network");
	if (t == 'E')  return("Telnet");
	if (t == 'R')  return("Rlogin");
	if (t == 'L')  return("Local");
	if (t == 'X')  return("Local");
	if (t == 'C')  return("CSLIP");
	if (t == 'S')  return("SLIP");
	if (t == 'P')  return("PPP");
	if (t == 'A')  return("PPP");

	return ("unknown");
}

/*
 *	Return address.
 */
char *address(struct utmp *ut, struct in_addr *ina)
{
	static char buf[32];
	char *p;

	ina->s_addr = 0;

#ifdef __linux__
	if (ut->ut_addr) {
		ina->s_addr = ut->ut_addr;
		return(inet_ntoa(*ina));
	}
#endif

	if ((p = strchr(ut->ut_host, ':')) == NULL || p[1] == 0)
		return("");
	p++;
	if (p[0]) p++;
	sprintf(buf, "%s%s", eth0_classB, p + 1);
	ina->s_addr = inet_addr(buf);

	return buf;
}


/*
 *	Show logged in users. If arg is `sessions', use the
 *	portmaster format. If arg is `who', use our own format.
 */
int do_show(int argc, char **argv)
{
	int n = 0;
	int i;
	char *type, *p;
	char name[64];
	char *addr;
	struct in_addr ina;
	struct utmp *ut;
	time_t now;
	int tm;
	int in, out, idle;

	if (argc > 1 && argv[1][0] == 's') n = 1;
	if (argc > 1 && argv[1][0] == 'w') n = 2;
	if (n == 0) {
		printf("Error: usage: show sessions,who\n");
		return 0;
	}

	read_utmp();
	now = time(NULL);

	if (n == 1)
		printf("%s%s%s%s", hdr1, eol, sep1, eol);
	else
		printf("%s%s%s%s", hdr2, eol, sep2, eol);

	for(i = 0; i <= max_port_ever; i++) {

		ut = lines + i;
		p = strchr(ut->ut_host, ':');
		if (p == NULL || ut->ut_name[0] == 0) {
			if (!finger_prog)
				printf(n == 1 ? idle1 : idle2, i, eol);
			continue;
		}

		strncpy(name, ut->ut_name, sizeof(ut->ut_name));
		name[sizeof(ut->ut_name) - 1] = 0;
		tm = (now - ut->ut_time) / 60;
		p++;

		ina.s_addr = 0;
		if (strchr("xERSCPLXA", *p)) {
			addr = address(ut, &ina);
			type = (n == 1) ? "Network" : xtype(*p);
		} else {
			addr = "";
			type = "Login";
		}

		in = out = idle = 0;
		if (n == 2 && ina.s_addr) {
			traffic_stats(ina, &in, &out);
		}
		idle_stat(ut->ut_line, &idle);
		idle /= 60;

		if (n == 1)
	printf("S%-3d %-15.15s %-16.16s %-7.7s In  ESTABLISHED   %6d %6d%s",
                i, name, addr, type, tm, idle, eol);
		else
	printf("S%-3d %-12.12s %-15.15s %-7.7s ESTA %5d %9d %9d %5d%s",
                i, name, addr, type, tm, in, out, idle, eol);

	}
	return 0;
}

/*
 *	Reset a terminal line (send SIGHUP)
 */
int do_reset(int argc, char **argv)
{
	int port;

	if (argc < 2 || (argv[1][0] != 's' && argv[1][0] != 'S')) {
		printf("Error: Usage: reset Port_Name\n");
		return 0;
	}
	port = atoi(argv[1] + 1);
	printf("Resetting port S%d (line %s, pid %d)\n",
		port, lines[port].ut_line, lines[port].ut_pid);
	if (lines[port].ut_pid > 0)
		kill(lines[port].ut_pid, SIGHUP);
	return 0;
}

/*
 *	Exit from this program.
 */
int do_quit(int argc, char **argv)
{
	return -1;
}

/*
 *	Pretend we're the finger daemon.
 */
int do_finger(void)
{
	char buf[128];
	char *args[3];

	finger_prog = 1;
	eol = "\r\n";
	fgets(buf, sizeof(buf), stdin);
	args[0] = "show";
	args[1] = "sessions";
	args[2] = NULL;
	do_show(2, args);
	return 0;
}


/*
 *	Short help text.
 */
int do_help(int arc, char **argv)
{
	printf("\nCtlportslave help:\n\n");
	printf("  show  <sessions|who>\n");
	printf("  reset Sxx\n");
	printf("  exit\n");
	printf("  quit\n\n");

	return 0;
}


/*
 *	Decipher portslave port from ut_host.
 */
int ps_port(char *data)
{
	if (isdigit(data[0]) && isdigit(data[1]) &&
	    isdigit(data[2]) && data[3] == ':')
		return (atoi(data));
	return -1;
}

/*
 *	Read the portslave UTMP file..
 */
int read_utmp(void)
{
	struct utmp *ut;
	int port;

	setutent();
	memset(lines, 0, sizeof(lines));
	while ((ut = getutent()) != NULL) {
		if ((port = ps_port(ut->ut_host)) < 0 || port > MAXLINES)
			continue;
		if (port > max_port_ever) max_port_ever = port;
		if (ut->ut_type != LOGIN_PROCESS &&
		    ut->ut_type != USER_PROCESS)
			continue;
		lines[port] = *ut;
	}
	endutent();

	return 0;
}


int main(int argc, char **argv)
{
	char hostname[128];
	char buf[64];
	char *args[16];
	char *p;
	int i, n;
	int quit = 0;
	struct ifreq ifr;
	struct sockaddr_in *sin;

	/*
	 *	Find out our ethernet address. We only need
	 *	the first two octets. This is because that's missing
	 *	from the information in the utmp file...
	 */
	if ((sockfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		perror("socket");
		exit(1);
	}
	memset(&ifr, 0, sizeof(ifr));
	strcpy(ifr.ifr_name, "eth0");
	ifr.ifr_addr.sa_family = AF_INET;
	if (ioctl(sockfd, SIOCGIFADDR, &ifr) < 0) {
		perror("SIOCGIFADDR(eth0)");
		exit(1);
	}
	
	sin = (struct sockaddr_in *)&(ifr.ifr_addr);
	strcpy(eth0_classB, inet_ntoa(sin->sin_addr));
	/*printf("ether is at %s\n", eth0_classB);*/
	p = eth0_classB;
	i = 0;
	while(*p && i < 2)
		if (*p++ == '.') i++;
	*p = 0;

	/*
	 *	See if we were called as `fingerd'.
	 */
	if (strstr(argv[0], "finger")) {
		do_finger();
		exit(0);
	}

	/*
	 *	Do some network/socket initialization.
	 */
	gethostname(hostname, sizeof(hostname));

	/*
	 *	Command Loop.
	 */
	while(!quit) {
		printf("%s> ", hostname);
		fflush(stdout);
		if (fgets(buf, sizeof(buf), stdin) == NULL) {
			printf("\n");
			break;
		}
		i = 0;
		p = strtok(buf, " \t\r\n");
		while(p && i < 15) {
			args[i++] = p;
			p = strtok(NULL, " \t\r\n");
		}
		args[i] = NULL;
		if (args[0] == NULL)
			continue;
		for(n = 0; commands[n].cmd; n++) {
			if (strcmp(args[0], commands[n].cmd) == 0) {
				if (commands[n].fun(i, args) < 0)
					quit = 1;
				break;
			}
		}
		if (commands[n].cmd == NULL)
			printf("Invalid command.\n");
	}
	printf("Goodbye....\n");

	return 0;
}

