/*
 * Copyright (c) 2005 Sendmail, Inc. and its suppliers.
 *	All rights reserved.
 *
 * By using this file, you agree to the terms and conditions set
 * forth in the LICENSE file which can be found at the top level of
 * the sendmail distribution.
 */

#include "sm/generic.h"
SM_RCSID("@(#)$Id: qm_rsr.c,v 1.4 2005/04/21 22:09:45 ca Exp $")
#include "sm/error.h"
#include "sm/assert.h"
#include "sm/qmgr.h"
#include "sm/qmgr-int.h"
#include "qmgr.h"
#include "qm_throttle.h"
#include "log.h"

/*
**  QMGR experienced a severe resource problem, e.g., memory or disk space,
**  which prevented an "essential" update (e.g., after a delivery attempt).
**  Deal with it... for now: shut down the system.
**  Question: how to do that? Some task must return EVTHR_TERM to the
**  evthr library.
*/

/*
**  QM_RSR_PROBLEM -- Deal with severe resource problem
**
**	Parameters:
**		qmgr_ctx -- QMGR context
**		resource -- which resource?
**		locktype -- kind of locking
**
**	Returns:
**		usual sm_error code; see qm_control()
**
**	Side Effects: sets qmgr_status
**
**	Locking:
**		locks qmgr_ctx if requested
**
**	Last code review: 2005-04-21 22:01:25
**	Last code change:
*/

sm_ret_T
qm_rsr_problem(qmgr_ctx_P qmgr_ctx, uint resource, thr_lock_T locktype)
{
	sm_ret_T ret;
	int r;

	SM_IS_QMGR_CTX(qmgr_ctx);
	SM_REQUIRE(resource <= QMGR_RFL_LAST_I);
	ret = SM_SUCCESS;

	if (thr_lock_it(locktype))
	{
		r = pthread_mutex_lock(&(qmgr_ctx->qmgr_mutex));
		SM_LOCK_OK(r);
		if (r != 0)
		{
			sm_log_write(qmgr_ctx->qmgr_lctx,
				QM_LCAT_CONTROL, QM_LMOD_CONTROL,
				SM_LOG_CRIT, 4,
				"sev=CRIT, func=qm_rsr, lock=%d\n", r);
			return sm_error_temp(SM_EM_Q, r);
		}
	}

	ret = qm_control(qmgr_ctx, 1, 100, resource, THR_NO_LOCK);
	qmgr_ctx->qmgr_status = QMGR_ST_RSR;

	if (thr_unl_always(locktype))
	{
		r = pthread_mutex_unlock(&(qmgr_ctx->qmgr_mutex));
		SM_ASSERT(r == 0);
		if (r != 0 && sm_is_success(ret))
			ret = sm_error_perm(SM_EM_Q, r);
	}
	return ret;
}
