/*
**  SA - OSSP Socket Abstraction Library
**  Copyright (c) 2001 Ralf S. Engelschall <rse@engelschall.com>
**  Copyright (c) 2001 The OSSP Project <http://www.ossp.org/>
**  Copyright (c) 2001 Cable & Wireless Deutschland <http://www.cw.com/de/>
**
**  This file is part of OSSP SA, a socket abstraction library which
**  can be found at http://www.ossp.org/pkg/sa/.
**
**  Permission to use, copy, modify, and distribute this software for
**  any purpose with or without fee is hereby granted, provided that
**  the above copyright notice and this permission notice appear in all
**  copies.
**
**  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
**  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
**  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
**  IN NO EVENT SHALL THE AUTHORS AND COPYRIGHT HOLDERS AND THEIR
**  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
**  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
**  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
**  USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
**  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
**  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
**  OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
**  SUCH DAMAGE.
**
**  sa_test.c: socket abstraction library test suite
*/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <time.h>
#include <sys/utsname.h>
#include <unistd.h>

#include "ts.h"
#include "sa.h"

TST_FUNC(test_saa_impexp)
{
    sa_addr_t *saa;
    sa_rc_t rv;
    char *cp;
    int i;
    struct {
        char *in;
        sa_rc_t rv;
        char *out;
    } table[] = {
        /* positive tests */
        { "inet://0.0.0.0:0", SA_OK, "inet://0.0.0.0:0" }, 
        { "inet://127.0.0.1:514", SA_OK, "inet://127.0.0.1:514" },
        { "inet://localhost:syslog#udp", SA_OK, "inet://127.0.0.1:514" }, 
        { "inet://localhost:smtp", SA_OK, "inet://127.0.0.1:25" },
        { "unix:/tmp/socket", SA_OK, "unix:/tmp/socket" },
        /* negative tests */
        { "inet:", SA_ERR_ARG, NULL },
        { "inet://1.2.3.4.5:0", SA_ERR_ARG, NULL },
        { "inet://just-hostname", SA_ERR_ARG, NULL },
        { "unix:", SA_ERR_ARG, NULL }
    };

    tst_check(TST, "sa_addr_create");
    if ((rv = sa_addr_create(&saa)) != SA_OK)
        tst_fail(TST, "sa_addr_create -> %d[%s] (expected %d[%s])", 
                 rv, sa_error(rv), SA_OK, sa_error(SA_OK));
    for (i = 0; i < sizeof(table)/sizeof(table[0]); i++) {
        tst_check(TST, "sa_addr_u2a(\"%s\")", table[i].in);
        if ((rv = sa_addr_u2a(saa, table[i].in)) != table[i].rv)
            tst_fail(TST, "sa_addr_a2u -> %d[%s] (expected %d[%s])",
                     rv, sa_error(rv), table[i].rv, sa_error(table[i].rv));
        tst_check(TST, "sa_addr_a2u");
        if ((rv = sa_addr_a2u(saa, &cp)) != SA_OK)
            tst_fail(TST, "sa_addr_u2a -> %d[%s] (expected %d[%s])",
                     rv, sa_error(rv), SA_OK, sa_error(SA_OK));
        if (table[i].rv == SA_OK)
            if (strcmp(cp, table[i].out) != 0)
                tst_fail(TST, "sa_addr_a2u -> \"%s\" (expected \"%s\")", cp, table[i].out);
    }
    tst_check(tst, "sa_addr_destroy");
    if ((rv = sa_addr_destroy(saa)) != SA_OK)
        tst_fail(TST, "sa_addr_destroy -> %d[%s] (expected %d[%s])", 
                 rv, sa_error(rv), SA_OK, sa_error(SA_OK));
}

TST_FUNC(test_saa_match)
{
    sa_addr_t *saa1;
    sa_addr_t *saa2;
    sa_rc_t rv;
    int i;
    struct {
        char *in;
        char *acl;
        int prefixlen;
        sa_rc_t rv;
    } table[] = {
        { "unix:/foo/bar", "unix:/foo/bar", -1, SA_OK },
        { "unix:/foo/bar", "unix:/foo/bar", 0, SA_OK },
        { "unix:/foo/bar", "unix:/foo", 4, SA_OK },
        { "unix:/foo/bar", "unix:/foo/quux", 4, SA_OK },
        { "unix:/foo/bar", "unix:/baz/quux", -1, SA_ERR_MTC },
        { "inet://0.0.0.0:0", "inet://0.0.0.0:0", 0, SA_OK },
        { "inet://127.0.0.1:514", "inet://127.0.0.1:514", -1, SA_OK },
        { "inet://127.0.0.1:514", "inet://0.0.0.0:0", 0, SA_OK },
        { "inet://127.0.0.1:514", "inet://12.34.56.78:9", 0, SA_OK },
        { "inet://127.0.0.1:514", "inet://12.34.56.78:9", -1, SA_ERR_MTC },
        { "inet://127.0.0.1:514", "inet://127.0.0.0:0", 24, SA_OK },
        { "inet://127.0.0.1:514", "inet://127.0.0.0:0", 32, SA_ERR_MTC },
        { "inet://141.1.23.20:25", "inet://141.1.23.40:25", 32, SA_ERR_MTC },
        { "inet://141.1.23.20:25", "inet://141.1.23.40:25", 24, SA_OK }
    };

    sa_addr_create(&saa1);
    sa_addr_create(&saa2);
    for (i = 0; i < sizeof(table)/sizeof(table[0]); i++) {
        if ((rv = sa_addr_u2a(saa1, table[i].in)) != SA_OK)
            continue;
        if ((rv = sa_addr_u2a(saa2, table[i].acl)) != SA_OK)
            continue;
        tst_check(TST, "sa_addr_match(\"%s\", \"%s\", %d)", 
                  table[i].in, table[i].acl, table[i].prefixlen);
        if ((rv = sa_addr_match(saa1, saa2, table[i].prefixlen)) != table[i].rv)
            tst_fail(TST, "sa_addr_match -> %d[%s] (expected %d[%s])",
                     rv, sa_error(rv), table[i].rv, sa_error(table[i].rv));
    }
    sa_addr_destroy(saa1);
    sa_addr_destroy(saa2);
}

int main(int argc, char *argv[])
{
    ts_t *ts;
    int n;

    ts = ts_new("OSSP SA (Socket Abstraction Library)");
    ts_test(ts, test_saa_impexp, "socket address abstraction (import/export)");
    ts_test(ts, test_saa_match,  "socket address abstraction (matching)");
    n = ts_run(ts);
    ts_free(ts);
    return n;
}

