/*
**  L2 - OSSP Logging Library
**  Copyright (c) 2001 The OSSP Project (http://www.ossp.org/)
**  Copyright (c) 2001 Cable & Wireless Deutschland (http://www.cw.com/de/)
**
**  This file is part of OSSP L2, a flexible logging library which
**  can be found at http://www.ossp.org/pkg/l2/.
**
**  Permission to use, copy, modify, and distribute this software for
**  any purpose with or without fee is hereby granted, provided that
**  the above copyright notice and this permission notice appear in all
**  copies.
**
**  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
**  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
**  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
**  IN NO EVENT SHALL THE AUTHORS AND COPYRIGHT HOLDERS AND THEIR
**  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
**  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
**  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
**  USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
**  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
**  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
**  OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
**  SUCH DAMAGE.
**
**  l2_test.c: C API test suite
*/

#include <stdio.h>

#include <sys/types.h>   /* following group of includes are for */
#include <sys/socket.h>  /* socket testing only                 */
#include <netinet/in.h>
#include <netdb.h>
#include <syslog.h>

#include "l2.h"

static void die(char *fmt, ...)
{
    va_list ap;

    va_start(ap, fmt);
    fprintf(stderr, "l2_test:ERROR: ");
    vfprintf(stderr, fmt, ap);
    fprintf(stderr, "\n");
    va_end(ap);
    exit(1);
}

static l2_result_t 
formatter(l2_context_t *ctx, const char id, const char *param,
          char *bufptr, size_t bufsize, size_t *buflen, va_list *ap)
{
    int i;
    
    i = va_arg(*ap, int);
    sprintf(bufptr, "[%d/%s]", i, param);
    *buflen = strlen(bufptr);
    return L2_OK;
}

int main(int argc, char *argv[])
{
    l2_channel_t *chPrefix;
    l2_channel_t *chBuf;
    l2_channel_t *chFile;
    l2_channel_t *chSyslog;
#ifdef WITH_SOCKET
    l2_channel_t *chSock;
#endif
#ifdef WITH_SMTP
    l2_channel_t *chSmtp;
#endif
    l2_stream_t *st;

    /*
     * Typical steps to use a buffered file logging stream
     */

    if ((st = l2_stream_create()) == NULL)
        die("failed to create stream");

#ifdef WITH_SOCKET
    /* Atenzione! Before doing any socket testing, make sure you have a valid */
    /* end point listening, or else you will only get an error message when   */
    /* the thing tries to connect.                                            */
    if ((chSock = l2_channel_create(&l2_handler_socket)) == NULL)
        die("failed to create socket channel");

    if (l2_channel_configure(chSock, "protocol,ipversion,host,port",\
        IPPROTO_TCP, AF_INET, "localhost", 2002) != L2_OK)
        die("failed to configure socket ipv4 channel");
#if 0
    if (l2_channel_configure(chSock, "protocol,ipversion,host,port",\
          IPPROTO_TCP, AF_INET6, "0:0:0:0:0:0:0:1", 2002) != L2_OK)
          die("failed to configure socket ipv6 channel");
#endif
    if (l2_channel_open(chSock) != L2_OK)
        die("failed to open socket channel");

    if (l2_stream_channel(st, chSock, L2_LEVEL_UPTO(L2_LEVEL_WARNING)) != L2_OK)
        die("failed to attach first channel into stream");
#endif

    if ((chPrefix  = l2_channel_create(&l2_handler_prefix)) == NULL) /* Prefix */
        die("failed to create prefix channel");

    if (l2_channel_configure(chPrefix, "prefix,timezone", "[%d-%m-%Y/%H:%M:%S] %L test[%P]: ", "local") != L2_OK)
        die("failed to configure prefix channel");

    if ((chBuf  = l2_channel_create(&l2_handler_buffer)) == NULL) /* Buffer */
        die("failed to create buffer channel");

    if (l2_channel_configure(chBuf, "size", 100) != L2_OK)
        die("failed to configure buffer channel");

    if ((chFile = l2_channel_create(&l2_handler_file)) == NULL)   /* File   */
        die("failed to create file channel");

    if (l2_channel_configure(chFile, "path,append,perm", "l2_test.log", TRUE, 0644) != L2_OK)
        die("failed to configure file channel");

    if (l2_channel_stack(chFile, chBuf) != L2_OK)
        die("failed to stack buffer channel on top of file channel");
    
    if (l2_channel_stack(chBuf, chPrefix) != L2_OK)
        die("failed to stack prefix channel on top of buffer channel");

    if (l2_channel_open(chPrefix) != L2_OK)
        die("failed to open channel stack");

    if (l2_stream_channel(st, chPrefix, L2_LEVEL_UPTO(L2_LEVEL_WARNING), L2_LEVEL_NONE) != L2_OK)
        die("failed to attach channel stack into stream");

    if ((chSyslog = l2_channel_create(&l2_handler_syslog)) == NULL)   /* Syslog */
        die("failed to create syslog channel");

    if (l2_channel_configure(chSyslog, "ident,logopts,facility,priority,maskpriority",\
        "Test", (LOG_PID | LOG_CONS), LOG_USER, (LOG_CRIT|LOG_ALERT|LOG_NOTICE),\
        0xFFFFFFFF) != L2_OK)
        die("failed to configure file channel");

    if (l2_channel_open(chSyslog) != L2_OK)
        die("failed to open channel stack");

    if (l2_stream_channel(st, chSyslog, L2_LEVEL_UPTO(L2_LEVEL_WARNING), L2_LEVEL_NONE) != L2_OK)
        die("failed to attach channel syslog into stream");

#ifdef WITH_SMTP
    if ((chSmtp = l2_channel_create(&l2_handler_smtp)) == NULL)   /* SMTP */
        die("failed to create smtp channel");

    if (l2_channel_configure(chSmtp, "rcpt,host,port", "rse@engelschall.com", "en1", "25") != L2_OK)
        die("failed to configure smtp channel");

    if (l2_channel_open(chSmtp) != L2_OK)
        die("failed to open smtp channel");

    if (l2_stream_channel(st, chSmtp, L2_LEVEL_UPTO(L2_LEVEL_ERROR), L2_LEVEL_NONE) != L2_OK)
        die("failed to attach smtp channel into stream");
#endif

    if (l2_stream_levels(st, L2_LEVEL_UPTO(L2_LEVEL_WARNING), L2_LEVEL_NONE) != L2_OK)
        die("failed to set global logging level");
    
    if (l2_stream_formatter(st, 'k', formatter, NULL) != L2_OK)
        die("failed to configure formatter for %%x");

    if (l2_stream_formatter(st, 'S', l2_util_fmt_dump, NULL) != L2_OK)
        die("failed to configure formatter for %%S");

    if (l2_stream_log(st, L2_LEVEL_PANIC, "Checking localhost %s %{myparm}k <%{text}S><%{hex}S><%{base64}S>\n", "foo", 12345, "foo\1bar", 7, "foo\1bar", 7, "foo\1bar", 7) != L2_OK)
        die("failed to log message to stream");

    if (l2_stream_destroy(st) != L2_OK)
        die("failed to destroy stream");

    return 0;
}

