/*
**  L2 - OSSP Logging Library
**  Copyright (c) 2001 The OSSP Project (http://www.ossp.org/)
**  Copyright (c) 2001 Cable & Wireless Deutschland (http://www.cw.com/de/)
**
**  This file is part of OSSP L2, a flexible logging library which
**  can be found at http://www.ossp.org/pkg/l2/.
**
**  Permission to use, copy, modify, and distribute this software for
**  any purpose with or without fee is hereby granted, provided that
**  the above copyright notice and this permission notice appear in all
**  copies.
**
**  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
**  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
**  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
**  IN NO EVENT SHALL THE AUTHORS AND COPYRIGHT HOLDERS AND THEIR
**  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
**  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
**  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
**  USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
**  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
**  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
**  OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
**  SUCH DAMAGE.
**
**  l2_ch_syslog.c: syslog(3) channel implementation
*/

#include "l2.h"

#include <syslog.h>

/* declare private channel configuration */
typedef struct {
    char *pszIdent; /* String to prepend to each syslog message */
    int iLogopt;    /* Manipulates how kernel interacts with syslogd */
    int iFacility;  /* Which part of the system generates this message */
    int iPriority;  /* EMERG, ALERT, CRIT, ERR, WARNING, NOTICE, INFO... */
    int iMaskpri;   /* Controls which priorities are or are not logged */
} l2_ch_syslog_t;

/* create channel */
static l2_result_t hook_create(l2_context_t *ctx, l2_channel_t *ch)
{
    l2_ch_syslog_t *cfg;

    /* allocate private channel configuration */
    if ((cfg = (l2_ch_syslog_t *)malloc(sizeof(l2_ch_syslog_t))) == NULL)
        return L2_ERR_MEM;

    /* initialize configuration with reasonable defaults */
    cfg->pszIdent  = NULL;
    cfg->iLogopt   = 0;
    cfg->iFacility = LOG_USER;
    cfg->iPriority = (LOG_DEBUG|LOG_ERR);
    cfg->iMaskpri  = (0xFFFFFFFF); /* Allow all priorities to pass through */

    /* link private channel configuration into channel context */
    ctx->vp = cfg;

    return L2_OK;
}

/* configure channel */
static l2_result_t hook_configure(l2_context_t *ctx, l2_channel_t *ch, const char *fmt, va_list ap)
{
    l2_ch_syslog_t *cfg = (l2_ch_syslog_t *)ctx->vp;
    l2_param_t pa[6];
    l2_result_t rv;

    /* feed and call generic parameter parsing engine */
    L2_PARAM_SET(pa[0], ident,   CHARPTR,  &cfg->pszIdent);
    L2_PARAM_SET(pa[1], logopts, INT,      &cfg->iLogopt);
    L2_PARAM_SET(pa[2], facility, INT,     &cfg->iFacility);
    L2_PARAM_SET(pa[3], priority, INT,     &cfg->iPriority);
    L2_PARAM_SET(pa[4], maskpriority, INT, &cfg->iMaskpri);
    L2_PARAM_END(pa[5]);
    rv = l2_util_setparams(pa, fmt, ap);

    return rv;
}

/* open channel */
static l2_result_t hook_open(l2_context_t *ctx, l2_channel_t *ch)
{
    l2_ch_syslog_t *cfg = (l2_ch_syslog_t *)ctx->vp;

    /* open channel syslog */
    openlog(cfg->pszIdent, cfg->iLogopt, cfg->iFacility);
    setlogmask(cfg->iMaskpri);

    return L2_OK;
}

/* write to channel */
static l2_result_t hook_write(l2_context_t *ctx, l2_channel_t *ch,
                              l2_level_t level, const char *buf, size_t buf_size)
{
    l2_ch_syslog_t *cfg = (l2_ch_syslog_t *)ctx->vp;

    /* write message to channel syslog */
    syslog(cfg->iPriority, buf);

    return L2_OK;
}

/* close channel */
static l2_result_t hook_close(l2_context_t *ctx, l2_channel_t *ch)
{
    /* close channel syslog */
    closelog();

    return L2_OK;
}

/* destroy channel */
static l2_result_t hook_destroy(l2_context_t *ctx, l2_channel_t *ch)
{
    l2_ch_syslog_t *cfg = (l2_ch_syslog_t *)ctx->vp;

    /* destroy channel configuration */
    free(cfg);

    return L2_OK;
}

/* exported channel handler structure */
l2_handler_t l2_handler_syslog = {
    L2_CHANNEL_OUTPUT,
    hook_create,
    hook_configure,
    hook_open,
    hook_write,
    NULL,
    hook_close,
    hook_destroy
};

