//
//  OSSP asgui - Accounting system graphical user interface
//  Copyright (c) 2002-2003 The OSSP Project (http://www.ossp.org/)
//  Copyright (c) 2002-2003 Cable & Wireless Deutschland (http://www.cw.com/de/)
//  Copyright (c) 2002-2003 Ralf S. Engelschall <rse@engelschall.com>
//  Copyright (c) 2002-2003 Michael Schloh von Bennewitz <michael@schloh.com>
//
//  This file is part of OSSP asgui, an accounting system graphical user
//  interface which can be found at http://www.ossp.org/pkg/tool/asgui/.
//
//  Permission to use, copy, modify, and distribute this software for
//  any purpose with or without fee is hereby granted, provided that
//  the above copyright notice and this permission notice appear in all
//  copies.
//
//  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
//  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
//  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
//  IN NO EVENT SHALL THE AUTHORS AND COPYRIGHT HOLDERS AND THEIR
//  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
//  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
//  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
//  USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
//  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
//  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
//  OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
//  SUCH DAMAGE.
//
//  titslot.cpp: ISO C++ implementation
//

// Qt headers
#include <qfiledialog.h>
#include <qcombobox.h>
#include <qclipboard.h>
#include <qmenudata.h>
//#include <qregexp.h>

// User interface
#include "as_gui.h"             // Main classes
#include "as_except.h"          // Exception classes
#include "as_tableitem.h"       // For our custom table items
#include "as_generic.h"         // Generic classes
#include "as_uuid.h"            // UUID classes

// RPC headers
#ifdef HAVE_ESOAP
#include <easysoap/SOAP.h>
#endif // HAVE_ESOAP
#ifdef HAVE_MICO
#include <coss/CosNaming.h>
#include "asdb.h"               // CORBA stubs and skeletons
#endif // HAVE_MICO

// Icon pixel maps
#include "as_gfx/cwlogo.xpm"    // static const char *s_kpcCwlogo_xpm[]
#include "as_gfx/ossplogo.xpm"  // static const char *s_kpcOssplogo_xpm[]
#include "as_gfx/statok.xpm"    // static const char *s_kpcStatokay_xpm[]
#include "as_gfx/staterr.xpm"   // static const char *s_kpcStaterror_xpm[]
#include "as_gfx/statwrn.xpm"   // static const char *s_kpcStatwarn_xpm[]
#include "as_gfx/statvoid.xpm"  // static const char *s_kpcStatvoid_xpm[]


//
// Cut an entry
//
void Titraqform::cutEntry(void)
{
    this->copyEntry();  // Reuse slot
    this->delEntry();   // Reuse slot
}

//
// Copy an entry
//
void Titraqform::copyEntry(void)
{
    QString Selection;  // Will hold the selected text
    QClipboard *pClip;  // Will reference the global clipboard

    // Initialize data and clipboard handle
    Selection = getRowdata();   // Use accessor
    pClip = QApplication::clipboard();

    Q_ASSERT(!Selection.isNull());
    pClip->setText(Selection, QClipboard::Selection); // Doesn't work on Windows
    pClip->setText(Selection, QClipboard::Clipboard); // Works on both equally
}

//
// Paste an entry
//
void Titraqform::pasteEntry(void)
{
    int nRows = 0;              // Paste so many rows as are stored
    QString Selection;          // Will receive the clipboard text
    QClipboard *pClip;          // Will reference the global clipboard

    if (Selection = pClip->text(QClipboard::Clipboard)) { // Windows and Unix
        nRows = Selection.contains(QChar('\n'));    // How many rows
        this->addEntry(nRows);                      // Reuse slot
        setRowdata(Selection);                      // Use accessor

        // Update line numbers for this new row and all subsequent rows
        for (int nIter = m_pMaintable->currentRow(); nIter < m_pMaintable->numRows(); nIter++)
            m_pMaintable->setText(nIter, TITRAQ_IDXLINE, QString::number(nIter).rightJustify(4, QChar('0')));

        // Do basic data validation to warn against missing fields
        for (int nIter = 0; nIter < nRows; nIter++)
            this->validateData(m_pMaintable->currentRow() + nIter, 0);

        updEdit(m_pMaintable->currentRow()); // Reflect in the update controls
    }
}

//
// Append a blank row entry
//
void Titraqform::addEntry(int nRows)
{
    QTableSelection Select; // Highlighted text
    int nTotal = 0;         // Total row select
    int nCurrent = 0;       // Current row
    std::auto_ptr<AS::Uuid> pGuid(new AS::Uuid); // For GUID production

    // Decide how many rows to add
    Select = m_pMaintable->selection(0);
    if (nRows > 0)
        nTotal = nRows;
    else
        nTotal = Select.bottomRow() - Select.topRow() + 1;

    // Optimize viewing by repainting cells only once after processing
    m_pMaintable->setUpdatesEnabled(false);

    // Add a row after selection and focus to the new row
    if (Select.bottomRow() + 1 != m_pMaintable->numRows()) { // Add upwards
        m_pMaintable->insertRows(Select.topRow(), nTotal);
        m_pMaintable->setDirty();   // Set data to dirty state
        m_pMaintable->setCurrentCell(Select.topRow(), m_pMaintable->currentColumn());

        // Update relevant data fields for all new rows
        for (int nIter = 0; nIter < nTotal; nIter++) {
            m_pMaintable->setText(Select.topRow() + nIter, TITRAQ_IDXSTATUS, QString(QChar('W')));
            m_pMaintable->setPixmap(Select.topRow() + nIter, TITRAQ_IDXSTATUS, QPixmap(s_kpcStatwarn_xpm));
            m_pMaintable->setText(Select.topRow() + nIter, TITRAQ_IDXUSER, m_pPrefs->getString(TITRAQ_PREFUSER, TITRAQ_DEFUSER));
            pGuid->genId();
            m_pMaintable->setText(Select.topRow() + nIter, TITRAQ_IDXGUID, pGuid->getString());
            m_pMaintable->setText(Select.topRow() + nIter, TITRAQ_IDXCRC, "0"); // 0 = invalid entry
            m_pMaintable->setText(Select.topRow() + nIter, TITRAQ_IDXREV, "0"); // Entry not revised
            m_pMaintable->setText(Select.topRow() + nIter, TITRAQ_IDXDATE, QDate::currentDate().toString("yyyy.MM.dd"));
            m_pMaintable->setText(Select.topRow() + nIter, TITRAQ_IDXSTART, "00:00");
            m_pMaintable->setText(Select.topRow() + nIter, TITRAQ_IDXFINISH, "00:00");
            m_pMaintable->setText(Select.topRow() + nIter, TITRAQ_IDXAMOUNT, "00:00");
//            m_pMaintable->setText(Select.topRow() + nIter, TITRAQ_IDXTASK, "/");
//            m_pMaintable->setText(Select.topRow() + nIter, TITRAQ_IDXREMARK, "Remark");
            m_pMaintable->setText(Select.topRow() + nIter, TITRAQ_IDXCRC, "1"); // 0 = invalid entry
            this->calcCrc(Select.topRow() + nIter, -1);
        }
    }
    else { // Special case on last row add downwards
        m_pMaintable->insertRows(Select.bottomRow() + 1, nTotal);
        m_pMaintable->setDirty();   // Set data to dirty state
        m_pMaintable->setCurrentCell(Select.bottomRow() + 1, m_pMaintable->currentColumn());
        m_pMaintable->ensureCellVisible(m_pMaintable->numRows() - 1, 0); // Scroll please

        // Update relevant data fields for all new rows
        for (int nIter = 1; nIter <= nTotal; nIter++) {
            m_pMaintable->setText(Select.bottomRow() + nIter, TITRAQ_IDXSTATUS, QString(QChar('W')));
            m_pMaintable->setPixmap(Select.bottomRow() + nIter, TITRAQ_IDXSTATUS, QPixmap(s_kpcStatwarn_xpm));
            m_pMaintable->setText(Select.bottomRow() + nIter, TITRAQ_IDXUSER, m_pPrefs->getString(TITRAQ_PREFUSER, TITRAQ_DEFUSER));
            pGuid->genId();
            m_pMaintable->setText(Select.bottomRow() + nIter, TITRAQ_IDXGUID, pGuid->getString());
            m_pMaintable->setText(Select.bottomRow() + nIter, TITRAQ_IDXCRC, "0");
            m_pMaintable->setText(Select.bottomRow() + nIter, TITRAQ_IDXREV, "0");
            m_pMaintable->setText(Select.bottomRow() + nIter, TITRAQ_IDXDATE, QDate::currentDate().toString("yyyy.MM.dd"));
            m_pMaintable->setText(Select.bottomRow() + nIter, TITRAQ_IDXSTART, "00:00");
            m_pMaintable->setText(Select.bottomRow() + nIter, TITRAQ_IDXFINISH, "00:00");
            m_pMaintable->setText(Select.bottomRow() + nIter, TITRAQ_IDXAMOUNT, "00:00");
//            m_pMaintable->setText(Select.bottomRow() + nIter, TITRAQ_IDXTASK, "/");
//            m_pMaintable->setText(Select.bottomRow() + nIter, TITRAQ_IDXREMARK, "Remark");
            this->calcCrc(Select.bottomRow() + nIter, -1);
        }
    }

    // Update line numbers for this new row and all subsequent rows
    for (int nIter = m_pMaintable->currentRow(); nIter < m_pMaintable->numRows(); nIter++)
        m_pMaintable->setText(nIter, TITRAQ_IDXLINE, QString::number(nIter).rightJustify(4, QChar('0')));

    updEdit(m_pMaintable->currentRow()); // Reflect in the update controls
    m_pStatusedit->setPixmap(QPixmap(s_kpcStatwarn_xpm)); // Show pixmap

    m_pMaintable->setUpdatesEnabled(true);  // Turn updates back on
    m_pMaintable->repaintContents(true);    // Do a general repaint of table
}

//
// Delete a row entry
//
void Titraqform::delEntry(int nRows)
{
    QTableSelection Select = m_pMaintable->selection(0);    // Highlighted text
    int nTotal = Select.bottomRow() - Select.topRow() + 1;  // Total row select
    QMemArray<int> Rowselect(nTotal);                       // Row array

    // Calculate rows to delete from selection highlight
    for (int nIter = 0; nIter < nTotal; ++nIter)
        Rowselect[nIter] = Select.topRow() + nIter;

    // Remove the row at selection and focus to the next row
    if (m_pMaintable->currentRow() + 1 != m_pMaintable->numRows()) {
        m_pMaintable->setCurrentCell(Select.bottomRow() + 1, m_pMaintable->currentColumn());
        m_pMaintable->removeRows(Rowselect);
        m_pMaintable->setDirty();   // Set data to dirty state
    }
    else {  // Special case to handle removing of only row or last row
        m_pMaintable->removeRows(Rowselect);
        m_pMaintable->setDirty();   // Set data to dirty state
    }

    // Update line numbers for this new row and all subsequent rows
    for (int nIter = m_pMaintable->currentRow(); nIter < m_pMaintable->numRows(); nIter++)
        m_pMaintable->setText(nIter, TITRAQ_IDXLINE, QString::number(nIter));
}

//
// Refresh the display of all data items
//
void Titraqform::refreshDisplay(void)
{
    int nIter = 0;                          // Matrix iterator
    int nRows = m_pMaintable->numRows();    // Total number of rows

    // Sweep through matrix validating linewise
    // data and updating line numbers for all rows
    while (nIter < nRows) {
        this->validateData(nIter, 0);
        m_pMaintable->setText(nIter, TITRAQ_IDXLINE, QString::number(nIter).rightJustify(4, QChar('0')));
        nIter++;
    }

    m_pMaintable->repaintContents(false);   // Do a general repaint of table
    m_pStatbar->message(trUtf8("Display was refreshed"), 4000); // Announce result
}

//
// Make and display a new document window
//
void Titraqform::newDoc(void)
{
    int nResult = 0;

    // Check modification state of current data
    if (m_pMaintable->isDirty()) {
        nResult = QMessageBox::information(this, trUtf8(TITRAQ_APPTITLE),
                  trUtf8(TITRAQ_SAVEFIRST), trUtf8("&Save"),
                  trUtf8("&Discard"), trUtf8("Cancel"), 0, 2);

        switch (nResult) {
        case 0: // First button selected, so save first
            this->saveFile();               // Save changes first
            break;
        case 1: // Second button selected, so don't save
            break;
        case 2: // Third button selected, so return sofort
        default:
            return;
            break;
        }
    }
    // Fall through to implicit new doc code
    this->enableIface(true);            // Enable the interface
    m_pMaintable->setNumRows(0);        // Remove all data in table
    this->setCaption(trUtf8("No file name"));
    m_pStatbar->message(trUtf8("New document"), 4000);
    this->setFilename("");
    this->updEdit(0);                   // Update edit controls
}

//
// Open and display an existing document
//
void Titraqform::openDoc(void)
{
    int nResult = 0;    // Holds return value from save first messagebox

    // Make sure we correctly get the name of the default file to open
    QString Openas = m_pPrefs->getString(TITRAQ_PREFASFILE, TITRAQ_DEFASFILE);
    if (Openas.startsWith(TITRAQ_HOMEDIRTOK))
        Openas = QDir::homeDirPath() + Openas.remove(0, QString(TITRAQ_HOMEDIRTOK).length() - 1);

    QString Filestring = QFileDialog::getOpenFileName(Openas, trUtf8("Accounting Data (*.as);;Text files (*.txt);;All Files (*)"), this, trUtf8("ChooserDialog"), trUtf8("Choose a file to open"), NULL, false);

    if (!Filestring.isEmpty()) {
        QFile Filetemp(Filestring); // File to load
        // Check modification state of current data
        if (m_pMaintable->isDirty()) {
            nResult = QMessageBox::information(this, trUtf8(TITRAQ_APPTITLE),
                      trUtf8(TITRAQ_SAVEFIRST), trUtf8("&Save"),
                      trUtf8("&Discard"), trUtf8("Cancel"), 0, 2);

            switch (nResult) {
            case 0: // Save first
                this->saveFile();               // Save changes first
                setFilename(Filestring);        // Set the new file name
                m_pMaintable->setNumRows(0);    // Clear out old data
                try {
                    loadData(Filetemp);         // Pass to helper method
                }
                catch (Genexcept& Genex) {
                    Genex.reportErr();
                }
                break;
            case 1: // Don't save first but do load
                setFilename(Filestring);        // Set the new file name
                m_pMaintable->setNumRows(0);    // Clear out old data
                try {
                    loadData(Filetemp);         // Pass to helper method
                }
                catch (Genexcept& Genex) {
                    Genex.reportErr();
                }
                break;
            case 2: // Don't do a load timesheet
            default:
                break;
            }
        }
        else {
            setFilename(Filestring);            // Set the new file name
            m_pMaintable->setNumRows(0);        // Clear out old data
            try {
                loadData(Filetemp);             // Pass to helper method
            }
            catch (Genexcept& Genex) {
                Genex.reportErr();
            }
        }

        // Fall through to implicit open code
        this->setCaption(Filestring);
        m_pStatbar->message(trUtf8("Loaded document ") + Filestring, 4000);
        this->enableIface(true);    // Turn on the lights
    }
    else
        m_pStatbar->message(trUtf8("Loading aborted"), 4000);
}

//
// Serialize current state to the current file
//
void Titraqform::saveFile(void)
{
    QString Fname;
    try {
        Fname = QString(*this->getFilename());
        // First time saves are really just saveAs in disguise
        if (Fname.isEmpty()) {
            this->saveAs();
            return;
        }
        // Try to open a file for writing to
        QFile Filetemp(Fname);
        this->saveData(Filetemp); // Pass to helper method
    }
    catch (Genexcept& Genex) {
        Genex.reportErr();
    }
    // Reset and give output to main window
    this->setCaption(Fname);
    m_pStatbar->message(trUtf8("File %1 saved").arg(Fname), 4000);
    m_pMaintable->setDirty(false);  // Set the clean state to allow close
}

//
// Serialize current state to a selected file
//
void Titraqform::saveAs(void)
{
    int nResult = 0; // For checking user's answer

    // Make sure we correctly get the name of the default file to open
    QString Openas = m_pPrefs->getString(TITRAQ_PREFASFILE, TITRAQ_DEFASFILE);
    if (Openas.startsWith(TITRAQ_HOMEDIRTOK))
        Openas = QDir::homeDirPath() + Openas.remove(0, QString(TITRAQ_HOMEDIRTOK).length() - 1);

    // And then get the name of the selected file to save to
    QString Filestring = QFileDialog::getSaveFileName(Openas, trUtf8("Accounting Data (*.as);;Text files (*.txt);;All Files (*)"), this, trUtf8("ChooserDialog"), trUtf8("Choose a file to save"), NULL, false);
    if (!Filestring.isEmpty()) {
        if (QFile::exists(Filestring)) {
            nResult = QMessageBox::warning(this, trUtf8(TITRAQ_APPTITLE),
                      trUtf8(TITRAQ_OVERWRITE), trUtf8("&Yes"), trUtf8("&No"), NULL, 1, 1);
            switch (nResult) {
            case 0: // Overwrite contents
                this->setFilename(Filestring);
                this->saveFile();
                break;
            case 1: // Don't overwrite
            default:
                break;
            }
        }
        else
            this->setFilename(Filestring);  // Set filename of object first
            this->saveFile();               // Finish by calling the save action
    }
    else {
        // User did not select a valid file and push okay button
        m_pStatbar->message(trUtf8("Saving aborted"), 4000);
    }
}

//
// Close current document, displaying in main window
//
void Titraqform::closeEvent(QCloseEvent *pClosit)
{
    int nResult = 0;

    if (!this->isOpen())    // Short circuit if user
        qApp->quit();       // selects close twice

    // Check modification state of current data
    if (m_pMaintable->isDirty()) {
        nResult = QMessageBox::information(this, trUtf8(TITRAQ_APPTITLE),
                  trUtf8(TITRAQ_SAVEFIRST), trUtf8("&Save"),
                  trUtf8("&Discard"), trUtf8("Cancel"), 0, 2);

        switch (nResult) {      // Maybe save before closing
        case 0: // Save first
            this->saveFile();   // Save changes first
            break;
        case 1: // Don't save first
            break;
        case 2: // Do nothing
        default:
            pClosit->ignore();
            return; // Go away without closing
            break;
        }
    }

    // Fall through to implicit close code
    this->enableIface(false);   // Turn off the lights
    this->setOpen(false);       // Set doc state to closed
    pClosit->ignore();          // Finish off by not closing
}

//
// Edit menu select all entries
//
void Titraqform::selAll(void)
{
    Prototype Unimp;
    Unimp.doMbox();
}

//
// Edit a table entry in place, without the usual edit controls
//
void Titraqform::inplaceEdit(int nRow, int nCol, int nButton, const QPoint &Mousepos)
{
    m_pMaintable->setReadOnly(false);
    m_pMaintable->editCell(nRow, nCol);
    m_pMaintable->setEdition(nCol);
    m_pMaintable->setReadOnly(true);
}

//
// Update the edit controls widget sizes
//
void Titraqform::updSizes(int nSection, int nOldsize, int nNewsize)
{
    switch (nSection) {
    case TITRAQ_IDXALLCTRLS:
        if (m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXSTATUS) > 0)
            m_pStatusedit->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXSTATUS) - TITRAQ_SPACING);
        if (m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXLINE) > 0)
            m_pLineedit->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXLINE) - TITRAQ_SPACING);
        if (m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXUSER) > 0)
            m_pUseredit->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXUSER) - TITRAQ_SPACING);
        if (m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXGUID) > 0)
            m_pGuidedit->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXGUID) - TITRAQ_SPACING);
        if (m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXCRC) > 0)
            m_pCrcedit->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXCRC) - TITRAQ_SPACING);
        if (m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXREV) > 0)
            m_pRevedit->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXREV) - TITRAQ_SPACING);
        if (m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXDATE) > 0)
            m_pDateedit->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXDATE) - TITRAQ_SPACING);
        if (m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXSTART) > 0)
            m_pStarttime->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXSTART) - TITRAQ_SPACING);
        if (m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXFINISH) > 0)
            m_pEndtime->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXFINISH) - TITRAQ_SPACING);
        if (m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXAMOUNT) > 0)
            m_pAmount->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXAMOUNT) - TITRAQ_SPACING);
        if (m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXTASK) > 0)
            m_pTasks->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXTASK) - TITRAQ_SPACING);
//        if (m_pColspopup->isItemChecked(m_pColspopup->idAt(TITRAQ_IDXRCOL)))
//            m_pRemark->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXREMARK) - TITRAQ_SPACING);
        break;
    case TITRAQ_IDXSTATUS:
        m_pStatusedit->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXSTATUS) - TITRAQ_SPACING);
        break;
    case TITRAQ_IDXLINE:
        m_pLineedit->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXLINE) - TITRAQ_SPACING);
        break;
    case TITRAQ_IDXUSER:
        m_pUseredit->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXUSER) - TITRAQ_SPACING);
        break;
    case TITRAQ_IDXGUID:
        m_pGuidedit->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXGUID) - TITRAQ_SPACING);
        break;
    case TITRAQ_IDXCRC:
        m_pCrcedit->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXCRC) - TITRAQ_SPACING);
        break;
    case TITRAQ_IDXREV:
        m_pRevedit->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXREV) - TITRAQ_SPACING);
        break;
    case TITRAQ_IDXDATE:
        m_pDateedit->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXDATE) - TITRAQ_SPACING);
        break;
    case TITRAQ_IDXSTART:
        m_pStarttime->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXSTART) - TITRAQ_SPACING);
        break;
    case TITRAQ_IDXFINISH:
        m_pEndtime->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXFINISH) - TITRAQ_SPACING);
        break;
    case TITRAQ_IDXAMOUNT:
        m_pAmount->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXAMOUNT) - TITRAQ_SPACING);
        break;
    case TITRAQ_IDXTASK:
        m_pTasks->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXTASK) - TITRAQ_SPACING);
        break;
    case TITRAQ_IDXREMARK:
//        m_pRemark->setFixedWidth(nNewsize);
        break;
    default:
        throw Genexcept("Unrecognized main window column header.");
        break;
    }

    // As a last and redundant step, adjust size of first visible control
    switch(this->getFirstcol()) {
    case TITRAQ_IDXSTATUS:
        m_pStatusedit->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXSTATUS) - TITRAQ_SPACING + TITRAQ_SPACING / 2);
        break;
    case TITRAQ_IDXLINE:
        m_pLineedit->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXLINE) - TITRAQ_SPACING + TITRAQ_SPACING / 2);
        break;
    case TITRAQ_IDXUSER:
        m_pUseredit->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXUSER) - TITRAQ_SPACING + TITRAQ_SPACING / 2);
        break;
    case TITRAQ_IDXGUID:
        m_pGuidedit->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXGUID) - TITRAQ_SPACING + TITRAQ_SPACING / 2);
        break;
    case TITRAQ_IDXCRC:
        m_pCrcedit->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXCRC) - TITRAQ_SPACING + TITRAQ_SPACING / 2);
        break;
    case TITRAQ_IDXREV:
        m_pRevedit->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXREV) - TITRAQ_SPACING + TITRAQ_SPACING / 2);
        break;
    case TITRAQ_IDXDATE:
        m_pDateedit->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXDATE) - TITRAQ_SPACING + TITRAQ_SPACING / 2);
        break;
    case TITRAQ_IDXSTART:
        m_pStarttime->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXSTART) - TITRAQ_SPACING + TITRAQ_SPACING / 2);
        break;
    case TITRAQ_IDXFINISH:
        m_pEndtime->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXFINISH) - TITRAQ_SPACING + TITRAQ_SPACING / 2);
        break;
    case TITRAQ_IDXAMOUNT:
        m_pAmount->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXAMOUNT) - TITRAQ_SPACING + TITRAQ_SPACING / 2);
        break;
    case TITRAQ_IDXTASK:
        m_pTasks->setFixedWidth(m_pMaintable->horizontalHeader()->sectionSize(TITRAQ_IDXTASK) - TITRAQ_SPACING + TITRAQ_SPACING / 2);
        break;
    default:    // Probably no columns are visible?
        break;
    }
}

//
// What to do if a data cell is modified
//
void Titraqform::dataChanged(int nRow, int nCol)
{
    if (nCol != TITRAQ_IDXSTATUS && nCol != TITRAQ_IDXLINE)
        m_pMaintable->setDirty();   // Mark this timesheet dirty, changes pending save
}

//
// Convenience member calculates CRC for current row
//
void Titraqform::calcCrc(void)
{
    this->calcCrc(-1, -1);
}

//
// Calculates CRC for the specified row, fires when data is changed
//
void Titraqform::calcCrc(int nRow, int nCol)
{
    if (nCol != TITRAQ_IDXSTATUS && nCol != TITRAQ_IDXLINE \
                                 && m_pMaintable->numRows() > 0) {
        Q_UINT16 Valcrc, Testcrc;
        QString Wholerow;
        int nRealrow = -1;
//        int nNewrev = -1;

        // Build the data that will be used in testing CRC calculation
        nRealrow = (nRow >= 0) ? nRow : m_pMaintable->currentRow();
        Wholerow = m_pMaintable->text(nRealrow, TITRAQ_IDXUSER);
        for (int nIter = TITRAQ_IDXUSER + 1; nIter < TITRAQ_IDXTAIL; nIter++)
            if (nIter != TITRAQ_IDXCRC) // Don't checksum the checksum!
                Wholerow += ' ' + m_pMaintable->text(nRealrow, nIter);

        // Update the checksum and revision only if necessary
        Testcrc = qChecksum(Wholerow.ascii(), Wholerow.length());
        if (Testcrc != m_pMaintable->text(nRealrow, TITRAQ_IDXCRC).toUInt()) {

//            // Bump the revision number of our entry to control conflicts
//            nNewrev = m_pMaintable->text(nRealrow, TITRAQ_IDXREV).toInt() + 1;
//            m_pMaintable->setText(nRealrow, TITRAQ_IDXREV, QString::number(nNewrev));

            // Build the data that will be used in setting CRC calculation
            Wholerow = m_pMaintable->text(nRealrow, TITRAQ_IDXUSER);
            for (int nIter = TITRAQ_IDXUSER + 1; nIter < TITRAQ_IDXTAIL; nIter++)
                if (nIter != TITRAQ_IDXCRC) // Don't checksum the checksum!
                    Wholerow += ' ' + m_pMaintable->text(nRealrow, nIter);

            // Finally set the checksum to its new value
            Valcrc = qChecksum(Wholerow.ascii(), Wholerow.length());
            m_pMaintable->setText(nRealrow, TITRAQ_IDXCRC, QString::number(Valcrc));
        }
    }
}

//
// Update the edit controls contents
//
void Titraqform::updEdit(int nRow, int nCol)
{
//    QRegExp Shorten("/(\\w+)$");    // For stripping prefix off the current task

    // Field strings to check for validity and process
    QString Textline(m_pMaintable->text(nRow, TITRAQ_IDXLINE));
    QString Textuser(m_pMaintable->text(nRow, TITRAQ_IDXUSER));
    QString Textguid(m_pMaintable->text(nRow, TITRAQ_IDXGUID));
    QString Textcrc(m_pMaintable->text(nRow, TITRAQ_IDXCRC));
    QString Textrev(m_pMaintable->text(nRow, TITRAQ_IDXREV));
    QString Textdate(m_pMaintable->text(nRow, TITRAQ_IDXDATE));
    QString Textstart(m_pMaintable->text(nRow, TITRAQ_IDXSTART));
    QString Textfinish(m_pMaintable->text(nRow, TITRAQ_IDXFINISH));
    QString Textamount(m_pMaintable->text(nRow, TITRAQ_IDXAMOUNT));
    QString Texttask(m_pMaintable->text(nRow, TITRAQ_IDXTASK));
    QString Textremark(m_pMaintable->text(nRow, TITRAQ_IDXREMARK));

    // Reset the edition state member
    m_pMaintable->setEdition();

    // Set text of member edit controls
    if (m_pMaintable->text(nRow, TITRAQ_IDXSTATUS).isEmpty())   // If row is empty
        m_pStatusedit->setPixmap(s_kpcStatvoid_xpm);            // add a placeholder
    else
        m_pStatusedit->setPixmap(m_pMaintable->pixmap(nRow, TITRAQ_IDXSTATUS));
    m_pLineedit->setText(Textline);
    m_pUseredit->setText(Textuser);
    m_pGuidedit->setText(Textguid);
    m_pCrcedit->setText(Textcrc);
    m_pRevedit->setText(Textrev);

    m_pAmount->setText(Textamount);
//    Texttask.remove(0, Shorten.search(Texttask) + 1); // Strip leading slash
    m_pTasks->setCurrentText(Texttask);
    m_pRemark->setText(Textremark);

    // Members not suitable for empty string text
    if (!Textdate.isEmpty())
        m_pDateedit->setDate(QDate::fromString(Textdate, Qt::ISODate));
    else
        m_pDateedit->setDate(*m_pDatezero);
    if (!Textstart.isEmpty())
        m_pStarttime->setTime(QTime::fromString(Textstart, Qt::ISODate));
    else
        m_pStarttime->setTime(QTime::QTime(0, 0));
    if (!Textfinish.isEmpty())
        m_pEndtime->setTime(QTime::fromString(Textfinish, Qt::ISODate));
    else
        m_pEndtime->setTime(QTime::QTime(0, 0));
}

//
// Validate current row of matrix data
//
void Titraqform::validateData(void)
{
    this->validateData(-1, -1);
}

//
// Validate specified row of matrix data
//
void Titraqform::validateData(int nRow, int nCol)
{
    int nRealrow = -1;

    if (!m_pMaintable->numRows() > 0) { // If no rows exist then short circuit
        m_pStatbar->message(trUtf8("Timesheet contains no data"), 4000);
        return;
    }

    nRealrow = (nRow >= 0) ? nRow : m_pMaintable->currentRow();
    QString Statis = m_pMaintable->text(nRealrow, TITRAQ_IDXSTATUS); // Get text

    // Quick review of data validity, and set pixmaps accordingly
    if (m_pMaintable->text(nRealrow, TITRAQ_IDXUSER).isEmpty() ||
        m_pMaintable->text(nRealrow, TITRAQ_IDXGUID).isEmpty() ||
        m_pMaintable->text(nRealrow, TITRAQ_IDXCRC).isEmpty() ||
        m_pMaintable->text(nRealrow, TITRAQ_IDXREV).isEmpty() ||
        m_pMaintable->text(nRealrow, TITRAQ_IDXDATE).isEmpty() ||
        m_pMaintable->text(nRealrow, TITRAQ_IDXSTART).isEmpty() ||
        m_pMaintable->text(nRealrow, TITRAQ_IDXFINISH).isEmpty() ||
        m_pMaintable->text(nRealrow, TITRAQ_IDXAMOUNT).isEmpty() ||
        m_pMaintable->text(nRealrow, TITRAQ_IDXTASK).isEmpty())
    { // No K&R style to show where actual code begins
        if (Statis.startsWith(QString("W"))) { // Conditionally set pixmap to avoid overhead
// FIXME: Next line commented out, and I see that this algorythm needs help
//            m_pStatusedit->setPixmap(m_pMaintable->pixmap(nRealrow, TITRAQ_IDXSTATUS));
        }
        else if (!Statis.startsWith(QString("E"))) { // Conditionally set pixmap to avoid overhead
            m_pMaintable->setText(nRealrow, TITRAQ_IDXSTATUS, Statis.replace(TITRAQ_IDXSTATERROR, sizeof(char), 'E'));
            m_pMaintable->setPixmap(nRealrow, TITRAQ_IDXSTATUS, QPixmap(s_kpcStaterror_xpm));
            m_pStatusedit->setPixmap(m_pMaintable->pixmap(nRealrow, TITRAQ_IDXSTATUS));
        }
    }
    else {
        if (!Statis.startsWith(QString("O"))) { // Conditionally set pixmap to avoid overhead
            m_pMaintable->setText(nRealrow, TITRAQ_IDXSTATUS, Statis.replace(TITRAQ_IDXSTATERROR, sizeof(char), 'O'));
            m_pMaintable->setPixmap(nRealrow, TITRAQ_IDXSTATUS, QPixmap(s_kpcStatokay_xpm));
            m_pStatusedit->setPixmap(m_pMaintable->pixmap(nRealrow, TITRAQ_IDXSTATUS));
        }
    }
}

//
// Update the current line number column item
//
void Titraqform::updateLine(const QString &Instring)
{
    m_pMaintable->setText(m_pMaintable->currentRow(), TITRAQ_IDXLINE, Instring);
}

//
// Update the current user column item
//
void Titraqform::updateUser(const QString &Instring)
{
    m_pMaintable->setText(m_pMaintable->currentRow(), TITRAQ_IDXUSER, Instring);
}

//
// Update the current GUID column item
//
void Titraqform::updateGuid(const QString &Instring)
{
    m_pMaintable->setText(m_pMaintable->currentRow(), TITRAQ_IDXGUID, Instring);
}

//
// Update the current CRC column item
//
void Titraqform::updateCrc(const QString &Instring)
{
    m_pMaintable->setText(m_pMaintable->currentRow(), TITRAQ_IDXCRC, Instring);
}

//
// Update the current rev column item
//
void Titraqform::updateRev(const QString &Instring)
{
    m_pMaintable->setText(m_pMaintable->currentRow(), TITRAQ_IDXREV, Instring);
}

//
// Update the current date column item
//
void Titraqform::updateDate(const QDate &Dateup)
{
    if (Dateup == *m_pDatezero) // Complete the already nasty hack
        m_pMaintable->setText(m_pMaintable->currentRow(), TITRAQ_IDXDATE, trUtf8(TITRAQ_DATEZERO));
    else
        m_pMaintable->setText(m_pMaintable->currentRow(), TITRAQ_IDXDATE, Dateup.toString("yyyy.MM.dd"));
}

//
// Update the current start column item
//
void Titraqform::updateStart(const QTime &Startup)
{
    m_pMaintable->setText(m_pMaintable->currentRow(), TITRAQ_IDXSTART, Startup.toString("hh:mm"));
}

//
// Update the current finish column item
//
void Titraqform::updateFinish(const QTime &Finishup)
{
    m_pMaintable->setText(m_pMaintable->currentRow(), TITRAQ_IDXFINISH, Finishup.toString("hh:mm"));
}

//
// Update the current amount column item
//
void Titraqform::updateAmount(const QString &Amountup)
{
    // Ignore incoming data and use widget managed data because its easier
    m_pMaintable->setText(m_pMaintable->currentRow(), TITRAQ_IDXAMOUNT, m_pAmount->text());
}

//
// Update the current task column item
//
void Titraqform::updateTask(const QString &Taskup)
{
//    // FIXME: Broken
//    RtTableItem *pTask = NULL;
//    pTask = static_cast<RtTableItem *>(m_pMaintable->item(m_pMaintable->currentRow(), TITRAQ_IDXTASK));
//    pTask->setText(Taskup);

    m_pMaintable->setText(m_pMaintable->currentRow(), TITRAQ_IDXTASK, Taskup);
}

//
// Update the current remark column item
//
void Titraqform::updateRemark(const QString &Remarkup)
{
    m_pMaintable->setText(m_pMaintable->currentRow(), TITRAQ_IDXREMARK, Remarkup);
}

//
// Confirm any recent editions on a whole row
//
void Titraqform::confirmEdit(void)
{
    RtTableItem *pTask = NULL;  // Task item is a derived class

    // Conversions from edit control data formats to native tabular format
    m_pMaintable->setText(m_pMaintable->currentRow(), TITRAQ_IDXDATE, m_pDateedit->date().toString(Qt::ISODate));
    m_pMaintable->setText(m_pMaintable->currentRow(), TITRAQ_IDXSTART, m_pStarttime->time().toString(Qt::ISODate));
    m_pMaintable->setText(m_pMaintable->currentRow(), TITRAQ_IDXFINISH, m_pEndtime->time().toString(Qt::ISODate));
    m_pMaintable->setText(m_pMaintable->currentRow(), TITRAQ_IDXAMOUNT, m_pAmount->text());

    // Specially handle task fields
    pTask = static_cast<RtTableItem *>(m_pMaintable->item(m_pMaintable->currentRow(), TITRAQ_IDXTASK));
    pTask->setText(m_pTasks->currentText());

    m_pMaintable->setText(m_pMaintable->currentRow(), TITRAQ_IDXREMARK, m_pRemark->text());
}

//
// Edit menu configure preferences
//
void Titraqform::configPrefs(void)
{
    Prototype Unimp;
    Unimp.doMbox();
}

//
// View menu normal
//
void Titraqform::normalView(void)
{
    // All view types except normal are disabled until implemention, so
    // this body can remain empty, causing nothing to happen on selection.
}

//
// View menu editing
//
void Titraqform::editingView(void)
{
    // All view types except normal are disabled until implemention, so
    // this body can remain empty, causing nothing to happen on selection.
}

//
// View menu timing
//
void Titraqform::timingView(void)
{
    // All view types except normal are disabled until implemention, so
    // this body can remain empty, causing nothing to happen on selection.
}

//
// View menu show file toolbar
//
void Titraqform::showFilebar(void)
{
    if (m_pFiletools->isVisible()) {
        m_pFiletools->hide();
        m_pTbarspopup->setItemChecked(m_pTbarspopup->idAt(TITRAQ_IDXFILEBAR), false);
    }
    else {
        m_pFiletools->show();
        m_pTbarspopup->setItemChecked(m_pTbarspopup->idAt(TITRAQ_IDXFILEBAR), true);
    }
}

//
// View menu show edit toolbar
//
void Titraqform::showEditbar(void)
{
    if (m_pEdittools->isVisible()) {
        m_pEdittools->hide();
        m_pTbarspopup->setItemChecked(m_pTbarspopup->idAt(TITRAQ_IDXEDITBAR), false);
    }
    else {
        m_pEdittools->show();
        m_pTbarspopup->setItemChecked(m_pTbarspopup->idAt(TITRAQ_IDXEDITBAR), true);
    }
}

//
// View menu show view toolbar
//
void Titraqform::showViewbar(void)
{
    if (m_pViewtools->isVisible()) {
        m_pViewtools->hide();
        m_pTbarspopup->setItemChecked(m_pTbarspopup->idAt(TITRAQ_IDXVIEWBAR), false);
    }
    else {
        m_pViewtools->show();
        m_pTbarspopup->setItemChecked(m_pTbarspopup->idAt(TITRAQ_IDXVIEWBAR), true);
    }
}

//
// View menu show whats this toolbar
//
void Titraqform::showWhatsbar(void)
{
    if (m_pWhatstools->isVisible()) {
        m_pWhatstools->hide();
        m_pTbarspopup->setItemChecked(m_pTbarspopup->idAt(TITRAQ_IDXWHATBAR), false);
    }
    else {
        m_pWhatstools->show();
        m_pTbarspopup->setItemChecked(m_pTbarspopup->idAt(TITRAQ_IDXWHATBAR), true);
    }
}

//
// View menu show status column
//
void Titraqform::showStatcol(void)
{
    // Test if column is currently shown, conditionally show or hide it
    if (m_pColspopup->isItemChecked(m_pColspopup->idAt(TITRAQ_IDXSTATCOL))) {
        m_pMaintable->hideColumn(TITRAQ_IDXSTATUS);
        m_pColspopup->setItemChecked(m_pColspopup->idAt(TITRAQ_IDXSTATCOL), false);
        m_pStatusedit->hide();
    }
    else {
        m_pMaintable->showColumn(TITRAQ_IDXSTATUS);
        m_pColspopup->setItemChecked(m_pColspopup->idAt(TITRAQ_IDXSTATCOL), true);
        m_pStatusedit->show();
    }

    // Make sure switch take place right away and sizes are handled
    m_pColspopup->updateItem(m_pColspopup->idAt(TITRAQ_IDXSTATCOL));
    this->updSizes(TITRAQ_IDXALLCTRLS, 0, 0); // Update size of this and next
}

//
// View menu show line numbers column
//
void Titraqform::showLinecol(void)
{
    // Test if column is currently shown, conditionally show or hide it
    if (m_pColspopup->isItemChecked(m_pColspopup->idAt(TITRAQ_IDXLCOL))) {
        m_pMaintable->hideColumn(TITRAQ_IDXLINE);
        m_pColspopup->setItemChecked(m_pColspopup->idAt(TITRAQ_IDXLCOL), false);
        m_pLineedit->hide();
    }
    else {
        m_pMaintable->showColumn(TITRAQ_IDXLINE);
        m_pColspopup->setItemChecked(m_pColspopup->idAt(TITRAQ_IDXLCOL), true);
        m_pLineedit->show();
    }

    // Make sure switch take place right away and sizes are handled
    m_pColspopup->updateItem(m_pColspopup->idAt(TITRAQ_IDXLCOL));
    this->updSizes(TITRAQ_IDXALLCTRLS, 0, 0); // Update size of this and next
}

//
// View menu show users column
//
void Titraqform::showUsercol(void)
{
    // Test if column is currently shown, conditionally show or hide it
    if (m_pColspopup->isItemChecked(m_pColspopup->idAt(TITRAQ_IDXUCOL))) {
        m_pMaintable->hideColumn(TITRAQ_IDXUSER);
        m_pColspopup->setItemChecked(m_pColspopup->idAt(TITRAQ_IDXUCOL), false);
        m_pUseredit->hide();
    }
    else {
        m_pMaintable->showColumn(TITRAQ_IDXUSER);
        m_pColspopup->setItemChecked(m_pColspopup->idAt(TITRAQ_IDXUCOL), true);
        m_pUseredit->show();
    }

    // Make sure switch take place right away and sizes are handled
    this->updSizes(TITRAQ_IDXALLCTRLS, 0, 0); // Update size of this and next
    m_pColspopup->updateItem(m_pColspopup->idAt(TITRAQ_IDXUCOL));
}

//
// View menu show GUIDs column
//
void Titraqform::showGuidcol(void)
{
    // Test if column is currently shown, conditionally show or hide it
    if (m_pColspopup->isItemChecked(m_pColspopup->idAt(TITRAQ_IDXGCOL))) {
        m_pMaintable->hideColumn(TITRAQ_IDXGUID);
        m_pColspopup->setItemChecked(m_pColspopup->idAt(TITRAQ_IDXGCOL), false);
        m_pGuidedit->hide();
    }
    else {
        m_pMaintable->showColumn(TITRAQ_IDXGUID);
        m_pColspopup->setItemChecked(m_pColspopup->idAt(TITRAQ_IDXGCOL), true);
        m_pGuidedit->show();
    }

    // Make sure switch take place right away and sizes are handled
    m_pColspopup->updateItem(m_pColspopup->idAt(TITRAQ_IDXGCOL));
    this->updSizes(TITRAQ_IDXALLCTRLS, 0, 0); // Update size of this and next
}

//
// View menu show CRC column
//
void Titraqform::showCrccol(void)
{
    // Test if column is currently shown, conditionally show or hide it
    if (m_pColspopup->isItemChecked(m_pColspopup->idAt(TITRAQ_IDXCCOL))) {
        m_pMaintable->hideColumn(TITRAQ_IDXCRC);
        m_pColspopup->setItemChecked(m_pColspopup->idAt(TITRAQ_IDXCCOL), false);
        m_pCrcedit->hide();
    }
    else {
        m_pMaintable->showColumn(TITRAQ_IDXCRC);
        m_pColspopup->setItemChecked(m_pColspopup->idAt(TITRAQ_IDXCCOL), true);
        m_pCrcedit->show();
    }

    // Make sure switch take place right away and sizes are handled
    m_pColspopup->updateItem(m_pColspopup->idAt(TITRAQ_IDXCCOL));
    this->updSizes(TITRAQ_IDXALLCTRLS, 0, 0); // Update size of this and next
}

//
// View menu show Rev column
//
void Titraqform::showRevcol(void)
{
    // Test if column is currently shown, conditionally show or hide it
    if (m_pColspopup->isItemChecked(m_pColspopup->idAt(TITRAQ_IDXREVCOL))) {
        m_pMaintable->hideColumn(TITRAQ_IDXREV);
        m_pColspopup->setItemChecked(m_pColspopup->idAt(TITRAQ_IDXREVCOL), false);
        m_pRevedit->hide();
    }
    else {
        m_pMaintable->showColumn(TITRAQ_IDXREV);
        m_pColspopup->setItemChecked(m_pColspopup->idAt(TITRAQ_IDXREVCOL), true);
        m_pRevedit->show();
    }

    // Make sure switch take place right away and sizes are handled
    m_pColspopup->updateItem(m_pColspopup->idAt(TITRAQ_IDXREVCOL));
    this->updSizes(TITRAQ_IDXALLCTRLS, 0, 0); // Update size of this and next
}

//
// View menu show dates column
//
void Titraqform::showDatecol(void)
{
    // Test if column is currently shown, conditionally show or hide it
    if (m_pColspopup->isItemChecked(m_pColspopup->idAt(TITRAQ_IDXDCOL))) {
        m_pMaintable->hideColumn(TITRAQ_IDXDATE);
        m_pColspopup->setItemChecked(m_pColspopup->idAt(TITRAQ_IDXDCOL), false);
        m_pDateedit->hide();
    }
    else {
        m_pMaintable->showColumn(TITRAQ_IDXDATE);
        m_pColspopup->setItemChecked(m_pColspopup->idAt(TITRAQ_IDXDCOL), true);
        m_pDateedit->show();
    }

    // Make sure switch take place right away and sizes are handled
    m_pColspopup->updateItem(m_pColspopup->idAt(TITRAQ_IDXDCOL));
    this->updSizes(TITRAQ_IDXALLCTRLS, 0, 0); // Update size of this and next
}

//
// View menu show start time column
//
void Titraqform::showStartcol(void)
{
    // Test if column is currently shown, conditionally show or hide it
    if (m_pColspopup->isItemChecked(m_pColspopup->idAt(TITRAQ_IDXSTARTCOL))) {
        m_pMaintable->hideColumn(TITRAQ_IDXSTART);
        m_pColspopup->setItemChecked(m_pColspopup->idAt(TITRAQ_IDXSTARTCOL), false);
        m_pStarttime->hide();
    }
    else {
        m_pMaintable->showColumn(TITRAQ_IDXSTART);
        m_pColspopup->setItemChecked(m_pColspopup->idAt(TITRAQ_IDXSTARTCOL), true);
        m_pStarttime->show();
    }

    // Make sure switch take place right away and sizes are handled
    m_pColspopup->updateItem(m_pColspopup->idAt(TITRAQ_IDXSTARTCOL));
    this->updSizes(TITRAQ_IDXALLCTRLS, 0, 0); // Update size of this and next
}

//
// View menu show finish time column
//
void Titraqform::showFinishcol(void)
{
    // Test if column is currently shown, conditionally show or hide it
    if (m_pColspopup->isItemChecked(m_pColspopup->idAt(TITRAQ_IDXFCOL))) {
        m_pMaintable->hideColumn(TITRAQ_IDXFINISH);
        m_pColspopup->setItemChecked(m_pColspopup->idAt(TITRAQ_IDXFCOL), false);
        m_pEndtime->hide();
    }
    else {
        m_pMaintable->showColumn(TITRAQ_IDXFINISH);
        m_pColspopup->setItemChecked(m_pColspopup->idAt(TITRAQ_IDXFCOL), true);
        m_pEndtime->show();
    }

    // Make sure switch take place right away and sizes are handled
    m_pColspopup->updateItem(m_pColspopup->idAt(TITRAQ_IDXFCOL));
    this->updSizes(TITRAQ_IDXALLCTRLS, 0, 0); // Update size of this and next
}

//
// View menu show Amounts column
//
void Titraqform::showAmountcol(void)
{
    // Test if column is currently shown, conditionally show or hide it
    if (m_pColspopup->isItemChecked(m_pColspopup->idAt(TITRAQ_IDXACOL))) {
        m_pMaintable->hideColumn(TITRAQ_IDXAMOUNT);
        m_pColspopup->setItemChecked(m_pColspopup->idAt(TITRAQ_IDXACOL), false);
        m_pAmount->hide();
    }
    else {
        m_pMaintable->showColumn(TITRAQ_IDXAMOUNT);
        m_pColspopup->setItemChecked(m_pColspopup->idAt(TITRAQ_IDXACOL), true);
        m_pAmount->show();
    }

    // Make sure switch take place right away and sizes are handled
    m_pColspopup->updateItem(m_pColspopup->idAt(TITRAQ_IDXACOL));
    this->updSizes(TITRAQ_IDXALLCTRLS, 0, 0); // Update size of this and next
}

//
// View menu show tasks column
//
void Titraqform::showTaskcol(void)
{
    // Test if column is currently shown, conditionally show or hide it
    if (m_pColspopup->isItemChecked(m_pColspopup->idAt(TITRAQ_IDXTCOL))) {
        m_pMaintable->hideColumn(TITRAQ_IDXTASK);
        m_pColspopup->setItemChecked(m_pColspopup->idAt(TITRAQ_IDXTCOL), false);
        m_pTasks->hide();
    }
    else {
        m_pMaintable->showColumn(TITRAQ_IDXTASK);
        m_pColspopup->setItemChecked(m_pColspopup->idAt(TITRAQ_IDXTCOL), true);
        m_pTasks->show();
    }

    // Make sure switch take place right away and sizes are handled
    m_pColspopup->updateItem(m_pColspopup->idAt(TITRAQ_IDXTCOL));
    this->updSizes(TITRAQ_IDXALLCTRLS, 0, 0); // Update size of this and next
}

//
// View menu show Remarks column
//
void Titraqform::showRemarkcol(void)
{
    // Test if column is currently shown, conditionally show or hide it
    if (m_pColspopup->isItemChecked(m_pColspopup->idAt(TITRAQ_IDXREMCOL))) {
        m_pMaintable->hideColumn(TITRAQ_IDXREMARK);
        m_pColspopup->setItemChecked(m_pColspopup->idAt(TITRAQ_IDXREMCOL), false);
        m_pRemark->hide();
    }
    else {
        m_pMaintable->showColumn(TITRAQ_IDXREMARK);
        m_pColspopup->setItemChecked(m_pColspopup->idAt(TITRAQ_IDXREMCOL), true);
        m_pRemark->show();
    }

    // Make sure switch take place right away and sizes are handled
    m_pColspopup->updateItem(m_pColspopup->idAt(TITRAQ_IDXREMCOL));
    this->updSizes(TITRAQ_IDXALLCTRLS, 0, 0); // Update size of this and next
}

//
// Generate a local formatted report
//
void Titraqform::genReport(void)
{
    Prototype Unimp;
    Unimp.doMbox();
}

//
// Syncronize data with server using IIOP
//
void Titraqform::syncIiop(void)
{
#ifdef HAVE_MICO
    int nNada = 0;          // Some false parameters
    char **ppcNada = NULL;  // to fake out the ORB
    CORBA::ORB_var Orb;     // The ORB

    CORBA::Object_var Nameobj;      // Name service reference
    CosNaming::NamingContext_var Namectx; // COSS ns context
    CosNaming::Name Cosname;        // Our requested obj name

    CORBA::Object_var Objcaster;    // Generic CORBA object
    Asdatabase_var Asdbase;         // Casted object to ASDB
    Astuple Singlerow;              // A single row of AS data

    try {
        // Initialization of the ORB and COSS naming service
        Orb = CORBA::ORB_init(nNada, ppcNada, "mico-local-orb");
        Nameobj = Orb->resolve_initial_references("NameService");
        Namectx = CosNaming::NamingContext::_narrow(Nameobj);
        if (CORBA::is_nil(Namectx)) { // Verify sanity
            m_pStatbar->message(trUtf8("Could not find the COSS naming service"));
            qWarning("Could not find the COSS naming service\n");
        }

        // Prepare the COSS name request
        Cosname.length(1);
        Cosname[0].id = CORBA::string_dup("Asdatabase");
        Cosname[0].kind = CORBA::string_dup("");

        try { // Resolve to a CORBA object
            Objcaster = Namectx->resolve(Cosname);
        }
        catch (CosNaming::NamingContext::NotFound &Cossex) {
            m_pStatbar->message(trUtf8("NotFound exception thrown"));
            qWarning("NotFound exception thrown\n");
            return;
        }
        catch (CosNaming::NamingContext::CannotProceed &Cossex) {
            m_pStatbar->message(trUtf8("CannotProceed exception thrown"));
            qWarning("CannotProceed exception thrown\n");
            return;
        }
        catch (CosNaming::NamingContext::InvalidName &Cossex) {
            m_pStatbar->message(trUtf8("InvalidName exception thrown"));
            qWarning("InvalidName exception thrown\n");
            return;
        }

        // Cast the generic CORBA object to a AS database type
        Asdbase = Asdatabase::_narrow(Objcaster);
        if (CORBA::is_nil(Asdbase)) { // Verify sanity
            m_pStatbar->message(trUtf8("Could not find the AS database"));
            qWarning("Could not find the AS database\n");
            return;
        }

        // Open an account object on the remote server
        Account_var Account = Asdbase->Open("/tmp/corbadb.txt"); // Security problem!
        if (CORBA::is_nil(Account)) { // Verify sanity
            m_pStatbar->message(trUtf8("Could not create an account object on the server"));
            qWarning("Could not create an account object on the server\n");
            return;
        }

        // Fill an account object to marshall and transmit
        int nRow = m_pMaintable->currentRow();
        Singlerow.nDate = m_pMaintable->text(nRow, TITRAQ_IDXDATE).toInt();
        Singlerow.nTime = m_pMaintable->text(nRow, TITRAQ_IDXSTART).toInt();
        Singlerow.szName = CORBA::string_dup(m_pMaintable->text(nRow, TITRAQ_IDXUSER));
        Singlerow.szTask = CORBA::string_dup(m_pMaintable->text(nRow, TITRAQ_IDXTASK));
        Singlerow.szRemark = CORBA::string_dup(m_pMaintable->text(nRow, TITRAQ_IDXREMARK));

        Account->Log(Singlerow); // Finally transmit to server
        m_pStatbar->message(trUtf8("Successful transmission of accounting data"));
    }
    catch (const CORBA::Exception &Corbex) {
        m_pStatbar->message(trUtf8("Caught CORBA exception: %1").arg(Corbex._repoid()));
        qWarning("Caught CORBA exception: %s\n", Corbex._repoid());
    }
    catch (...) {
        qWarning("Caught unknown exception\n");
    }
#endif // HAVE_MICO
}

//
// Syncronize data with server using SOAP
//
void Titraqform::syncSoap(void)
{
#ifdef HAVE_ESOAP
    USING_EASYSOAP_NAMESPACE

    try {
        const char *pSpace    = "http://soap.europalab.com/asdb";
        const char *pEndpoint = "http://www.europalab.com/cgi-bin/asdbserv";
        SOAPProxy Proxy(pEndpoint);
        SOAPMethod Logmeth("Log", pSpace);  // SOAP remote method name
        SOAPString Clistr;                  // Outgoing parameter to marshall
        int nCrc;                           // SOAP unmarshalled return value

        // A quasi diff summary to transmit to the server
        QString Syncthis;
        int nRow = m_pMaintable->currentRow();
        Syncthis += m_pMaintable->text(nRow, TITRAQ_IDXUSER);
        for (int nIter = TITRAQ_IDXUSER + 1; nIter < TITRAQ_IDXTAIL; nIter++)
            Syncthis += ' ' + m_pMaintable->text(nRow, nIter);

        Clistr = Syncthis;                          // Build RPC parameter
        Logmeth.AddParameter("Tuple") << Clistr;    // Prepare for marshalling
        const SOAPResponse &Logresp = Proxy.Execute(Logmeth);
        Logresp.GetReturnValue() >> nCrc;
        m_pStatbar->message(trUtf8("Successful transmission, CRC returned %1").arg(nCrc));
    }
    catch (SOAPException& Soapex) { // Announce the exception we received
        m_pStatbar->message(trUtf8("Caught SOAP exception: %1").arg(Soapex.What().Str()));
        qDebug("Caught SOAP exception: %s\n", Soapex.What().Str());
    }
    catch (...) {
        qDebug("Caught unknown exception\n");
    }
#endif // HAVE_ESOAP
}

//
// Save user preferences
//
void Titraqform::savePrefs(void)
{
    // Get check status from column menu and pass it to prefs handler
    m_pPrefs->setBool(TITRAQ_PREFSTATCOLON, m_pColspopup->isItemChecked(m_pColspopup->idAt(TITRAQ_IDXSTATCOL)));
    m_pPrefs->setBool(TITRAQ_PREFLCOLON, m_pColspopup->isItemChecked(m_pColspopup->idAt(TITRAQ_IDXLCOL)));
    m_pPrefs->setBool(TITRAQ_PREFUCOLON, m_pColspopup->isItemChecked(m_pColspopup->idAt(TITRAQ_IDXUCOL)));
    m_pPrefs->setBool(TITRAQ_PREFGCOLON, m_pColspopup->isItemChecked(m_pColspopup->idAt(TITRAQ_IDXGCOL)));
    m_pPrefs->setBool(TITRAQ_PREFCCOLON, m_pColspopup->isItemChecked(m_pColspopup->idAt(TITRAQ_IDXCCOL)));
    m_pPrefs->setBool(TITRAQ_PREFREVCOLON, m_pColspopup->isItemChecked(m_pColspopup->idAt(TITRAQ_IDXREVCOL)));
    m_pPrefs->setBool(TITRAQ_PREFDCOLON, m_pColspopup->isItemChecked(m_pColspopup->idAt(TITRAQ_IDXDCOL)));
    m_pPrefs->setBool(TITRAQ_PREFSTARTCOLON, m_pColspopup->isItemChecked(m_pColspopup->idAt(TITRAQ_IDXSTARTCOL)));
    m_pPrefs->setBool(TITRAQ_PREFFCOLON, m_pColspopup->isItemChecked(m_pColspopup->idAt(TITRAQ_IDXFCOL)));
    m_pPrefs->setBool(TITRAQ_PREFACOLON, m_pColspopup->isItemChecked(m_pColspopup->idAt(TITRAQ_IDXACOL)));
    m_pPrefs->setBool(TITRAQ_PREFFCOLON, m_pColspopup->isItemChecked(m_pColspopup->idAt(TITRAQ_IDXFCOL)));
    m_pPrefs->setBool(TITRAQ_PREFREMCOLON, m_pColspopup->isItemChecked(m_pColspopup->idAt(TITRAQ_IDXREMCOL)));

    // Get check status from toolbar menu and pass it to prefs handler
    m_pPrefs->setBool(TITRAQ_PREFFILEBAR, m_pTbarspopup->isItemChecked(m_pTbarspopup->idAt(TITRAQ_IDXFILEBAR)));
    m_pPrefs->setBool(TITRAQ_PREFEDITBAR, m_pTbarspopup->isItemChecked(m_pTbarspopup->idAt(TITRAQ_IDXEDITBAR)));
    m_pPrefs->setBool(TITRAQ_PREFWHATBAR, m_pTbarspopup->isItemChecked(m_pTbarspopup->idAt(TITRAQ_IDXWHATBAR)));

    // Get column widths from main table and pass it to prefs handler
    if (m_pMaintable->columnWidth(TITRAQ_IDXSTATUS) > 0)
        m_pPrefs->setNumber(TITRAQ_PREFSTATCOLWIDTH, (long)m_pMaintable->columnWidth(TITRAQ_IDXSTATUS));
    if (m_pMaintable->columnWidth(TITRAQ_IDXLINE) > 0)
        m_pPrefs->setNumber(TITRAQ_PREFLCOLWIDTH, (long)m_pMaintable->columnWidth(TITRAQ_IDXLINE));
    if (m_pMaintable->columnWidth(TITRAQ_IDXUSER) > 0)
        m_pPrefs->setNumber(TITRAQ_PREFUCOLWIDTH, (long)m_pMaintable->columnWidth(TITRAQ_IDXUSER));
    if (m_pMaintable->columnWidth(TITRAQ_IDXGUID) > 0)
        m_pPrefs->setNumber(TITRAQ_PREFGCOLWIDTH, (long)m_pMaintable->columnWidth(TITRAQ_IDXGUID));
    if (m_pMaintable->columnWidth(TITRAQ_IDXCRC) > 0)
        m_pPrefs->setNumber(TITRAQ_PREFCCOLWIDTH, (long)m_pMaintable->columnWidth(TITRAQ_IDXCRC));
    if (m_pMaintable->columnWidth(TITRAQ_IDXREV) > 0)
        m_pPrefs->setNumber(TITRAQ_PREFREVCOLWIDTH, (long)m_pMaintable->columnWidth(TITRAQ_IDXREV));
    if (m_pMaintable->columnWidth(TITRAQ_IDXDATE) > 0)
        m_pPrefs->setNumber(TITRAQ_PREFDCOLWIDTH, (long)m_pMaintable->columnWidth(TITRAQ_IDXDATE));
    if (m_pMaintable->columnWidth(TITRAQ_IDXSTART) > 0)
        m_pPrefs->setNumber(TITRAQ_PREFSTARTCOLWIDTH, (long)m_pMaintable->columnWidth(TITRAQ_IDXSTART));
    if (m_pMaintable->columnWidth(TITRAQ_IDXFINISH) > 0)
        m_pPrefs->setNumber(TITRAQ_PREFFCOLWIDTH, (long)m_pMaintable->columnWidth(TITRAQ_IDXFINISH));
    if (m_pMaintable->columnWidth(TITRAQ_IDXAMOUNT) > 0)
        m_pPrefs->setNumber(TITRAQ_PREFACOLWIDTH, (long)m_pMaintable->columnWidth(TITRAQ_IDXAMOUNT));
    if (m_pMaintable->columnWidth(TITRAQ_IDXTASK) > 0)
        m_pPrefs->setNumber(TITRAQ_PREFTCOLWIDTH, (long)m_pMaintable->columnWidth(TITRAQ_IDXTASK));
    if (m_pMaintable->columnWidth(TITRAQ_IDXREMARK) > 0)
        m_pPrefs->setNumber(TITRAQ_PREFREMCOLWIDTH, (long)m_pMaintable->columnWidth(TITRAQ_IDXREMARK));

    // Get sorting order and direction from table and pass it to prefs handler
    m_pPrefs->setNumber(TITRAQ_PREFSORTCOL, (long)m_pMaintable->getSortcol());
    m_pPrefs->setBool(TITRAQ_PREFSORTDIR, (long)m_pMaintable->getSortdir());

    // Set frame geometry preferences
    m_pPrefs->setNumber(TITRAQ_PREFFRAMEWIDTH, (long)this->width());
    m_pPrefs->setNumber(TITRAQ_PREFFRAMEHEIGHT, (long)this->height());

    // Remember main window layout and doc positions
    QString Laystring;
    QTextStream Laystream(&Laystring, IO_WriteOnly);
    Laystream << *this; // Persist the main window
    m_pPrefs->setString(TITRAQ_PREFFRAMELAY, Laystring);
}

//
// Get help on Titraq functionality
//
void Titraqform::helpContents(void)
{
    Prototype Unimp;
    Unimp.doMbox();
}

//
// Learn more about this program itself
//
void Titraqform::aboutTitraq(void)
{
    QMessageBox *pCwmsg = new QMessageBox("OSSP titraq",
        QObject::trUtf8("OSSP titraq is a time and task-based\n"
               "accounting system that acts as both a\n"
               "work-like punch card and time tracker.\n"
               "Development of titraq is sponsored by\n"
               "Cable & Wireless Deutschland GmbH."),
        QMessageBox::NoIcon, QMessageBox::Ok | QMessageBox::Default,
        QMessageBox::NoButton, QMessageBox::NoButton,
        NULL, "Titraqmessage", true, Qt::WStyle_NormalBorder);

    pCwmsg->setIconPixmap(QPixmap(s_kpcCwlogo_xpm));
    pCwmsg->exec();
}

//
// Learn more about the OSSP
//
void Titraqform::aboutOSSP(void)
{
    QMessageBox *pOsspmsg = new QMessageBox("OSSP titraq",
        QObject::trUtf8("The open source software project (OSSP) is\n"
               "a collective effort aimed at implementing\n"
               "high-quality Unix software components,\n"
               "ranging from networking, multi-threading\n"
               "and algorithmic libraries to networking\n"
               "servers and development tools."),
        QMessageBox::NoIcon, QMessageBox::Ok | QMessageBox::Default,
        QMessageBox::NoButton, QMessageBox::NoButton,
        NULL, "Osspmessage", true, Qt::WStyle_NormalBorder);

    pOsspmsg->setIconPixmap(QPixmap(s_kpcOssplogo_xpm));
    pOsspmsg->exec();
}

//
// Learn more about this program and Qt
//
void Titraqform::aboutQt(void)
{
    QMessageBox::aboutQt(this, "OSSP titraq");
}
