
/*
 Copyright 2013, Kovid Goyal <kovid at kovidgoyal.net>
 Released under the GPLv3 License
 */

(function() {
  var INHERITED_PROPS, PreviewIntegration, find_containing_block, get_color, get_matched_css, get_sourceline_address, get_style_properties, in_table, is_block, is_hidden, log, process_rules, scroll_to_node,
    bind = function(fn, me){ return function(){ return fn.apply(me, arguments); }; };

  if (typeof window !== "undefined" && window !== null ? window.calibre_utils : void 0) {
    log = window.calibre_utils.log;
  }

  is_hidden = function(elem) {
    while (elem) {
      if (elem.style && (elem.style.visibility === 'hidden' || elem.style.display === 'none')) {
        return true;
      }
      elem = elem.parentNode;
    }
    return false;
  };

  is_block = function(elem) {
    var ref, style;
    style = window.getComputedStyle(elem);
    return (ref = style.display) === 'block' || ref === 'flex-box' || ref === 'box';
  };

  in_table = function(elem) {
    var ref;
    while (elem) {
      if (((ref = elem.tagName) != null ? ref.toLowerCase() : void 0) === 'table') {
        return true;
      }
      elem = elem.parentNode;
    }
    return false;
  };

  find_containing_block = function(elem) {
    while (elem && elem.getAttribute('data-is-block') !== '1') {
      elem = elem.parentNode;
    }
    return elem;
  };

  INHERITED_PROPS = {
    'azimuth': '2',
    'border-collapse': '2',
    'border-spacing': '2',
    'caption-side': '2',
    'color': '2',
    'cursor': '2',
    'direction': '2',
    'elevation': '2',
    'empty-cells': '2',
    'fit': '3',
    'fit-position': '3',
    'font': '2',
    'font-family': '2',
    'font-size': '2',
    'font-size-adjust': '2',
    'font-stretch': '2',
    'font-style': '2',
    'font-variant': '2',
    'font-weight': '2',
    'hanging-punctuation': '3',
    'hyphenate-after': '3',
    'hyphenate-before': '3',
    'hyphenate-character': '3',
    'hyphenate-lines': '3',
    'hyphenate-resource': '3',
    'hyphens': '3',
    'image-resolution': '3',
    'letter-spacing': '2',
    'line-height': '2',
    'line-stacking': '3',
    'line-stacking-ruby': '3',
    'line-stacking-shift': '3',
    'line-stacking-strategy': '3',
    'list-style': '2',
    'list-style-image': '2',
    'list-style-position': '2',
    'list-style-type': '2',
    'marquee-direction': '3',
    'orphans': '2',
    'overflow-style': '3',
    'page': '2',
    'page-break-inside': '2',
    'pitch': '2',
    'pitch-range': '2',
    'presentation-level': '3',
    'punctuation-trim': '3',
    'quotes': '2',
    'richness': '2',
    'ruby-align': '3',
    'ruby-overhang': '3',
    'ruby-position': '3',
    'speak': '2',
    'speak-header': '2',
    'speak-numeral': '2',
    'speak-punctuation': '2',
    'speech-rate': '2',
    'stress': '2',
    'text-align': '2',
    'text-align-last': '3',
    'text-emphasis': '3',
    'text-height': '3',
    'text-indent': '2',
    'text-justify': '3',
    'text-outline': '3',
    'text-replace': '?',
    'text-shadow': '3',
    'text-transform': '2',
    'text-wrap': '3',
    'visibility': '2',
    'voice-balance': '3',
    'voice-family': '2',
    'voice-rate': '3',
    'voice-pitch': '3',
    'voice-pitch-range': '3',
    'voice-stress': '3',
    'voice-volume': '3',
    'volume': '2',
    'white-space': '2',
    'white-space-collapse': '3',
    'widows': '2',
    'word-break': '3',
    'word-spacing': '2',
    'word-wrap': '3',
    '-moz-force-broken-image-icon': 'm',
    '-moz-image-region': 'm',
    '-moz-stack-sizing': 'm',
    '-moz-user-input': 'm',
    '-x-system-font': 'm',
    '-xv-voice-balance': 'o',
    '-xv-voice-pitch': 'o',
    '-xv-voice-pitch-range': 'o',
    '-xv-voice-rate': 'o',
    '-xv-voice-stress': 'o',
    '-xv-voice-volume': 'o',
    '-ms-text-align-last': 'e',
    '-ms-text-justify': 'e',
    '-ms-word-break': 'e',
    '-ms-word-wrap': 'e'
  };

  get_sourceline_address = function(node) {
    var elem, j, len, ref, sourceline, tags;
    sourceline = parseInt(node.getAttribute('data-lnum'));
    tags = [];
    ref = document.querySelectorAll('[data-lnum="' + sourceline + '"]');
    for (j = 0, len = ref.length; j < len; j++) {
      elem = ref[j];
      tags.push(elem.tagName.toLowerCase());
      if (elem === node) {
        break;
      }
    }
    return [sourceline, tags];
  };

  get_color = function(property, val) {
    var color, error;
    color = null;
    if (property.indexOf('color') > -1) {
      try {
        color = parseCSSColor(val);
      } catch (_error) {
        error = _error;
        color = null;
      }
    }
    return color;
  };

  get_style_properties = function(style, all_properties, node_style, is_ancestor) {
    var cval, i, properties, property, ref, val;
    i = 0;
    properties = [];
    while (i < style.length) {
      property = (ref = style.item(i)) != null ? ref.toLowerCase() : void 0;
      val = style.getPropertyValue(property);
      if (property && val && (!is_ancestor || INHERITED_PROPS.hasOwnProperty(property))) {
        properties.push([property, val, style.getPropertyPriority(property), get_color(property, val)]);
        if (!all_properties.hasOwnProperty(property)) {
          cval = node_style.getPropertyValue(property);
          all_properties[property] = [cval, get_color(property, cval)];
        }
      }
      i += 1;
    }
    return properties;
  };

  process_rules = function(node, cssRules, address, sheet, sheet_index, matching_selectors, all_properties, node_style, is_ancestor, ans) {
    var data, href, j, k, len, len1, parts, properties, q, results, rule, rule_address, rule_index, st, type;
    results = [];
    for (rule_index = j = 0, len = cssRules.length; j < len; rule_index = ++j) {
      rule = cssRules[rule_index];
      rule_address = address.concat([rule_index]);
      if (rule.type === CSSRule.MEDIA_RULE) {
        process_rules(node, rule.cssRules, rule_address, sheet, sheet_index, matching_selectors, all_properties, node_style, is_ancestor, ans);
        continue;
      }
      if (rule.type !== CSSRule.STYLE_RULE) {
        continue;
      }
      st = rule.selectorText;
      if (st && (matching_selectors.hasOwnProperty(st) || (rule_address.length > 1 && node.webkitMatchesSelector(st)))) {
        type = 'sheet';
        href = sheet.href;
        if (href === null) {
          href = get_sourceline_address(sheet.ownerNode);
          type = 'elem';
        }
        parts = st.split(',');
        if (parts.length > 1) {
          for (k = 0, len1 = parts.length; k < len1; k++) {
            q = parts[k];
            if (node.webkitMatchesSelector(q)) {
              st = q;
              break;
            }
          }
        }
        properties = get_style_properties(rule.style, all_properties, node_style, is_ancestor);
        if (properties.length > 0) {
          data = {
            'selector': st,
            'type': type,
            'href': href,
            'properties': properties,
            'rule_address': rule_address,
            'sheet_index': sheet_index
          };
          results.push(ans.push(data));
        } else {
          results.push(void 0);
        }
      } else {
        results.push(void 0);
      }
    }
    return results;
  };

  get_matched_css = function(node, is_ancestor, all_properties) {
    var ans, data, j, k, len, len1, matching_selectors, node_style, properties, ref, rule, rules, sheet, sheet_index;
    rules = node.ownerDocument.defaultView.getMatchedCSSRules(node, '');
    if (!rules) {
      rules = [];
    }
    matching_selectors = {};
    for (j = 0, len = rules.length; j < len; j++) {
      rule = rules[j];
      matching_selectors[rule.selectorText] = true;
    }
    ans = [];
    node_style = window.getComputedStyle(node);
    ref = document.styleSheets;
    for (sheet_index = k = 0, len1 = ref.length; k < len1; sheet_index = ++k) {
      sheet = ref[sheet_index];
      if (sheet.disabled || !sheet.cssRules) {
        continue;
      }
      process_rules(node, sheet.cssRules, [], sheet, sheet_index, matching_selectors, all_properties, node_style, is_ancestor, ans);
    }
    if (node.getAttribute('style')) {
      properties = get_style_properties(node.style, all_properties, node_style, is_ancestor);
      if (properties.length > 0) {
        data = {
          'selector': null,
          'type': 'inline',
          'href': get_sourceline_address(node),
          'properties': properties,
          'rule_address': null,
          'sheet_index': null
        };
        ans.push(data);
      }
    }
    return ans.reverse();
  };

  scroll_to_node = function(node) {
    if (node === document.body) {
      return window.scrollTo(0, 0);
    } else {
      return node.scrollIntoView();
    }
  };

  PreviewIntegration = (function() {

    /*
     * Namespace to expose all the functions used for integration with the Tweak
     * Book Preview Panel.
     */
    function PreviewIntegration() {
      this.live_css = bind(this.live_css, this);
      this.go_to_anchor = bind(this.go_to_anchor, this);
      this.onclick = bind(this.onclick, this);
      this.onload = bind(this.onload, this);
      this.report_split = bind(this.report_split, this);
      this.split_mode = bind(this.split_mode, this);
      this.find_blocks = bind(this.find_blocks, this);
      this.line_numbers = bind(this.line_numbers, this);
      this.go_to_sourceline_address = bind(this.go_to_sourceline_address, this);
      this.go_to_line = bind(this.go_to_line, this);
      if (!this instanceof arguments.callee) {
        throw new Error('PreviewIntegration constructor called as function');
      }
      this.blocks_found = false;
      this.in_split_mode = false;
    }

    PreviewIntegration.prototype.go_to_line = function(lnum) {
      var j, len, node, ref, results;
      ref = document.querySelectorAll('[data-lnum="' + lnum + '"]');
      results = [];
      for (j = 0, len = ref.length; j < len; j++) {
        node = ref[j];
        if (is_hidden(node)) {
          continue;
        }
        results.push(scroll_to_node(node));
      }
      return results;
    };

    PreviewIntegration.prototype.go_to_sourceline_address = function(sourceline, tags) {
      var index, j, len, node, ref;
      ref = document.querySelectorAll('[data-lnum="' + sourceline + '"]');
      for (index = j = 0, len = ref.length; j < len; index = ++j) {
        node = ref[index];
        if (index >= tags.length || node.tagName.toLowerCase() !== tags[index]) {
          break;
        }
        if (index === tags.length - 1 && !is_hidden(node)) {
          return scroll_to_node(node);
        }
      }
      return this.go_to_line(sourceline);
    };

    PreviewIntegration.prototype.line_numbers = function() {
      var ans, found_body, j, len, node, ref;
      found_body = false;
      ans = [];
      ref = document.getElementsByTagName('*');
      for (j = 0, len = ref.length; j < len; j++) {
        node = ref[j];
        if (!found_body && node.tagName.toLowerCase() === "body") {
          found_body = true;
        }
        if (found_body) {
          ans.push(node.getAttribute("data-lnum"));
        }
      }
      return ans;
    };

    PreviewIntegration.prototype.find_blocks = function() {
      var elem, j, len, ref;
      if (this.blocks_found) {
        return;
      }
      ref = document.body.getElementsByTagName('*');
      for (j = 0, len = ref.length; j < len; j++) {
        elem = ref[j];
        if (is_block(elem) && !in_table(elem)) {
          elem.setAttribute('data-is-block', '1');
        }
      }
      return this.blocks_found = true;
    };

    PreviewIntegration.prototype.split_mode = function(enabled) {
      this.in_split_mode = enabled;
      document.body.setAttribute('data-in-split-mode', enabled ? '1' : '0');
      if (enabled) {
        return this.find_blocks();
      }
    };

    PreviewIntegration.prototype.report_split = function(node) {
      var loc, num, parent, sibling, totals;
      loc = [];
      totals = [];
      parent = find_containing_block(node);
      while (parent && parent.tagName.toLowerCase() !== 'body') {
        totals.push(parent.parentNode.children.length);
        num = 0;
        sibling = parent.previousElementSibling;
        while (sibling) {
          num += 1;
          sibling = sibling.previousElementSibling;
        }
        loc.push(num);
        parent = parent.parentNode;
      }
      loc.reverse();
      totals.reverse();
      return window.py_bridge.request_split(JSON.stringify(loc), JSON.stringify(totals));
    };

    PreviewIntegration.prototype.onload = function() {
      return window.document.body.addEventListener('click', this.onclick, true);
    };

    PreviewIntegration.prototype.onclick = function(event) {
      var address, e, href, ref, tn;
      event.preventDefault();
      if (this.in_split_mode) {
        this.report_split(event.target);
      } else {
        e = event.target;
        address = get_sourceline_address(e);
        href = tn = '';
        while (e && e !== document.body && e !== document && (tn !== 'a' || !href)) {
          tn = (ref = e.tagName) != null ? ref.toLowerCase() : void 0;
          href = e.getAttribute('href');
          e = e.parentNode;
        }
        window.py_bridge.request_sync(tn, href, JSON.stringify(address));
      }
      return false;
    };

    PreviewIntegration.prototype.go_to_anchor = function(anchor, lnum) {
      var address, elem;
      elem = document.getElementById(anchor);
      if (!elem) {
        elem = document.querySelector('[name="' + anchor + '"]');
      }
      if (elem) {
        elem.scrollIntoView();
        address = get_sourceline_address(elem);
      }
      return window.py_bridge.request_sync('', '', address);
    };

    PreviewIntegration.prototype.live_css = function(sourceline, tags) {
      var all_properties, ans, css, i, is_ancestor, j, len, node, original_target, ref, ref1, ref2, target;
      target = null;
      i = 0;
      ref = document.querySelectorAll('[data-lnum="' + sourceline + '"]');
      for (j = 0, len = ref.length; j < len; j++) {
        node = ref[j];
        if (((ref1 = node.tagName) != null ? ref1.toLowerCase() : void 0) !== tags[i]) {
          return JSON.stringify(null);
        }
        i += 1;
        target = node;
        if (i >= tags.length) {
          break;
        }
      }
      all_properties = {};
      original_target = target;
      ans = {
        'nodes': [],
        'computed_css': all_properties
      };
      is_ancestor = false;
      while (target && target.ownerDocument) {
        css = get_matched_css(target, is_ancestor, all_properties);
        if (css.length > 0 || !is_ancestor) {
          ans['nodes'].push({
            'name': (ref2 = target.tagName) != null ? ref2.toLowerCase() : void 0,
            'css': css,
            'is_ancestor': is_ancestor,
            'sourceline': target.getAttribute('data-lnum')
          });
        }
        target = target.parentNode;
        is_ancestor = true;
      }
      return JSON.stringify(ans);
    };

    return PreviewIntegration;

  })();

  window.calibre_preview_integration = new PreviewIntegration();

  window.onload = window.calibre_preview_integration.onload;

}).call(this);
