/*******************************************************************************
 * Copyright (c) 2000, 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.rdb.sqleditor.internal;

import java.util.ResourceBundle;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.source.ISourceViewer;
import org.eclipse.jface.text.source.IVerticalRuler;
import org.eclipse.jface.text.source.SourceViewerConfiguration;
import org.eclipse.jface.text.source.projection.ProjectionSupport;
import org.eclipse.jface.text.source.projection.ProjectionViewer;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.editors.text.TextEditor;
import org.eclipse.ui.help.WorkbenchHelp;
import org.eclipse.ui.texteditor.DefaultRangeIndicator;
import org.eclipse.ui.texteditor.IDocumentProvider;
import org.eclipse.ui.texteditor.ITextEditorActionDefinitionIds;
import org.eclipse.ui.texteditor.TextOperationAction;
import org.eclipse.ui.views.contentoutline.IContentOutlinePage;
import org.eclipse.wst.rdb.connection.internal.ui.wizards.ConnectionSelectionWizard;
import org.eclipse.wst.rdb.internal.core.connection.ConnectionInfo;
import org.eclipse.wst.rdb.internal.models.sql.schema.Database;
import org.eclipse.wst.rdb.sqleditor.internal.actions.SQLConnectAction;
import org.eclipse.wst.rdb.sqleditor.internal.actions.SQLDisconnectAction;
import org.eclipse.wst.rdb.sqleditor.internal.actions.SQLRunAction;
import org.eclipse.wst.rdb.sqleditor.internal.actions.SQLSetStatementTerminatorAction;
import org.eclipse.wst.rdb.sqleditor.internal.editor.SQLEditorContentOutlinePage;
import org.eclipse.wst.rdb.sqleditor.internal.editor.SQLSourceViewer;
import org.eclipse.wst.rdb.sqleditor.internal.editor.SQLSourceViewerConfiguration;
import org.eclipse.wst.rdb.sqleditor.internal.sql.ISQLDBProposalsService;
import org.eclipse.wst.rdb.sqleditor.internal.sql.SQLCodeScanner;
import org.eclipse.wst.rdb.sqleditor.internal.sql.SQLDBProposalsService;
import org.eclipse.wst.rdb.sqleditor.internal.utils.SQLColorProvider;
import org.eclipse.wst.rdb.sqleditor.internal.utils.SQLStatementSupport;


/**
 * This class is responsible for configuring the SQL editor.
 */
public class SQLEditor extends TextEditor implements IPropertyChangeListener {
	public static final String PLUGIN_NAME = "org.eclipse.wst.rdb.sqleditor"; //$NON-NLS-1$
	public static final String HELP_CONTEXT_ID = PLUGIN_NAME + ".sqleditorhelp"; //$NON-NLS-1$
	
    /** The content outline page providing the outline view for the editor content. */
    private SQLEditorContentOutlinePage fOutlinePage;
    /** The projection (code folding) support object. */
    private ProjectionSupport fProjectionSupport;
    /** The connection info object, which holds info about the connected database. */
    private ConnectionInfo fConnectionInfo;
    /** The database model object, which provides an anchor for DB catalog information.  */
    private Database fDatabase;
    /** The default schema name to use with the database associated with the editor. */
    private String fDefaultSchemaName;
    /** The SQL code scanner, which is used for colorizing the edit text. */
    private SQLCodeScanner fSQLCodeScanner;
    /** The document setup participant object, which is used partition the edit text. */
    private SQLEditorDocumentSetupParticipant fDocSetupParticipant;
    /** The content assist proposal service for database objects. */
    private ISQLDBProposalsService fDBProposalsService;
    /** The statement terminator support object. */
    private SQLStatementSupport fStmtSupport;

    /**
     * Constructs an instance of this class. This is the default constructor.
     */
    public SQLEditor() {
        super();
    }

    /**
     * Creates and installs the editor actions.
     * 
     * @see org.eclipse.ui.texteditor.AbstractTextEditor#createActions()
     */
    protected void createActions() {
        super.createActions();
        ResourceBundle bundle = getResourceBundle();

        IAction a = new TextOperationAction( bundle,
                "ContentAssistProposal.", this, ISourceViewer.CONTENTASSIST_PROPOSALS ); //$NON-NLS-1$
        a.setActionDefinitionId( ITextEditorActionDefinitionIds.CONTENT_ASSIST_PROPOSALS );
        setAction( "ContentAssistProposal", a ); //$NON-NLS-1$

        a = new TextOperationAction( bundle, "ContentAssistTip.", this, ISourceViewer.CONTENTASSIST_CONTEXT_INFORMATION ); //$NON-NLS-1$
        a.setActionDefinitionId( ITextEditorActionDefinitionIds.CONTENT_ASSIST_CONTEXT_INFORMATION );
        setAction( "ContentAssistTip", a ); //$NON-NLS-1$

        a = new TextOperationAction( bundle, "ContentFormat.", this, ISourceViewer.FORMAT ); //$NON-NLS-1$
        setAction( "ContentFormat", a ); //$NON-NLS-1$

        a = new SQLConnectAction( bundle, "SQLEditor.connectAction." ); //$NON-NLS-1$
        setAction( "SQLEditor.connectAction", a ); //$NON-NLS-1$

        a = new SQLDisconnectAction( bundle, "SQLEditor.disconnectAction." ); //$NON-NLS-1$
        setAction( "SQLEditor.disconnectAction", a ); //$NON-NLS-1$
        
        a = new SQLRunAction( bundle, "SQLEditor.runAction." ); //$NON-NLS-1$
        setAction( "SQLEditor.runAction", a ); //$NON-NLS-1$
        
        a = new SQLSetStatementTerminatorAction( bundle, "SQLEditor.setStatementTerminatorAction." ); //$NON-NLS-1$
        setAction( "SQLEditor.setStatementTerminatorAction", a ); //$NON-NLS-1$
    }

    /**
     * Creates and returns the content outline page for the SQL editor.
     * 
     * @return the new content outline page
     */
    protected SQLEditorContentOutlinePage createContentOutlinePage() {
        SQLEditorContentOutlinePage outlinePage = new SQLEditorContentOutlinePage( getDocumentProvider(), this );

        return outlinePage;
    }

    /**
     * Creates the SWT controls for the editor.
     * 
     * @see org.eclipse.ui.texteditor.AbstractTextEditor#createPartControl(org.eclipse.swt.widgets.Composite)
     */
    public void createPartControl( Composite parent ) {
        super.createPartControl( parent );
        setProjectionSupport( createProjectionSupport() );
        
        /* Now that we have enabled source folding, make sure everything is
         * expanded.
         */
        ProjectionViewer viewer = (ProjectionViewer) getSourceViewer();
        viewer.doOperation( ProjectionViewer.TOGGLE );
        
        /* Set a help context ID to enable F1 help. */
        WorkbenchHelp.setHelp( parent, HELP_CONTEXT_ID );
    }

    /**
     * Creates, configures, and returns a <code>ProjectionSupport</code>
     * object for this editor.
     * 
     * @return the <code>ProjectSupport</code> object to use with this editor
     */
    protected ProjectionSupport createProjectionSupport() {
        ProjectionViewer viewer = (ProjectionViewer) getSourceViewer();
        ProjectionSupport projSupport = new ProjectionSupport( viewer, getAnnotationAccess(), getSharedColors() );
        projSupport.addSummarizableAnnotationType( "org.eclipse.ui.workbench.texteditor.error" ); //$NON-NLS-1$
        projSupport.addSummarizableAnnotationType( "org.eclipse.ui.workbench.texteditor.warning" ); //$NON-NLS-1$
        projSupport.install();

        return projSupport;
    }

    /**
     * Creates the source viewer to be used by this editor.
     * 
     * @see org.eclipse.ui.texteditor.AbstractTextEditor#createSourceViewer(org.eclipse.swt.widgets.Composite,
     *      org.eclipse.jface.text.source.IVerticalRuler, int)
     */
    protected ISourceViewer createSourceViewer( Composite parent, IVerticalRuler ruler, int styles ) {
        SQLSourceViewer viewer = new SQLSourceViewer( parent, ruler, getOverviewRuler(), isOverviewRulerVisible(),
                styles );

        return viewer;
    }

    /**
     * Creates the source viewer configuation to be used by this editor.
     * 
     * @return the new source viewer configuration object
     */
    protected SQLSourceViewerConfiguration createSourceViewerConfiguration() {
        SQLSourceViewerConfiguration config = new SQLSourceViewerConfiguration( this );
        ISQLDBProposalsService proposalsService = getDBProposalsService();
        if (proposalsService != null) {
            config.setDBProposalsService( proposalsService );
        }

        return config;
    }

    /**
     * Dispose of resources held by this editor.
     * 
     * @see IWorkbenchPart#dispose()
     */
    public void dispose() {
        SQLEditorContentOutlinePage outlinePage = getOutlinePage();
        if (outlinePage != null) {
            outlinePage.setInput( null );
        }
        super.dispose();
    }

    /**
     * Abandons all modifications applied to this text editor's input element's
     * textual presentation since the last save operation.
     * 
     * @see ITextEditor#doRevertToSaved()
     */
    public void doRevertToSaved() {
        super.doRevertToSaved();
        updateOutlinePage();
    }

    /**
     * Saves the content of this editor.
     * 
     * @param progressMonitor the progress monitor for communicating result
     *            state or <code>null</code>
     * @see org.eclipse.ui.ISaveablePart#doSave(org.eclipse.core.runtime.IProgressMonitor)
     */
    public void doSave( IProgressMonitor monitor ) {
        super.doSave( monitor );
        updateOutlinePage();
    }

    /**
     * Saves the contents of this editor to another object.
     * 
     * @see org.eclipse.ui.ISaveablePart#doSaveAs()
     */
    public void doSaveAs() {
        super.doSaveAs();
        updateOutlinePage();
    }

    /**
     * Sets the input of the outline page after this class has set input.
     * 
     * @param input the new input for the editor
     * @see org.eclipse.ui.editors.text.TextEditor#doSetInput(org.eclipse.ui.IEditorInput)
     */
    public void doSetInput( IEditorInput input ) throws CoreException {
        super.doSetInput( input );

        /* Make sure the document partitioner is set up. The document setup
         * participant sets up document partitioning, which is used for text
         * colorizing and other text features.
         */
        IDocumentProvider docProvider = this.getDocumentProvider();
        if (docProvider != null) {
            IDocument doc = docProvider.getDocument( input );
            if (doc != null) {
                SQLEditorDocumentSetupParticipant docSetupParticipant = getDocumentSetupParticipant();
                docSetupParticipant.setup( doc );
            }
        }

        /* Determine if our input object is an instance of ISQLEditorInput.  If so,
         * get all the information that it contains.
         */
        ConnectionInfo connInfo = null;
        Database db = null;
        String defaultSchemaName = null;
        if (input instanceof ISQLEditorInput) {
            ISQLEditorInput sqlInput = (ISQLEditorInput) input;
            connInfo = sqlInput.getConnectionInfo();
            db = sqlInput.getDatabase();
            defaultSchemaName = sqlInput.getDefaultSchemaName();
        }
        
        /* If we didn't get a database but we did get a connection, try to get
         * the database from the connection.
         */
        if (connInfo != null && db == null) {
            db = connInfo.getSharedDatabase();
        }
        
        /* Save away the connection and the database information. */
        setConnectionInfo( connInfo );
        setDatabase( db );
        setDefaultSchemaName( defaultSchemaName );

        /* Show the connection status in the status area at the bottom of the
         * workbench window.
         */
        refreshConnectionStatus();
        
        /* Pass the input along to the outline page. */
        SQLEditorContentOutlinePage outlinePage = getOutlinePage();
        if (outlinePage != null) {
            outlinePage.setInput( input );
        }
    }

    /**
     * Sets up this editor's context menu before it is made visible.
     * 
     * @see org.eclipse.ui.texteditor.AbstractTextEditor#editorContextMenuAboutToShow(org.eclipse.jface.action.IMenuManager)
     */
    protected void editorContextMenuAboutToShow( IMenuManager menu ) {
        super.editorContextMenuAboutToShow( menu );

        menu.add( new Separator() );
        addAction( menu, "ContentAssistProposal" ); //$NON-NLS-1$
        addAction( menu, "ContentAssistTip" ); //$NON-NLS-1$
        addAction( menu, "ContentFormat" ); //$NON-NLS-1$

        menu.add( new Separator() );
        addAction( menu, "SQLEditor.connectAction" ); //$NON-NLS-1$
        addAction( menu, "SQLEditor.runAction" ); //$NON-NLS-1$
        addAction( menu, "SQLEditor.setStatementTerminatorAction" ); //$NON-NLS-1$
        
        updateActionState(menu);
    }

    /**
     * Gets an adapter for the given class. Returns the SQL content outline page
     * if the get request is for an outline page. Otherwise returns a projection
     * adapter if one hasn't already been created.
     * 
     * @see org.eclipse.core.runtime.IAdaptable#getAdapter(java.lang.Class)
     * @see org.eclipse.jface.text.source.projection.ProjectionSupport#getAdapter(org.eclipse.jface.text.source.ISourceViewer,
     *      java.lang.Class)
     */
    public Object getAdapter( Class classForWhichAdapterNeeded ) {
        Object adapter = null;

        /* Get and return the content outline page, if that's what's requested. */
        if (IContentOutlinePage.class.equals( classForWhichAdapterNeeded )) {
            SQLEditorContentOutlinePage outlinePage = getOutlinePage();
//            if (outlinePage == null) {
//                outlinePage = createContentOutlinePage();
//                setOutlinePage( outlinePage );
//                if (getEditorInput() != null) {
//                    outlinePage.setInput( getEditorInput() );
//                }
//            }
            adapter = outlinePage;
        }
        /* Delegate getting the adapter to the projection support object,
         * if there is one. Projection refers to the ability to visibly collapse
         * and expand sections of the document.
         */
        else if (adapter == null) {
            ProjectionSupport projSupport = getProjectionSupport();
            if (projSupport != null) {
                adapter = projSupport.getAdapter( getSourceViewer(), classForWhichAdapterNeeded );
            }
        }

        /* If we still don't have an adapter let the superclass handle it. */
        if (adapter == null) {
            adapter = super.getAdapter( classForWhichAdapterNeeded );
        }

        return adapter;
    }

    /**
     * Gets the connection info object for this editor.
     * 
     * @return the current connection info object if the editor has one,
     *         otherwise null
     */
    public ConnectionInfo getConnectionInfo() {
        return fConnectionInfo;
    }

    /**
     * Gets the <code>Database</code> object associated with this input.
     * 
     * @return the <code>Database</code> object associated with this input
     */
    public Database getDatabase() {
        return fDatabase;
    }
    
    /**
     * Gets the <code>DBProposalsService</code> object that provides content
     * assist services for this editor.
     * 
     * @return the current <code>DBProposalsService</code> object
     */
    public ISQLDBProposalsService getDBProposalsService() {
        return fDBProposalsService;
    }

    /**
     * Gets the default schema name to use with the connection or database
     * associated with this input.
     * 
     * @return the default schema name to use with this input, or null if none
     */
    public String getDefaultSchemaName() {
        return fDefaultSchemaName;
    }

    /**
     * Gets the document setup participant object associated with this editor.
     * The setup participant sets the partitioning type for the document.
     * 
     * @return the current document setup participant
     */
    public SQLEditorDocumentSetupParticipant getDocumentSetupParticipant() {
        if (fDocSetupParticipant == null) {
            fDocSetupParticipant = new SQLEditorDocumentSetupParticipant();
        }
        return fDocSetupParticipant;
    }

    /**
     * Gets the outline page associated with this editor, if there is one.
     * 
     * @return the outline page associated with this editor
     */
    protected SQLEditorContentOutlinePage getOutlinePage() {
        return fOutlinePage;
    }

    /**
     * Gets the <code>ProjectionSupport</code> object associated with this
     * editor.
     * 
     * @return the current <code>ProjectionSupport</code> object
     */
    protected ProjectionSupport getProjectionSupport() {
        return fProjectionSupport;
    }

    /**
     * Gets the resource bundle associated with this editor.
     * 
     * @return the resource bundle associated with this editor.
     */
    public ResourceBundle getResourceBundle() {
        return SQLEditorResources.getResourceBundle();
    }

    /**
     * Gets the SQL source code text scanner. Creates a default one if it
     * doesn't exist yet.
     * 
     * @return the SQL source code text scanner
     */
    public SQLCodeScanner getSQLCodeScanner() {
        if (fSQLCodeScanner == null) {
            fSQLCodeScanner = new SQLCodeScanner( SQLEditorPlugin.getDefault().getSQLColorProvider() );
        }
        return fSQLCodeScanner;
    }

    /**
     * Gets the color provider for colorizing SQL source code.
     * 
     * @return the SQL color provider
     */
    public SQLColorProvider getSQLColorProvider() {
        return SQLEditorPlugin.getDefault().getSQLColorProvider();
    }

    /**
     * Gets the statement support object associated with this editor.
     *  
     * @return the statement support object 
     */
    public SQLStatementSupport getSQLStatementSupport() {
        if (fStmtSupport == null) {
            fStmtSupport = new SQLStatementSupport();
        }
            
        IDocumentProvider docProvider = getDocumentProvider();
        IEditorInput editorInput = getEditorInput();
        IDocument doc = docProvider.getDocument( editorInput );
        fStmtSupport.setDocument( doc );
        
        return fStmtSupport;
    }

    /**
     * Initializes the editor.
     * 
     * @see org.eclipse.ui.editors.text.TextEditor#initializeEditor()
     */
    protected void initializeEditor() {
        super.initializeEditor();
        setSourceViewerConfiguration( createSourceViewerConfiguration() );
        setRangeIndicator( new DefaultRangeIndicator() );
    }
    
    /**
     * Handles notifications to the object that a property has changed.
     * 
     * @param event the property change event object describing which property
     *            changed and how
     */
    public void propertyChange( PropertyChangeEvent event ) {
        if (event.getProperty().equals( SQLConnectAction.CONNECTION )) {
            ConnectionInfo connInfo = (ConnectionInfo) event.getNewValue();
            setConnectionInfo( connInfo );
            refreshConnectionStatus();
        }
    }

    /**
     * Prompts the user to select an existing connection or create a new one.
     * Returns a <code>ConnectionInfo</code> object for the chosen connection. 
     * 
     * @return the <code>ConnectionInfo</code> object for the selected connection or
     * null if none selected
     */
    public ConnectionInfo requestConnectionFromUser() {
        ConnectionSelectionWizard connWizard = new ConnectionSelectionWizard();
        Shell shell = 
            SQLEditorPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getShell();
        WizardDialog dialog = new WizardDialog(shell, connWizard);
        dialog.setBlockOnOpen(true);
        if (dialog.open() == 0) {
            ConnectionInfo info = connWizard.getConInfo();
            return info;            
        }
        return null;
    }

    /**
     * Refreshes the status area indicating the connection state.
     */
    public void refreshConnectionStatus() {
        IEditorSite editorSite = getEditorSite();
        if (editorSite != null) {
            IActionBars actionBars = editorSite.getActionBars();
            if (actionBars != null) {
                IStatusLineManager statusLineMgr = actionBars.getStatusLineManager();
                ConnectionInfo connInfo = getConnectionInfo();
                if (connInfo != null) {
                    statusLineMgr.setErrorMessage( null );
                    String connStatus = connInfo.getName() + " (" + connInfo.getDatabaseName() + ": " + connInfo.getURL() + ")"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
                    statusLineMgr.setMessage( connStatus );
                }
                else {
                    String connStatus = SQLEditorResources.getString( "SQLEditor.connection.status.noConnection" ); //$NON-NLS-1$
                    statusLineMgr.setErrorMessage( connStatus );
                }
                actionBars.updateActionBars();
            }
        }
    }

    /**
     * Sets the <code>ConnectionInfo</code> object to be used by this editor.
     * 
     * @param connInfo the <code>ConnectionInfo</code> object to use
     */
    public void setConnectionInfo( ConnectionInfo connInfo ) {
        fConnectionInfo = connInfo;
        
        IEditorInput input = getEditorInput();
        if (input instanceof ISQLEditorInput) {
            ISQLEditorInput sqlEditorInput = (ISQLEditorInput) input;
            sqlEditorInput.setConnectionInfo( connInfo );
        }
        
        if (fConnectionInfo != null) {
            fDBProposalsService = new SQLDBProposalsService( connInfo );
            SourceViewerConfiguration config = getSourceViewerConfiguration();
            if (config != null && config instanceof SQLSourceViewerConfiguration) {
                SQLSourceViewerConfiguration sqlConfig = (SQLSourceViewerConfiguration) config;
                sqlConfig.setDBProposalsService( fDBProposalsService );
            }
        }
    }

    /**
     * Sets the <code>Database</code> object associated with this input to the
     * given object.
     * 
     * @param the <code>Database</code> object to set
     */
    public void setDatabase( Database database ) {
        fDatabase = database;

        IEditorInput input = getEditorInput();
        if (input instanceof ISQLEditorInput) {
            ISQLEditorInput sqlEditorInput = (ISQLEditorInput) input;
            sqlEditorInput.setDatabase( database );
        }
    }

    /**
     * Sets the default schema name to use with the connection or database
     * associated with this input.
     * 
     * @param the default schema name to set
     */
    public void setDefaultSchemaName( String schemaName ) {
        fDefaultSchemaName = schemaName;
        
        IEditorInput input = getEditorInput();
        if (input instanceof ISQLEditorInput) {
            ISQLEditorInput sqlEditorInput = (ISQLEditorInput) input;
            sqlEditorInput.setDefaultSchemaName( schemaName );
        }
    }
    /**
     * Sets the document setup participant object associated with this editor to
     * the given object. The setup participant sets the partitioning type for
     * the document.
     * 
     * @return the current document setup participant
     */
    public void setDocumentSetupParticipant( SQLEditorDocumentSetupParticipant docSetupParticipant ) {
        fDocSetupParticipant = docSetupParticipant;
    }

    
    /**
     * Asks this part to take focus within the workbench.
     * 
     * @see org.eclipse.ui.IWorkbenchPart#setFocus()
     */
    public void setFocus() {
        super.setFocus();
        
        /* Make sure the connection status message is up to date.  It doesn't
         * automatically get updated if the user switches editor instances.
         */
        refreshConnectionStatus();
    }
    
	/**
	 * Returns the orientation of this part.  Make sure this editor is always
	 * oriented left-to-right.  (That is, prevent mirroring for BiDi languages.)
	 * 
	 * @see org.eclipse.ui.part.IWorkbenchPartOrientation#getOrientation()
	 */
	public int getOrientation() {
		return SWT.LEFT_TO_RIGHT;
	}

    /**
     * Sets the current outline page to the given outline page object.
     * 
     * @param outlinePage the outline page to use
     */
    protected void setOutlinePage( SQLEditorContentOutlinePage outlinePage ) {
        fOutlinePage = outlinePage;
    }

    /**
     * Sets the <code>ProjectionSupport</code> object associated with this
     * editor.
     * 
     * @param projSupport the <code>ProjectionSupport</code> object to use
     */
    protected void setProjectionSupport( ProjectionSupport projSupport ) {
        fProjectionSupport = projSupport;
    }

    /**
     * Sets the <code>SQLStatementSupport</code> object associated with this editor
     * to the given object.
     * 
     * @param stmtSupport the statement support object to use
     */
    public void setSQLStatementSupport( SQLStatementSupport stmtSupport ) {
        fStmtSupport = stmtSupport;
    }
    
    /**
     * Updates the outline page (if there is one) to reflect the latest changes
     * in the editor.
     */
    protected void updateOutlinePage() {
        SQLEditorContentOutlinePage outlinePage = getOutlinePage();
        if (outlinePage != null) {
            outlinePage.update();
        }
    }
    
	/**
     * Updates the actions based on the current state of the connection.
     */
    protected void updateActionState(IMenuManager menu) {
        IAction runAction = getAction("SQLEditor.runAction");
        ConnectionInfo coninfo = getConnectionInfo();
        /* Disable the Run action if we're working with an off-line connection. */
        if (coninfo != null && coninfo.getSharedConnection() == null && coninfo.getSharedDatabase() != null) {
            runAction.setEnabled(false);
        }
        else {
            runAction.setEnabled(true);
        }     
	}

} // end class
