from __future__ import unicode_literals

import weakref
from datetime import date, datetime
from functools import wraps
from ..compat import string_types, quote_plus

# parts of URL to be omitted
SKIP_IN_PATH = (None, '', b'', [], ())

def _escape(value):
    """
    Escape a single value of a URL string or a query parameter. If it is a list
    or tuple, turn it into a comma-separated string first.
    """

    # make sequences into comma-separated stings
    if isinstance(value, (list, tuple)):
        value = ','.join(value)

    # dates and datetimes into isoformat
    elif isinstance(value, (date, datetime)):
        value = value.isoformat()

    # make bools into true/false strings
    elif isinstance(value, bool):
        value = str(value).lower()

    # encode strings to utf-8
    if isinstance(value, string_types):
        try:
            return value.encode('utf-8')
        except UnicodeDecodeError:
            # Python 2 and str, no need to re-encode
            pass
    
    return str(value)

def _make_path(*parts):
    """
    Create a URL string from parts, omit all `None` values and empty strings.
    Convert lists nad tuples to comma separated values.
    """
    #TODO: maybe only allow some parts to be lists/tuples ?
    return '/' + '/'.join(
        # preserve ',' and '*' in url for nicer URLs in logs
        quote_plus(_escape(p), b',*') for p in parts if p not in SKIP_IN_PATH)

# parameters that apply to all methods
GLOBAL_PARAMS = ('pretty', 'format', 'filter_path')

def query_params(*es_query_params):
    """
    Decorator that pops all accepted parameters from method's kwargs and puts
    them in the params argument.
    """
    def _wrapper(func):
        @wraps(func)
        def _wrapped(*args, **kwargs):
            params = kwargs.pop('params', {})
            for p in es_query_params + GLOBAL_PARAMS:
                if p in kwargs:
                    params[p] = _escape(kwargs.pop(p))

            # don't treat ignore and request_timeout as other params to avoid escaping
            for p in ('ignore', 'request_timeout'):
                if p in kwargs:
                    params[p] = kwargs.pop(p)
            return func(*args, params=params, **kwargs)
        return _wrapped
    return _wrapper


class NamespacedClient(object):
    def __init__(self, client):
        self.client = client

    @property
    def transport(self):
        return self.client.transport

class AddonClient(NamespacedClient):
    @classmethod
    def infect_client(cls, client):
        addon = cls(weakref.proxy(client))
        setattr(client, cls.namespace, addon)
        return client
