;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;
;;;                Centre for Speech Technology Research                  ;;
;;;                     University of Edinburgh, UK                       ;;
;;;                         Copyright (c) 1998                            ;;
;;;                        All Rights Reserved.                           ;;
;;;                                                                       ;;
;;;  Permission is hereby granted, free of charge, to use and distribute  ;;
;;;  this software and its documentation without restriction, including   ;;
;;;  without limitation the rights to use, copy, modify, merge, publish,  ;;
;;;  distribute, sublicense, and/or sell copies of this work, and to      ;;
;;;  permit persons to whom this work is furnished to do so, subject to   ;;
;;;  the following conditions:                                            ;;
;;;   1. The code must retain the above copyright notice, this list of    ;;
;;;      conditions and the following disclaimer.                         ;;
;;;   2. Any modifications must be clearly marked as such.                ;;
;;;   3. Original authors' names are not deleted.                         ;;
;;;   4. The authors' names are not used to endorse or promote products   ;;
;;;      derived from this software without specific prior written        ;;
;;;      permission.                                                      ;;
;;;                                                                       ;;
;;;  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        ;;
;;;  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      ;;
;;;  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   ;;
;;;  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     ;;
;;;  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    ;;
;;;  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   ;;
;;;  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          ;;
;;;  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       ;;
;;;  THIS SOFTWARE.                                                       ;;
;;;                                                                       ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Definition of Festival lexicon, derived from CMUDICT-0.4
;;;

(defvar cmulexdir (path-append lexdir "cmu"))

(require 'pos)

(define (cmu_lts_function word feats)
  "(cmu_lts_function word feats)
Function called for CMULEX when word is not found in lexicon.  Uses
LTS rules trained from the original lexicon, and lexical stress
prediction rules."
  (require 'lts)
  (if (not (boundp 'cmu_lts_rules))
      (load (path-append cmulexdir "cmu_lts_rules.scm")))
  (let ((dcword (downcase word))
	(syls) (phones))
    (if (string-matches dcword "[a-z]*")
	(begin
	  (set! phones (lts_predict dcword cmu_lts_rules))
	  (set! syls (lex.syllabify.phstress phones))
	  )
	(set! syls nil))
    (list word nil syls)))


(define (cmulex_addenda)
  "(cmulex_addenda)
Add entries to the current lexicon (radio/darpa).  These are basically
words that are not in the CMU lexicon."
  (lex.add.entry 
   '("t" n (((t iy) 1))))
  (lex.add.entry 
   '("I'll" v (((ay l) 1))))
  (lex.add.entry 
   '("I'd" v (((ay d) 1))))
  (lex.add.entry 
   '("I'm" v (((ay m) 1))))
  (lex.add.entry 
   '("uk" n (((y uw) 1) ((k ey) 1))))
  (lex.add.entry 
   '("w" n (((d ah b) 1) ((ah l) 0) ((y uw) 1))))
  (lex.add.entry
   '("'s" pos (((ax z) 0))))
  (lex.add.entry
   '("bought" v (((b ao t) 1))))
  (lex.add.entry
   '("edinburgh" n (((eh d) 1) ((ah n) 0) ((b ax ) 0) ((r ow) 0))))
  (lex.add.entry
   '("non-intoxicating" () 
      (((n aa n) 1) ((ih n t) 0) ((aa k) 1) ((s ih k) 0) ((ey t) 1) ((ih ng) 0))))
  (lex.add.entry 
   '("AT&T" n (((ey) 1) ((t iy) 1) ((ah n d) 0) ((t iy) 1))))
  (lex.add.entry 
   '("cepstra" n (((k eh p) 1) ((s t r aa) 0))))
  (lex.add.entry 
   '("cepstral" n (((k eh p) 1) ((s t r ah l) 0))))
  (lex.add.entry 
   '("cepstrum" n (((k eh p) 1) ((s t r ah m) 0))))
  (lex.add.entry
   '("diphone" n (((d ay) 1) ((f ow n) 0))))
  (lex.add.entry
   '("diphones" n (((d ay) 1) ((f ow n s) 0))))
  (lex.add.entry
   '("Dr" n (((d aa k) 1) ((t er) 0))))
  (lex.add.entry
   '("emacs" n (((iy) 1) ((m ae k s) 0))))
  (lex.add.entry 
   '("hz" n (((hh eh r t z) 1))))
  (lex.add.entry 
   '("khz" n (((k ih) 1) ((l ax) 0) ((hh eh r t z) 1))))
  (lex.add.entry
   '("lived" v (((l ih v d) 1))))
  (lex.add.entry
   '("Ltd" n (((l ih m) 1) ((ah t) 0) ((ah d) 0))))
  (lex.add.entry
   '("Mrs" n (((m ih s) 1) ((ah s) 0))))
  (lex.add.entry 
   '("mhz" n (((m eh) 1) ((g ax) 0) ((hh eh r t z) 1))))
  (lex.add.entry
   '("NASA" n (((n ae) 1) ((s ae) 1))))
  (lex.add.entry 
   '("lead" led (((l eh d) 1))))
  (lex.add.entry 
   '("lead" liid (((l iy d) 1))))
  (lex.add.entry
   '("pasadena" n (((p ae s) 1) ((ae d) 0) ((iy n) 1) ((ax) 0))))
  (lex.add.entry 
   '("read" red (((r eh d) 1))))
  (lex.add.entry 
   '("read" riid (((r iy d) 1))))
  (lex.add.entry
   '("reuters" n (((r oy t) 1) ((er z) 0))))
  (lex.add.entry
   '("to" to (((t ax) 0))))
;  (lex.add.entry
;   '("usually" r (((y uw zh) 1) ((ax l) 0) ((iy) 0))))
;  (lex.add.entry
;   '("usual" r (((y uw zh) 1) ((ax l) 0))))
  (lex.add.entry
   '("Jan" n (((jh ae n y) 1) ((uw) 0) ((eh r) 1) ((iy) 0))))
  (lex.add.entry
   '("Feb" n (((f eh b) 1) ((r ax) 0) ((er) 0) ((iy) 0))))
  (lex.add.entry
   '("February" n (((f eh b) 1) ((r ax) 0) ((er) 0) ((iy) 0))))
  (lex.add.entry
   '("Mar" n (((m aa r ch) 0))))
  (lex.add.entry
   '("Apr" n (((ey p) 1) ((r ah l) 0))))
  (lex.add.entry
   '("Jun" n  (((jh uw n) 1))))
  (lex.add.entry
   '("Jul" n  (((jh uw l) 1) ((ay) 1))))
  (lex.add.entry
   '("Aug" n  (((aa g) 1) ((ah s t) 0))))
  (lex.add.entry
   '("Sep" n (((s eh p) 0) ((t eh m b) 1) ((er) 0))))
  (lex.add.entry
   '("Sept" n (((s eh p) 0) ((t eh m b) 1) ((er) 0))))
  (lex.add.entry
   '("Oct" n (((aa k) 0) ((t ow b) 1) ((er) 0))))
  (lex.add.entry
   '("Nov" n (((n ow v) 0) ((eh m b) 1) ((er) 0))))
  (lex.add.entry
   '("Dec" n (((d ih s) 0) ((eh m b) 1) ((er) 0))))
  (lex.add.entry
   '("'s" pos (((ax z) 0))))
  (lex.add.entry 
   '("*" n (((ae s) 1) ((t er) 0) ((ih s k) 0)) ((pos "K6%"))))
  (lex.add.entry 
   '("%" n (((p er) 1) ((s eh n t) 1)) ((pos "K9%"))))
  (lex.add.entry 
   '("&" n (((ae m p) 1) ((er s) 0) ((ae n d) 0))))
  (lex.add.entry 
   '("$" n (((d aa l) 1) ((er) 0))))
  (lex.add.entry 
   '("#" n (((hh ae sh) 1))))
  (lex.add.entry 
   '("@" n (((ae t) 1))))
  (lex.add.entry 
   '("+" n (((p l ah s) 0)) ((pos "K7%" "OA%" "T-%"))))
  (lex.add.entry 
   '("^" n (((k eh r) 1) ((eh t) 0)) ((pos "K6$"))))
  (lex.add.entry 
   '("~" n (((t ih l d) 1) ((ah) 0)) ((pos "K6$"))))
  (lex.add.entry 
   '("=" n (((iy k) 1) ((w ah l) 0))))
  (lex.add.entry 
   '("/" n (((s l ae sh) 1))))
  (lex.add.entry 
   '("\\" n (((b ae k) 1) ((s l ae sh) 1))))
  (lex.add.entry 
   '("_" n (((ah n d) 1) ((er s) 0) ((k ao r) 1))))
  (lex.add.entry 
   '("|" n (((v er t) 1) ((ih k) 0) ((ah l) 0) ((b aa r) 1))))
  (lex.add.entry 
   '(">" n ((( g r ey ) 1) ((t er) 0) ((dh ae n) 1))))
  (lex.add.entry 
   '("<" n ((( l eh s) 1) ((dh ae n) 1))))
  (lex.add.entry 
   '("[" n ((( l eh f t) 1) ((b r ae k) 1) ((ih t) 0))))
  (lex.add.entry 
   '("]" n ((( r ay t) 1) ((b r ae k) 1) ((ih t) 0))))
  (lex.add.entry 
   '(" " n (((s p ey s) 1))))
  (lex.add.entry 
   '("\t" n (((t ae b) 1))))
  (lex.add.entry 
   '("\n" n (((n uw) 1) ((l ay n) 1))))
  (lex.add.entry '("." punc nil))
  (lex.add.entry '("'" punc nil))
  (lex.add.entry '(":" punc nil))
  (lex.add.entry '(";" punc nil))
  (lex.add.entry '("," punc nil))
  (lex.add.entry '("-" punc nil))
  (lex.add.entry '("\"" punc nil))
  (lex.add.entry '("`" punc nil))
  (lex.add.entry '("?" punc nil))
  (lex.add.entry '("!" punc nil))
  (lex.add.entry
   '(before ()
	    (((b iy) 0) ((f ao r) 1))))
  )

(lex.create "cmu")
(lex.set.compile.file (path-append cmulexdir "cmudict-0.4.out"))
(lex.set.phoneset "radio")
(lex.set.lts.method 'cmu_lts_function)
(lex.set.pos.map english_pos_map_wp39_to_wp20)
(cmulex_addenda)

(provide 'cmulex)

